package com.tencent.imsdk.conversation;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.message.DraftMessage;

import java.util.List;

public class ConversationManager {
    private static final String TAG = "ConversationManager";

    private static class ConversationManagerHolder {
        private static final ConversationManager conversationManager = new ConversationManager();
    }

    public static ConversationManager getInstance() {
        return ConversationManager.ConversationManagerHolder.conversationManager;
    }

    private ConversationListener mInternalConversationListener;
    private ConversationListener mConversationListener;

    public void init() {
        initInternalConversationListener();
    }

    private void initInternalConversationListener() {
        if (mInternalConversationListener == null) {
            mInternalConversationListener = new ConversationListener() {
                @Override
                public void onSyncServerStart() {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onSyncServerStart();
                            }
                        }
                    });
                }

                @Override
                public void onSyncServerFinish() {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onSyncServerFinish();
                            }
                        }
                    });
                }

                @Override
                public void onSyncServerFailed() {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onSyncServerFailed();
                            }
                        }
                    });
                }

                @Override
                public void onNewConversation(final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onNewConversation(conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationChanged(final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationChanged(conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onTotalUnreadMessageCountChanged(final long totalUnreadCount) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onTotalUnreadMessageCountChanged(totalUnreadCount);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupCreated(final String groupName, final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationGroupCreated(groupName, conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupDeleted(final String groupName) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationGroupDeleted(groupName);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupNameChanged(final String oldName, final String newName) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationGroupNameChanged(oldName, newName);
                            }
                        }
                    });
                }

                @Override
                public void onConversationsAddedToGroup(final String groupName, final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationsAddedToGroup(groupName, conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationsDeletedFromGroup(final String groupName, final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationsDeletedFromGroup(groupName, conversationList);
                            }
                        }
                    });
                }
            };
        }
        nativeSetConversationListener(mInternalConversationListener);
    }

    public void setConversationListener(ConversationListener conversationListener) {
        mConversationListener = conversationListener;
    }

    public void getConversationList(long nextSeq, int count, IMCallback<ConversationResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationList(nextSeq, count, callback);
    }

    public void getConversationListByFilter(ConversationListFilter filter, IMCallback<ConversationResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationListByFilter(filter, callback);
    }

    public void getConversationList(List<ConversationKey> conversationKeyList, IMCallback<List<Conversation>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversations(conversationKeyList, callback);
    }

    public void getConversationInfo(ConversationKey conversationKey, IMCallback<Conversation> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationInfo(conversationKey, callback);
    }

    public void deleteConversation(ConversationKey conversationKey, boolean clearCloudMessage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDeleteConversation(conversationKey, clearCloudMessage, callback);
    }

    public void setConversationDraft(ConversationKey conversationKey, DraftMessage draftMessage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetConversationDraft(conversationKey, draftMessage, callback);
    }

    public void pinConversation(ConversationKey conversationKey, boolean isPinned, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativePinConversation(conversationKey, isPinned, callback);
    }

    public void getTotalUnreadMessageCount(IMCallback<Long> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetTotalUnreadMessageCount(callback);
    }

    public void setCosSaveRegionForConversation(ConversationKey conversationKey, String cosSaveRegion, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetCosSaveRegionForConversation(conversationKey, cosSaveRegion, callback);
    }

    public void clearUnreadMessage(boolean clearC2CUnreadMessage, boolean clearGroupUnreadMessage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeClearUnreadMessage(clearC2CUnreadMessage, clearGroupUnreadMessage, callback);
    }

    public void markConversation(List<ConversationKey> conversationKeyList, long mark_type, boolean enable_mark, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeMarkConversation(conversationKeyList, mark_type, enable_mark, callback);
    }

    public void setConversationCustomData(List<ConversationKey> conversationKeyList, String customData, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetConversationCustomData(conversationKeyList, customData, callback);
    }

    public void createConversationGroup(String group_name, List<ConversationKey> conversationKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeCreateConversationGroup(group_name, conversationKeyList, callback);
    }

    public void getConversationGroupList(IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationGroupList(callback);
    }

    public void deleteConversationGroup(String group_name, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDeleteConversationGroup(group_name, callback);
    }

    public void renameConversationGroup(String old_name, String new_name, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeRenameConversationGroup(old_name, new_name, callback);
    }

    public void addConversationsToGroup(String group_name, List<ConversationKey> conversationKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeAddConversationsToGroup(group_name, conversationKeyList, callback);
    }

    public void deleteConversationsFromGroup(String group_name, List<ConversationKey> conversationKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDeleteConversationsFromGroup(group_name, conversationKeyList, callback);
    }

    protected native void nativeSetConversationListener(ConversationListener listener);

    protected native void nativeGetConversationList(long nextSeq, int count, IMCallback callback);

    protected native void nativeGetConversationListByFilter(ConversationListFilter filter, IMCallback callback);

    protected native void nativeGetConversations(List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeGetConversationInfo(ConversationKey conversationKey, IMCallback callback);

    protected native void nativeDeleteConversation(ConversationKey conversationKey, boolean clearCloudMessage, IMCallback callback);

    protected native void nativeSetConversationDraft(ConversationKey conversationKey, DraftMessage draftMessage, IMCallback callback);

    protected native void nativePinConversation(ConversationKey conversationKey, boolean isPinned, IMCallback callback);

    protected native void nativeGetTotalUnreadMessageCount(IMCallback callback);

    protected native void nativeSetCosSaveRegionForConversation(ConversationKey conversationKey, String cosSaveRegion, IMCallback callback);

    protected native void nativeClearUnreadMessage(boolean clearC2CUnreadMessage, boolean clearGroupUnreadMessage, IMCallback callback);

    protected native void nativeMarkConversation(List<ConversationKey> conversationKeyList, long mark_type, boolean enable_mark, IMCallback callback);

    protected native void nativeSetConversationCustomData(List<ConversationKey> conversationKeyList, String customData, IMCallback callback);

    protected native void nativeCreateConversationGroup(String group_name, List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeGetConversationGroupList(IMCallback callback);

    protected native void nativeDeleteConversationGroup(String group_name, IMCallback callback);

    protected native void nativeRenameConversationGroup(String old_name, String new_name, IMCallback callback);

    protected native void nativeAddConversationsToGroup(String group_name, List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeDeleteConversationsFromGroup(String group_name, List<ConversationKey> conversationKeyList, IMCallback callback);
}
