package com.tencent.imsdk.signaling;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.message.Message;
import com.tencent.imsdk.message.MessageOfflinePushInfo;

import java.util.List;

public class SignalingManager {
    private static String TAG = "SignalingManager";
    private SignalingListener mInternalSignalingListener;
    private SignalingListener mSignalingListener;

    private static class SignalingManagerHolder {
        private static final SignalingManager signalingManager = new SignalingManager();
    }

    public static SignalingManager getInstance() {
        return SignalingManagerHolder.signalingManager;
    }

    public void init() {
        initSignalingListener();
    }

    private void initSignalingListener() {
        if (mInternalSignalingListener == null) {
            mInternalSignalingListener = new SignalingListener() {
                @Override
                public void onReceiveNewInvitation(final String inviteID, final String inviter, final String groupID,
                                                   final List<String> inviteeList, final String data) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mSignalingListener != null) {
                                mSignalingListener.onReceiveNewInvitation(inviteID, inviter, groupID, inviteeList, data);
                            }
                        }
                    });
                }

                @Override
                public void onInviteeAccepted(final String inviteID, final String invitee, final String data) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mSignalingListener != null) {
                                mSignalingListener.onInviteeAccepted(inviteID, invitee, data);
                            }
                        }
                    });
                }

                @Override
                public void onInviteeRejected(final String inviteID, final String invitee, final String data) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mSignalingListener != null) {
                                mSignalingListener.onInviteeRejected(inviteID, invitee, data);
                            }
                        }
                    });
                }

                @Override
                public void onInvitationCancelled(final String inviteID, final String inviter, final String data) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mSignalingListener != null) {
                                mSignalingListener.onInvitationCancelled(inviteID, inviter, data);
                            }
                        }
                    });
                }

                @Override
                public void onInvitationTimeout(final String inviteID, final List<String> inviteeList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mSignalingListener != null) {
                                mSignalingListener.onInvitationTimeout(inviteID, inviteeList);
                            }
                        }
                    });
                }

                @Override
                public void onInvitationModified(final String inviteID, final String data) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mSignalingListener != null) {
                                mSignalingListener.onInvitationModified(inviteID, data);
                            }
                        }
                    });
                }
            };
        }
        nativeSetSignalingObserver(mInternalSignalingListener);
    }

    public void setSignalingListener(SignalingListener listener) {
        mSignalingListener = listener;
    }

    public String invite(String invitee,
                       String data,
                       boolean onlineUserOnly,
                       MessageOfflinePushInfo offlinePushInfo,
                       int timeout,
                       IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return null;
        }
        return nativeInvite(invitee, data, onlineUserOnly, offlinePushInfo, timeout, callback);
    }

    public String inviteInGroup(String groupID,
                              List<String> inviteeList,
                              String data,
                              boolean onlineUserOnly,
                              int timeout,
                              IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return null;
        }
        return nativeInviteInGroup(groupID, inviteeList, data, onlineUserOnly, timeout, callback);
    }

    public void cancel(String inviteID, String data, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeCancel(inviteID, data, callback);
    }

    public void accept(String inviteID, String data, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeAccept(inviteID, data, callback);
    }

    public void reject(String inviteID, String data, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeReject(inviteID, data, callback);
    }

    public SignalingInfo getSignalingInfo(Message message) {
        if (!BaseManager.getInstance().isInited()) {
            IMLog.e(TAG, "getSignalingInfo error, sdk not init");
            return null;
        }
        return nativeGetSignalingInfo(message);
    }

    public void addInvitedSignaling(SignalingInfo signalingInfo, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            IMLog.e(TAG, "addInvitedSignaling error, sdk not init");
            return;
        }
        nativeAddInvitedSignaling(signalingInfo, callback);
    }

    public void modifyInvitation(String inviteID, String data, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            IMLog.e(TAG, "modifyInvitation error, sdk not init");
            return;
        }
        nativeModifyInvitation(inviteID, data, callback);
    }

    protected native void nativeSetSignalingObserver(SignalingListener listener);

    protected native String nativeInvite(String invitee,
                                       String data,
                                       boolean onlineUserOnly,
                                       MessageOfflinePushInfo messageOfflinePushInfo,
                                       int timeout,
                                       IMCallback callback);

    protected native String nativeInviteInGroup(String groupID,
                                                List<String> inviteeList,
                                                String data,
                                                boolean onlineUserOnly,
                                                int timeout,
                                                IMCallback callback);

    protected native void nativeCancel(String inviteID, String data, IMCallback callback);

    protected native void nativeAccept(String inviteID, String data, IMCallback callback);

    protected native void nativeReject(String inviteID, String data, IMCallback callback);

    protected native SignalingInfo nativeGetSignalingInfo(Message message);

    protected native void nativeAddInvitedSignaling(SignalingInfo signalingInfo, IMCallback callback);

    protected native void nativeModifyInvitation(String inviteID, String data, IMCallback callback);
}
