package com.tencent.imsdk.v2;

import android.text.TextUtils;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.conversation.Conversation;
import com.tencent.imsdk.conversation.ConversationKey;
import com.tencent.imsdk.conversation.ConversationListFilter;
import com.tencent.imsdk.conversation.ConversationListener;
import com.tencent.imsdk.conversation.ConversationManager;
import com.tencent.imsdk.conversation.ConversationOperationResult;
import com.tencent.imsdk.conversation.ConversationResult;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.message.DraftMessage;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class V2TIMConversationManagerImpl extends V2TIMConversationManager {

    private static class V2TIMConversationManagerImplHolder {
        private static final V2TIMConversationManagerImpl v2TIMConversationManagerImpl = new V2TIMConversationManagerImpl();
    }

    static V2TIMConversationManagerImpl getInstance() {
        return V2TIMConversationManagerImplHolder.v2TIMConversationManagerImpl;
    }

    private ConversationListener mConversationListener;
    private V2TIMConversationListener mV2TIMConversationListener;
    private final List<V2TIMConversationListener> mV2TIMConversationListenerList = new ArrayList<>();

    private V2TIMConversationManagerImpl() {
        initListener();
    }

    private void initListener() {
        if (mConversationListener == null) {
            mConversationListener = new ConversationListener() {
                @Override
                public void onSyncServerStart() {
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onSyncServerStart();
                    }
                }

                @Override
                public void onSyncServerFinish() {
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onSyncServerFinish();
                    }
                }

                @Override
                public void onSyncServerFailed() {
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onSyncServerFailed();
                    }
                }

                @Override
                public void onNewConversation(List<Conversation> conversationList) {
                    List<V2TIMConversation> v2TIMConversationList = new ArrayList<>();
                    for (Conversation conversation : conversationList) {
                        V2TIMConversation v2TIMConversation = new V2TIMConversation();
                        v2TIMConversation.setConversation(conversation);
                        v2TIMConversationList.add(v2TIMConversation);
                    }
                    List<V2TIMConversation> unmodifiableList = Collections.unmodifiableList(v2TIMConversationList);
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onNewConversation(unmodifiableList);
                    }
                }

                @Override
                public void onConversationChanged(List<Conversation> conversationList) {
                    List<V2TIMConversation> v2TIMConversationList = new ArrayList<>();
                    for (Conversation conversation : conversationList) {
                        V2TIMConversation v2TIMConversation = new V2TIMConversation();
                        v2TIMConversation.setConversation(conversation);
                        v2TIMConversationList.add(v2TIMConversation);
                    }
                    List<V2TIMConversation> unmodifiableList = Collections.unmodifiableList(v2TIMConversationList);
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onConversationChanged(unmodifiableList);
                    }
                }

                @Override
                public void onTotalUnreadMessageCountChanged(long totalUnreadCount) {
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onTotalUnreadMessageCountChanged(totalUnreadCount);
                    }
                }

                @Override
                public void onConversationGroupCreated(String groupName, List<Conversation> conversationList) {
                    List<V2TIMConversation> v2TIMConversationList = new ArrayList<>();
                    for (Conversation conversation : conversationList) {
                        V2TIMConversation v2TIMConversation = new V2TIMConversation();
                        v2TIMConversation.setConversation(conversation);
                        v2TIMConversationList.add(v2TIMConversation);
                    }
                    List<V2TIMConversation> unmodifiableList = Collections.unmodifiableList(v2TIMConversationList);
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onConversationGroupCreated(groupName, unmodifiableList);
                    }
                }

                @Override
                public void onConversationGroupDeleted(String groupName) {
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onConversationGroupDeleted(groupName);
                    }
                }

                @Override
                public void onConversationGroupNameChanged(String oldName, String newName) {
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onConversationGroupNameChanged(oldName, newName);
                    }
                }

                @Override
                public void onConversationsAddedToGroup(String groupName, List<Conversation> conversationList) {
                    List<V2TIMConversation> v2TIMConversationList = new ArrayList<>();
                    for (Conversation conversation : conversationList) {
                        V2TIMConversation v2TIMConversation = new V2TIMConversation();
                        v2TIMConversation.setConversation(conversation);
                        v2TIMConversationList.add(v2TIMConversation);
                    }
                    List<V2TIMConversation> unmodifiableList = Collections.unmodifiableList(v2TIMConversationList);
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onConversationsAddedToGroup(groupName, unmodifiableList);
                    }
                }

                @Override
                public void onConversationsDeletedFromGroup(String groupName,
                                                            List<Conversation> conversationList) {
                    List<V2TIMConversation> v2TIMConversationList = new ArrayList<>();
                    for (Conversation conversation : conversationList) {
                        V2TIMConversation v2TIMConversation = new V2TIMConversation();
                        v2TIMConversation.setConversation(conversation);
                        v2TIMConversationList.add(v2TIMConversation);
                    }
                    List<V2TIMConversation> unmodifiableList = Collections.unmodifiableList(v2TIMConversationList);
                    for (V2TIMConversationListener listener : mV2TIMConversationListenerList) {
                        listener.onConversationsDeletedFromGroup(groupName, unmodifiableList);
                    }
                }
            };
        }
        ConversationManager.getInstance().setConversationListener(mConversationListener);
    }

    @Override
    public void setConversationListener(final V2TIMConversationListener listener) {
        IMContext.getInstance().runOnMainThread(new Runnable() {
            @Override
            public void run() {
                if (mV2TIMConversationListener != null) {
                    mV2TIMConversationListenerList.remove(mV2TIMConversationListener);
                }
                if (listener != null) {
                    mV2TIMConversationListenerList.add(listener);
                }
                mV2TIMConversationListener = listener;
            }
        });
    }

    @Override
    public void addConversationListener(final V2TIMConversationListener listener) {
        if (listener == null) {
            return;
        }
        IMContext.getInstance().runOnMainThread(new Runnable() {
            @Override
            public void run() {
                if (mV2TIMConversationListenerList.contains(listener)) {
                    return;
                }
                mV2TIMConversationListenerList.add(listener);
            }
        });
    }

    @Override
    public void removeConversationListener(final V2TIMConversationListener listener) {
        if (listener == null) {
            return;
        }
        IMContext.getInstance().runOnMainThread(new Runnable() {
            @Override
            public void run() {
                mV2TIMConversationListenerList.remove(listener);
            }
        });
    }

    @Override
    public void getConversationList(long nextSeq, int count, final V2TIMValueCallback<V2TIMConversationResult> callback) {
        if (count <= 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "count is invalid");
            }
            return;
        }

        V2TIMValueCallback<ConversationResult> v2callback = new V2TIMValueCallback<ConversationResult>() {
            @Override
            public void onSuccess(ConversationResult conversationResult) {
                if (callback != null) {
                    V2TIMConversationResult v2TIMConversationResult = new V2TIMConversationResult();
                    v2TIMConversationResult.setConversationResult(conversationResult);
                    callback.onSuccess(v2TIMConversationResult);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().getConversationList(nextSeq, count, new IMCallback<ConversationResult>(v2callback) {
            @Override
            public void success(ConversationResult data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });

        BaseManager.getInstance().checkTUIComponent(BaseManager.TUI_COMPONENT_CONVERSATION);
    }

    @Override
    public void getConversationListByFilter(V2TIMConversationListFilter filter,
                                            final V2TIMValueCallback<V2TIMConversationResult> callback) {
        if (filter == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "filter is null");
            }
            return;
        }

        if (filter.getCount() <= 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "count is invalid");
            }
            return;
        }

        V2TIMValueCallback<ConversationResult> v2callback = new V2TIMValueCallback<ConversationResult>() {
            @Override
            public void onSuccess(ConversationResult conversationResult) {
                if (callback != null) {
                    V2TIMConversationResult v2TIMConversationResult = new V2TIMConversationResult();
                    v2TIMConversationResult.setConversationResult(conversationResult);
                    callback.onSuccess(v2TIMConversationResult);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().getConversationListByFilter(filter.getConversationFilter(), new IMCallback<ConversationResult>(v2callback) {
            @Override
            public void success(ConversationResult data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getConversationList(List<String> conversationIDList, final V2TIMValueCallback<List<V2TIMConversation>> callback) {
        if (conversationIDList == null || conversationIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationIDList cannot be empty");
            }
            return;
        }
        List<ConversationKey> conversationKeyList = new ArrayList<>();
        for (String conversationID : conversationIDList) {
            ConversationKey conversationKey = getConversationKey(conversationID);
            conversationKeyList.add(conversationKey);
        }
        V2TIMValueCallback<List<Conversation>> v2callback = new V2TIMValueCallback<List<Conversation>>() {
            @Override
            public void onSuccess(List<Conversation> conversations) {
                List<V2TIMConversation> v2TIMConversationList = new ArrayList<>();
                for (Conversation conversation : conversations) {
                    V2TIMConversation v2TIMConversation = new V2TIMConversation();
                    v2TIMConversation.setConversation(conversation);
                    v2TIMConversationList.add(v2TIMConversation);
                }
                if (callback != null) {
                    callback.onSuccess(v2TIMConversationList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().getConversationList(conversationKeyList, new IMCallback<List<Conversation>>(v2callback) {
            @Override
            public void success(List<Conversation> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getConversation(String conversationID, final V2TIMValueCallback<V2TIMConversation> callback) {
        if (TextUtils.isEmpty(conversationID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationID is empty");
            }
            return;
        }
        ConversationKey conversationKey = getConversationKey(conversationID);

        V2TIMValueCallback<Conversation> v2TIMValueCallback = new V2TIMValueCallback<Conversation>() {
            @Override
            public void onSuccess(Conversation conversation) {
                if (callback != null) {
                    V2TIMConversation v2TIMConversation = new V2TIMConversation();
                    v2TIMConversation.setConversation(conversation);

                    callback.onSuccess(v2TIMConversation);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().getConversationInfo(conversationKey, new IMCallback<Conversation>(v2TIMValueCallback) {
            @Override
            public void success(Conversation data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteConversation(String conversationID, V2TIMCallback callback) {
        if (TextUtils.isEmpty(conversationID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationID is empty");
            }
            return;
        }
        ConversationKey conversationKey = getConversationKey(conversationID);

        ConversationManager.getInstance().deleteConversation(conversationKey, true, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void setConversationDraft(String conversationID, String draftText, V2TIMCallback callback) {
        if (TextUtils.isEmpty(conversationID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationID is empty");
            }
            return;
        }
        ConversationKey conversationKey = getConversationKey(conversationID);
        DraftMessage draftMessage = new DraftMessage();
        if (draftText != null) {
            draftMessage.setUserDefinedData(draftText.getBytes());
        }

        ConversationManager.getInstance().setConversationDraft(conversationKey, draftMessage, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void pinConversation(String conversationID, boolean isPinned, V2TIMCallback callback) {
        if (TextUtils.isEmpty(conversationID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationID is empty");
            }
            return;
        }
        ConversationKey conversationKey = getConversationKey(conversationID);
        ConversationManager.getInstance().pinConversation(conversationKey, isPinned, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getTotalUnreadMessageCount(V2TIMValueCallback<Long> callback) {
        ConversationManager.getInstance().getTotalUnreadMessageCount(new IMCallback<Long>(callback) {
            @Override
            public void success(Long data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void markConversation(List<String> conversationIDList, long markType, boolean enableMark,
                                 final V2TIMValueCallback<List<V2TIMConversationOperationResult>> callback) {
        if (conversationIDList == null || conversationIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationIDList is empty");
            }
            return;
        }

        if (markType == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "markType is invalid");
            }
            return;
        }

        List<ConversationKey> conversationKeyList = new ArrayList<>();
        for (String conversationID : conversationIDList) {
            ConversationKey conversationKey = getConversationKey(conversationID);
            conversationKeyList.add(conversationKey);
        }

        V2TIMValueCallback<List<ConversationOperationResult>> v2Callback = new V2TIMValueCallback<List<ConversationOperationResult>>() {
            @Override
            public void onSuccess(List<ConversationOperationResult> resultList) {
                List<V2TIMConversationOperationResult> v2ResultList = new ArrayList<>();
                for (ConversationOperationResult result : resultList) {
                    V2TIMConversationOperationResult v2Result = new V2TIMConversationOperationResult();
                    v2Result.setConversationOperationResult(result);
                    v2ResultList.add(v2Result);
                }

                if (callback != null) {
                    callback.onSuccess(v2ResultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().markConversation(conversationKeyList, markType, enableMark, new IMCallback(v2Callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void setConversationCustomData(List<String> conversationIDList, String customData,
                                          final V2TIMValueCallback<List<V2TIMConversationOperationResult>> callback) {
        if (conversationIDList == null || conversationIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationIDList is empty");
            }
            return;
        }

        List<ConversationKey> conversationKeyList = new ArrayList<>();
        for (String conversationID : conversationIDList) {
            ConversationKey conversationKey = getConversationKey(conversationID);
            conversationKeyList.add(conversationKey);
        }

        V2TIMValueCallback<List<ConversationOperationResult>> v2Callback = new V2TIMValueCallback<List<ConversationOperationResult>>() {
            @Override
            public void onSuccess(List<ConversationOperationResult> resultList) {
                List<V2TIMConversationOperationResult> v2ResultList = new ArrayList<>();
                for (ConversationOperationResult result : resultList) {
                    V2TIMConversationOperationResult v2Result = new V2TIMConversationOperationResult();
                    v2Result.setConversationOperationResult(result);
                    v2ResultList.add(v2Result);
                }

                if (callback != null) {
                    callback.onSuccess(v2ResultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().setConversationCustomData(conversationKeyList, customData, new IMCallback(v2Callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void createConversationGroup(String groupName, List<String> conversationIDList,
                                        final V2TIMValueCallback<List<V2TIMConversationOperationResult>> callback) {
        if (TextUtils.isEmpty(groupName)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "group name is empty");
            }
            return;
        }

        List<ConversationKey> conversationKeyList = new ArrayList<>();
        for (String conversationID : conversationIDList) {
            ConversationKey conversationKey = getConversationKey(conversationID);
            conversationKeyList.add(conversationKey);
        }

        V2TIMValueCallback<List<ConversationOperationResult>> v2Callback = new V2TIMValueCallback<List<ConversationOperationResult>>() {
            @Override
            public void onSuccess(List<ConversationOperationResult> resultList) {
                List<V2TIMConversationOperationResult> v2ResultList = new ArrayList<>();
                for (ConversationOperationResult result : resultList) {
                    V2TIMConversationOperationResult v2Result = new V2TIMConversationOperationResult();
                    v2Result.setConversationOperationResult(result);
                    v2ResultList.add(v2Result);
                }

                if (callback != null) {
                    callback.onSuccess(v2ResultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().createConversationGroup(groupName, conversationKeyList, new IMCallback(v2Callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getConversationGroupList(V2TIMValueCallback<List<String>> callback) {
        ConversationManager.getInstance().getConversationGroupList(new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteConversationGroup(String groupName, V2TIMCallback callback) {
        if (TextUtils.isEmpty(groupName)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "group name is empty");
            }
            return;
        }

        ConversationManager.getInstance().deleteConversationGroup(groupName, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void renameConversationGroup(String oldName, String newName, V2TIMCallback callback) {
        if (TextUtils.isEmpty(oldName)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "oldName is empty");
            }
            return;
        }

        if (TextUtils.isEmpty(newName)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "newName is empty");
            }
            return;
        }

        ConversationManager.getInstance().renameConversationGroup(oldName, newName, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void addConversationsToGroup(String groupName, List<String> conversationIDList,
                                        final V2TIMValueCallback<List<V2TIMConversationOperationResult>> callback) {
        if (TextUtils.isEmpty(groupName)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupName is empty");
            }
            return;
        }

        if (conversationIDList == null || conversationIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationIDList is empty");
            }
            return;
        }

        List<ConversationKey> conversationKeyList = new ArrayList<>();
        for (String conversationID : conversationIDList) {
            ConversationKey conversationKey = getConversationKey(conversationID);
            conversationKeyList.add(conversationKey);
        }

        V2TIMValueCallback<List<ConversationOperationResult>> v2Callback = new V2TIMValueCallback<List<ConversationOperationResult>>() {
            @Override
            public void onSuccess(List<ConversationOperationResult> resultList) {
                List<V2TIMConversationOperationResult> v2ResultList = new ArrayList<>();
                for (ConversationOperationResult result : resultList) {
                    V2TIMConversationOperationResult v2Result = new V2TIMConversationOperationResult();
                    v2Result.setConversationOperationResult(result);
                    v2ResultList.add(v2Result);
                }

                if (callback != null) {
                    callback.onSuccess(v2ResultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().addConversationsToGroup(groupName, conversationKeyList, new IMCallback(v2Callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteConversationsFromGroup(String groupName, List<String> conversationIDList,
                                             final V2TIMValueCallback<List<V2TIMConversationOperationResult>> callback) {
        if (TextUtils.isEmpty(groupName)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupName is empty");
            }
            return;
        }

        if (conversationIDList == null || conversationIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "conversationIDList is empty");
            }
            return;
        }

        List<ConversationKey> conversationKeyList = new ArrayList<>();
        for (String conversationID : conversationIDList) {
            ConversationKey conversationKey = getConversationKey(conversationID);
            conversationKeyList.add(conversationKey);
        }

        V2TIMValueCallback<List<ConversationOperationResult>> v2Callback = new V2TIMValueCallback<List<ConversationOperationResult>>() {
            @Override
            public void onSuccess(List<ConversationOperationResult> resultList) {
                List<V2TIMConversationOperationResult> v2ResultList = new ArrayList<>();
                for (ConversationOperationResult result : resultList) {
                    V2TIMConversationOperationResult v2Result = new V2TIMConversationOperationResult();
                    v2Result.setConversationOperationResult(result);
                    v2ResultList.add(v2Result);
                }

                if (callback != null) {
                    callback.onSuccess(v2ResultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().deleteConversationsFromGroup(groupName, conversationKeyList, new IMCallback(v2Callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    protected ConversationKey getConversationKey(String conversationID) {
        ConversationKey conversationKey = new ConversationKey();
        if (TextUtils.isEmpty(conversationID)) {
            return conversationKey;
        }

        String v1ConvID = "";
        int index = conversationID.indexOf(V2TIMConversation.CONVERSATION_C2C_PREFIX);
        if (index == 0) {
            v1ConvID = conversationID.substring(V2TIMConversation.CONVERSATION_C2C_PREFIX.length());
            conversationKey.setConversationType(ConversationKey.TYPE_C2C);
            conversationKey.setConversationID(v1ConvID);
        } else {
            index = conversationID.indexOf(V2TIMConversation.CONVERSATION_GROUP_PREFIX);
            if (index == 0) {
                v1ConvID = conversationID.substring(V2TIMConversation.CONVERSATION_GROUP_PREFIX.length());
                conversationKey.setConversationType(ConversationKey.TYPE_GROUP);
                conversationKey.setConversationID(v1ConvID);
            }
        }
        return conversationKey;
    }
}
