package com.tencent.imsdk.v2;

import android.text.TextUtils;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.conversation.ConversationManager;
import com.tencent.imsdk.group.GroupManager;
import com.tencent.imsdk.group.GroupMemberInfo;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.message.CustomElement;
import com.tencent.imsdk.message.FaceElement;
import com.tencent.imsdk.message.FileElement;
import com.tencent.imsdk.message.GroupMessageReadMembers;
import com.tencent.imsdk.message.GroupMessageReceipt;
import com.tencent.imsdk.message.ImageElement;
import com.tencent.imsdk.message.LocationElement;
import com.tencent.imsdk.message.MergerElement;
import com.tencent.imsdk.message.Message;
import com.tencent.imsdk.message.MessageCenter;
import com.tencent.imsdk.message.MessageExtension;
import com.tencent.imsdk.message.MessageExtensionResult;
import com.tencent.imsdk.message.MessageKey;
import com.tencent.imsdk.message.MessageListGetOption;
import com.tencent.imsdk.message.MessageListener;
import com.tencent.imsdk.message.C2CMessageReceipt;
import com.tencent.imsdk.message.MessageSearchResult;
import com.tencent.imsdk.message.MessageUploadProgressCallback;
import com.tencent.imsdk.message.SoundElement;
import com.tencent.imsdk.message.TextElement;
import com.tencent.imsdk.message.VideoElement;
import com.tencent.imsdk.relationship.ReceiveMessageOptInfo;
import com.tencent.imsdk.relationship.RelationshipManager;
import com.tencent.imsdk.relationship.UserInfo;


import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * 消息接口实现类
 */
public class V2TIMMessageManagerImpl extends V2TIMMessageManager{
    private final String TAG = "V2TIMMessageManagerImpl";

    private final int MAX_FORWARD_COUNT = 300;
    private final int MAX_ABSTRACT_COUNT  = 5;
    private final int MAX_ABSTRACT_LENGTH = 100;

    private Object mLockObject = new Object();
    private MessageListener mMessageListener;
    private List<V2TIMAdvancedMsgListener> mV2TIMMsgListenerList;

    private static class V2TIMMessageManagerImplHolder {
        private static final V2TIMMessageManagerImpl v2TIMMessageManagerImpl = new V2TIMMessageManagerImpl();
    }

    static V2TIMMessageManagerImpl getInstance() {
        return V2TIMMessageManagerImplHolder.v2TIMMessageManagerImpl;
    }

    private V2TIMMessageManagerImpl() {
        mV2TIMMsgListenerList = new CopyOnWriteArrayList<>();
    }

    void initListener() {
        mMessageListener = new MessageListener() {
            @Override
            public void onReceiveNewMessage(List<Message> messageList) {
                if (messageList == null || messageList.isEmpty()) {
                    return;
                }

                for (Message message : messageList) {
                    V2TIMMessage v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);
                    onRecvNewMessage(v2TIMMessage);
                }

            }

            @Override
            public void onReceiveC2CMessageReceipt(List<C2CMessageReceipt> receiptList) {
                List<V2TIMMessageReceipt> v2TIMMessageReceiptList = new ArrayList<>();
                boolean isMessageIDExist = false;
                for (int i = 0; i < receiptList.size(); i++) {
                    C2CMessageReceipt messageReceipt = receiptList.get(i);
                    V2TIMMessageReceipt v2TIMMessageReceipt = new V2TIMMessageReceipt();
                    v2TIMMessageReceipt.setC2CMessageReceipt(messageReceipt);
                    v2TIMMessageReceiptList.add(v2TIMMessageReceipt);

                    if (i == 0) {
                        if (TextUtils.isEmpty(messageReceipt.getMessageID())) {
                            isMessageIDExist = false;
                        } else {
                            isMessageIDExist = true;
                        }
                    }
                }
                synchronized (mLockObject) {
                    for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                        if (isMessageIDExist) {
                            listener.onRecvMessageReadReceipts(v2TIMMessageReceiptList);
                        } else {
                            listener.onRecvC2CReadReceipt(v2TIMMessageReceiptList);
                        }
                    }
                }
            }

            @Override
            public void onReceiveGroupMessageReceipt(List<GroupMessageReceipt> receiptList) {
                List<V2TIMMessageReceipt> v2TIMMessageReadReceiptList = new ArrayList<>();
                for (GroupMessageReceipt messageReceipt : receiptList) {
                    V2TIMMessageReceipt v2TIMMessageReadReceipt = new V2TIMMessageReceipt();
                    v2TIMMessageReadReceipt.setGroupMessageReceipt(messageReceipt);
                    v2TIMMessageReadReceiptList.add(v2TIMMessageReadReceipt);
                }

                synchronized (mLockObject) {
                    for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                        listener.onRecvMessageReadReceipts(v2TIMMessageReadReceiptList);
                    }
                }
            }

            @Override
            public void onReceiveMessageRevoked(List<MessageKey> keyList) {
                if (keyList == null || keyList.isEmpty()) {
                    return;
                }
                for (MessageKey messageKey : keyList) {
                    synchronized (mLockObject) {
                        for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                            listener.onRecvMessageRevoked(messageKey.getMessageID());
                        }
                    }
                }
            }

            @Override
            public void onReceiveMessageModified(List<Message> messageList) {
                for (Message message : messageList) {
                    V2TIMMessage v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);
                    synchronized (mLockObject) {
                        for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                            listener.onRecvMessageModified(v2TIMMessage);
                        }
                    }
                }
            }

            @Override
            public void onReceiveMessageExtensionsChanged(MessageKey messageKey, List<MessageExtension> extensions) {
                String msgID = "";
                if (messageKey != null) {
                    msgID = messageKey.getMessageID();
                }
                List<V2TIMMessageExtension> v2TIMMessageExtensionList = new ArrayList<>();
                for (MessageExtension messageExtension : extensions) {
                    V2TIMMessageExtension v2TIMMessageExtension = new V2TIMMessageExtension();
                    v2TIMMessageExtension.setMessageExtension(messageExtension);
                    v2TIMMessageExtensionList.add(v2TIMMessageExtension);
                }
                synchronized (mLockObject) {
                    for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                        listener.onRecvMessageExtensionsChanged(msgID, v2TIMMessageExtensionList);
                    }
                }
            }

            @Override
            public void onReceiveMessageExtensionsDeleted(MessageKey messageKey, List<MessageExtension> extensions) {
                String msgID = "";
                if (messageKey != null) {
                    msgID = messageKey.getMessageID();
                }
                List<String> v2TIMMessageExtensionList = new ArrayList<>();
                for (MessageExtension messageExtension : extensions) {
                    v2TIMMessageExtensionList.add(messageExtension.getExtensionKey());
                }
                synchronized (mLockObject) {
                    for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                        listener.onRecvMessageExtensionsDeleted(msgID, v2TIMMessageExtensionList);
                    }
                }
            }
        };
        MessageCenter.getInstance().addMessageListener(mMessageListener);
    }

    protected void onRecvNewMessage(V2TIMMessage v2TIMMessage) {
        synchronized (mLockObject) {
            for (V2TIMAdvancedMsgListener listener : mV2TIMMsgListenerList) {
                listener.onRecvNewMessage(v2TIMMessage);
            }
        }
    }

    @Override
    public void addAdvancedMsgListener(V2TIMAdvancedMsgListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (mLockObject) {
            if (mV2TIMMsgListenerList.contains(listener)) {
                return;
            }
            mV2TIMMsgListenerList.add(listener);
        }
    }

    @Override
    public void removeAdvancedMsgListener(V2TIMAdvancedMsgListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (mLockObject) {
            mV2TIMMsgListenerList.remove(listener);
        }
    }

    @Override
    public V2TIMMessage createTextMessage(String text) {
        if (TextUtils.isEmpty(text)) {
            IMLog.e(TAG, "text cannot be empty");
            return null;
        }
        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        TextElement textElement = new TextElement();
        textElement.setTextContent(text);
        message.addElement(textElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createTextAtMessage(String text, List<String> atUserList) {
        if (TextUtils.isEmpty(text)) {
            IMLog.e(TAG, "text cannot be empty");
            return null;
        }

        if (atUserList == null || atUserList.isEmpty()) {
            IMLog.e(TAG, "atUserList cannot be empty");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        TextElement textElement = new TextElement();
        textElement.setTextContent(text);
        message.addElement(textElement);
        v2TIMMessage.setGroupAtUserList(atUserList);
        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createCustomMessage(byte[] data) {
        if (data == null || data.length == 0) {
            IMLog.e(TAG, "data cannot be empty");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        CustomElement customElement = new CustomElement();
        customElement.setData(data);
        message.addElement(customElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createCustomMessage(byte[] data, String description, byte[] extension) {
        if (data == null || data.length == 0) {
            IMLog.e(TAG, "data cannot be empty");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        CustomElement customElement = new CustomElement();
        customElement.setData(data);
        customElement.setDescription(description);
        customElement.setExtension(extension);
        message.addElement(customElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createImageMessage(String imagePath) {
        if (TextUtils.isEmpty(imagePath)) {
            IMLog.e(TAG, "imagePath cannot be empty");
            return null;
        }

        File file = new File(imagePath);
        if (!file.exists()) {
            IMLog.e(TAG, "file not exist");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        ImageElement imageElement = new ImageElement();
        imageElement.setOriginImageFilePath(imagePath);
        message.addElement(imageElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createSoundMessage(String soundPath, int duration) {
        if (TextUtils.isEmpty(soundPath)) {
            IMLog.e(TAG, "soundPath cannot be empty");
            return null;
        }

        File file = new File(soundPath);
        if (!file.exists()) {
            IMLog.e(TAG, "file not exist");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        SoundElement soundElement = new SoundElement();
        if (duration < 0) {
            duration = 0;
        }
        soundElement.setSoundDuration(duration);
        soundElement.setSoundFilePath(soundPath);
        message.addElement(soundElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createVideoMessage(String videoFilePath, String type, int duration, String snapshotPath) {
        if (TextUtils.isEmpty(videoFilePath)) {
            IMLog.e(TAG, "videoFilePath cannot be empty");
            return null;
        }

        File videoFile = new File(videoFilePath);
        if (!videoFile.exists()) {
            IMLog.e(TAG, "video file not exist");
            return null;
        }

        if (TextUtils.isEmpty(snapshotPath)) {
            IMLog.e(TAG, "snapshotPath cannot be empty");
            return null;
        }

        File snapshotFile = new File(snapshotPath);
        if (!snapshotFile.exists()) {
            IMLog.e(TAG, "snapshot file not exist");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        VideoElement videoElement = new VideoElement();
        videoElement.setVideoFilePath(videoFilePath);
        videoElement.setSnapshotFilePath(snapshotPath);
        if (duration < 0) {
            duration = 0;
        }
        videoElement.setVideoDuration(duration);
        videoElement.setVideoType(type);
        message.addElement(videoElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createFileMessage(String filePath, String fileName) {
        if (TextUtils.isEmpty(filePath)) {
            IMLog.e(TAG, "filePath cannot be empty");
            return null;
        }

        File file = new File(filePath);
        if (!file.exists()) {
            return null;
        }

        if (TextUtils.isEmpty(fileName)) {
            IMLog.e(TAG, "fileName cannot be empty");
            return null;
        }

        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        FileElement fileElement = new FileElement();
        fileElement.setFilePath(filePath);
        fileElement.setFileName(fileName);
        message.addElement(fileElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createLocationMessage(String desc, double longitude, double latitude) {
        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        LocationElement locationElement = new LocationElement();
        locationElement.setLongitude(longitude);
        locationElement.setLatitude(latitude);
        locationElement.setDescription(desc);
        message.addElement(locationElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createFaceMessage(int index, byte[] data) {
        V2TIMMessage v2TIMMessage = new V2TIMMessage();

        Message message = v2TIMMessage.getMessage();
        FaceElement faceElement = new FaceElement();
        faceElement.setFaceData(data);
        faceElement.setFaceIndex(index);
        message.addElement(faceElement);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createMergerMessage(List<V2TIMMessage> messageList,
                                            String title,
                                            List<String> abstractList,
                                            String compatibleText) {
        if (messageList == null || messageList.size() == 0 || messageList.size() > MAX_FORWARD_COUNT) {
            IMLog.e(TAG, "messageList invalid, the number of messageList must be between 1 and " + MAX_FORWARD_COUNT);
            return null;
        }


        List<String> abstractListAdjust = new ArrayList<>();
        if (abstractList != null) {
            int abstractCount = abstractList.size();
            abstractCount = abstractCount >= MAX_ABSTRACT_COUNT ? MAX_ABSTRACT_COUNT : abstractCount;
            for (int i = 0; i < abstractCount; ++i) {
                String abstractItem = abstractList.get(i);
                if (abstractItem != null) {
                    if (abstractItem.length() > MAX_ABSTRACT_LENGTH) {
                        abstractItem = abstractItem.substring(0, MAX_ABSTRACT_LENGTH);
                    }
                    abstractListAdjust.add(abstractItem);
                }
            }
        }

        for (V2TIMMessage v2TIMMessage : messageList) {
            if (V2TIMMessage.V2TIM_MSG_STATUS_SEND_SUCC != v2TIMMessage.getStatus()) {
                IMLog.e(TAG, "message status must be V2TIM_MSG_STATUS_SEND_SUCC");
                return null;
            }
            if (V2TIMMessage.V2TIM_ELEM_TYPE_GROUP_TIPS == v2TIMMessage.getElemType()) {
                IMLog.e(TAG, "group tips message is not support");
                return null;
            }
        }

        if (null == compatibleText) {
            IMLog.e(TAG, "compatibleText invalid, compatibleText cannot be null");
            return null;
        }

        MergerElement mergerElement = new MergerElement();
        mergerElement.setTitle(title);
        mergerElement.setAbstractList(abstractListAdjust);
        mergerElement.setCompatibleText(compatibleText);
        List<Message> nativeMessageList = new ArrayList<>();
        for (V2TIMMessage v2TIMMessage : messageList) {
            nativeMessageList.add(v2TIMMessage.getMessage());
        }
        mergerElement.setMessageList(nativeMessageList);

        V2TIMMessage v2TIMMessage = new V2TIMMessage();
        Message message = v2TIMMessage.getMessage();
        message.addElement(mergerElement);
        message.setForward(true);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createForwardMessage(V2TIMMessage message) {
        if (message == null) {
            IMLog.e(TAG, "createForwardMessage, message cannot be null");
            return null;
        }
        if (V2TIMMessage.V2TIM_MSG_STATUS_SEND_SUCC != message.getStatus()) {
            IMLog.e(TAG, "message status must be V2TIM_MSG_STATUS_SEND_SUCC");
            return null;
        }
        if (V2TIMMessage.V2TIM_ELEM_TYPE_GROUP_TIPS == message.getElemType()) {
            IMLog.e(TAG, "group tips message is not support");
            return null;
        }
        V2TIMMessage v2TIMMessage = new V2TIMMessage();
        Message nativeMessage = v2TIMMessage.getMessage();
        nativeMessage.setMessageBaseElements(message.getMessage().getMessageBaseElements());
        nativeMessage.setForward(true);

        return v2TIMMessage;
    }

    @Override
    public V2TIMMessage createTargetedGroupMessage(V2TIMMessage message, List<String> receiverList) {
        if (message == null) {
            IMLog.e(TAG, "createTargetedGroupMessage, message cannot be null");
            return null;
        }

        if (receiverList == null || receiverList.isEmpty()) {
            IMLog.e(TAG, "receiverList cannot be empty");
            return null;
        }

        // 不支持群 @ 消息
        List<String> atUserList = message.getGroupAtUserList();
        if (atUserList != null && !atUserList.isEmpty()) {
            IMLog.e(TAG, "targeted group message does not support at message");
            return null;
        }

        message.getMessage().setTargetGroupMemberList(receiverList);
        return message;
    }

    @Override
    public String sendMessage(final V2TIMMessage v2TIMMessage,
                              String receiver,
                              String groupID,
                              int priority,
                              boolean onlineUserOnly,
                              V2TIMOfflinePushInfo offlinePushInfo,
                              final V2TIMSendCallback<V2TIMMessage> sendCallback) {
        if (v2TIMMessage == null) {
            if (sendCallback != null) {
                sendCallback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "message is null");
            }
            return null;
        }

        if (TextUtils.isEmpty(receiver) && TextUtils.isEmpty(groupID)) {
            if (sendCallback != null) {
                sendCallback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "receiver and groupID cannot be empty at the same time!");
            }
            return null;
        }

        Message message = v2TIMMessage.getMessage();
        if (!TextUtils.isEmpty(groupID)) {
            message.setMessageType(Message.MESSAGE_TYPE_GROUP);
            message.setGroupID(groupID);
            List<String> targetGroupMemberList = message.getTargetGroupMemberList();
            // groupID 和 receiver 都有值，表示在群里给 receiver 发定向消息。
            // 如果 targetGroupMemberList 有值，表明已经通过 createTargetedGroupMessage 接口设置过接收成员列表，这里不再设置。
            if (!TextUtils.isEmpty(receiver) &&
                    (targetGroupMemberList == null || message.getTargetGroupMemberList().isEmpty())) {
                List<String> atUserList = v2TIMMessage.getGroupAtUserList();
                // 如果 atUserList 有值，表明是群 @ 消息，这里不支持设置定向消息。
                if (atUserList != null && !atUserList.isEmpty()) {
                    if (sendCallback != null) {
                        sendCallback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "targeted group message does not support group @ message");
                    }
                    return null;
                } else {
                    List<String> targetList = new ArrayList<>();
                    targetList.add(receiver);
                    message.setTargetGroupMemberList(targetList);
                }
            }
        } else {
            message.setMessageType(Message.MESSAGE_TYPE_C2C);
            message.setReceiverUserID(receiver);
        }

        message.setPriority(priority);
        if (onlineUserOnly) {
            message.setLifeTime(0);
        }

        if (offlinePushInfo != null) {
            message.setOfflinePushInfo(offlinePushInfo.getMessageOfflinePushInfo());
        }

        message.setPlatform(Message.PLATFORM_ANDROID);

        final MessageUploadProgressCallback progressCallback = new MessageUploadProgressCallback() {
            @Override
            public void onUploadProgress(int elemIndex, final int currentSize, final int totalSize) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        if (sendCallback != null) {
                            int progress = 0;
                            if (totalSize > 0) {
                                float decimalProgress = (float) currentSize / totalSize;
                                progress = (int) (decimalProgress * 100);
                            }
                            sendCallback.onProgress(progress);
                        }
                    }
                });
            }
        };

        V2TIMValueCallback<Message> v2TIMValueCallback = new V2TIMValueCallback<Message>() {
            @Override
            public void onSuccess(Message message) {
                if (sendCallback != null) {
                    v2TIMMessage.setMessage(message);
                    sendCallback.onSuccess(v2TIMMessage);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (sendCallback != null) {
                    sendCallback.onError(code, desc);
                }
            }
        };

        String messageID = MessageCenter.getInstance().sendMessage(message, progressCallback, new IMCallback(v2TIMValueCallback) {
            @Override
            public void success(Object data) {
                Message message = v2TIMMessage.getMessage();
                message.update((Message)data);
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage, Object data) {
                Message message = v2TIMMessage.getMessage();
                message.update((Message)data);
                super.fail(code, errorMessage, data);
            }
        });

        BaseManager.getInstance().checkTUIComponent(BaseManager.TUI_COMPONENT_CHAT);

        return messageID;
    }

    @Override
    public void setC2CReceiveMessageOpt(List<String> userIDList, int opt, V2TIMCallback callback) {
        if (userIDList == null || userIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid userIDList");
            }
            return;
        }

        int receive_option = UserInfo.USER_RECEIVE_MESSAGE_NATIVE;
        if (opt == V2TIMMessage.V2TIM_RECEIVE_MESSAGE) {
            receive_option = UserInfo.USER_RECEIVE_MESSAGE_NATIVE;
        } else if (opt == V2TIMMessage.V2TIM_NOT_RECEIVE_MESSAGE) {
            receive_option = UserInfo.USER_NOT_RECEIVE_MESSAGE_NATIVE;
        } else if (opt == V2TIMMessage.V2TIM_RECEIVE_NOT_NOTIFY_MESSAGE) {
            receive_option = UserInfo.USER_RECEIVE_NOT_NOTIFY_MESSAGE_NATIVE;
        } else {
            IMLog.e(TAG, "setC2CReceiveMessageOpt error opt = " + opt);
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "opt is error");
            }
            return;
        }

        RelationshipManager.getInstance().setC2CReceiveMessageOpt(userIDList, receive_option, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getC2CReceiveMessageOpt(List<String> userIDList, final V2TIMValueCallback<List<V2TIMReceiveMessageOptInfo>> callback) {
        if (userIDList == null || userIDList.size() == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid userIDList");
            }
            return;
        }

        V2TIMValueCallback<List<ReceiveMessageOptInfo>> v2Callback = new V2TIMValueCallback<List<ReceiveMessageOptInfo>>() {
            @Override
            public void onSuccess(List<ReceiveMessageOptInfo> userInfos) {
                if (callback != null) {
                    List<V2TIMReceiveMessageOptInfo> v2UserInfoList = new ArrayList<>();
                    for (ReceiveMessageOptInfo item : userInfos) {
                        V2TIMReceiveMessageOptInfo v2UserInfo = new V2TIMReceiveMessageOptInfo();
                        v2UserInfo.setUserID(item.getUserID());
                        v2UserInfo.setC2CReceiveMessageOpt(item.getC2CReceiveMessageOpt());
                        v2UserInfoList.add(v2UserInfo);
                    }
                    callback.onSuccess(v2UserInfoList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        RelationshipManager.getInstance().getC2CReceiveMessageOpt(userIDList, new IMCallback<List<ReceiveMessageOptInfo>>(v2Callback) {
            @Override
            public void success(List<ReceiveMessageOptInfo> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void setGroupReceiveMessageOpt(String groupID, int opt, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(groupID)) {
            IMLog.e(TAG, "setReceiveMessageOpt err, groupID is empty");
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
            }
            return;
        }

        int receive_option = GroupMemberInfo.MESSAGE_RECEIVE_OPTION_AUTO_RECEIVE;
        if (opt == V2TIMMessage.V2TIM_RECEIVE_MESSAGE) {
            receive_option = GroupMemberInfo.MESSAGE_RECEIVE_OPTION_AUTO_RECEIVE;
        } else if (opt == V2TIMMessage.V2TIM_NOT_RECEIVE_MESSAGE) {
            receive_option = GroupMemberInfo.MESSAGE_RECEIVE_OPTION_NOT_RECEIVE;
        } else if (opt == V2TIMMessage.V2TIM_RECEIVE_NOT_NOTIFY_MESSAGE) {
            receive_option = GroupMemberInfo.MESSAGE_RECEIVE_OPTION_RECEIVE_WITH_NO_OFFLINE_PUSH;
        } else {
            IMLog.e(TAG, "setReceiveMessageOpt error opt = " + opt);
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "opt is error");
            }
            return;
        }

        GroupManager.getInstance().setGroupReceiveMessageOpt(groupID, receive_option, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getC2CHistoryMessageList(String userID, int count, V2TIMMessage lastMsg, final V2TIMValueCallback<List<V2TIMMessage>> callback) {
        if (TextUtils.isEmpty(userID) || count == 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "userID is empty or count is 0");
            }
            return;
        }

        V2TIMValueCallback<List<Message>> v2TIMValueCallback = new V2TIMValueCallback<List<Message>>() {
            @Override
            public void onSuccess(List<Message> messageList) {
                List<V2TIMMessage> v2TIMMessageList = new ArrayList<>();

                for (Message message : messageList) {
                    V2TIMMessage v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);

                    v2TIMMessageList.add(v2TIMMessage);
                }

                if (callback != null) {
                    callback.onSuccess(v2TIMMessageList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }

        };

        MessageKey messageKey = null;
        if (lastMsg != null) {
            messageKey = lastMsg.getMessage().getMessageKey();
        }

        MessageListGetOption messageListGetOption = new MessageListGetOption();
        messageListGetOption.setCount(count);
        messageListGetOption.setToOlderMessage(true);
        messageListGetOption.setGetCloudMessage(true);
        messageListGetOption.setMessageKey(messageKey);

        MessageCenter.getInstance().getC2CHistoryMessageList(userID, messageListGetOption, new IMCallback<List<Message>>(v2TIMValueCallback) {
            @Override
            public void success(List<Message> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getGroupHistoryMessageList(String groupID, int count, V2TIMMessage lastMsg, final V2TIMValueCallback<List<V2TIMMessage>> callback) {
        if (TextUtils.isEmpty(groupID) || count <= 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty or count is 0");
            }
            return;
        }
        V2TIMValueCallback<List<Message>> v2TIMValueCallback = new V2TIMValueCallback<List<Message>>() {
            @Override
            public void onSuccess(List<Message> messageList) {
                List<V2TIMMessage> v2TIMMessageList = new ArrayList<>();

                for (Message message : messageList) {
                    V2TIMMessage v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);

                    v2TIMMessageList.add(v2TIMMessage);
                }

                if (callback != null) {
                    callback.onSuccess(v2TIMMessageList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }

        };

        MessageKey messageKey = null;
        if (lastMsg != null) {
            messageKey = lastMsg.getMessage().getMessageKey();
        }

        MessageListGetOption messageListGetOption = new MessageListGetOption();
        messageListGetOption.setCount(count);
        messageListGetOption.setToOlderMessage(true);
        messageListGetOption.setGetCloudMessage(true);
        messageListGetOption.setMessageKey(messageKey);

        MessageCenter.getInstance().getGroupHistoryMessageList(groupID, messageListGetOption, new IMCallback<List<Message>>(v2TIMValueCallback) {
            @Override
            public void success(List<Message> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getHistoryMessageList(V2TIMMessageListGetOption option, final V2TIMValueCallback<List<V2TIMMessage>> callback) {
        if (option == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "option is null");
            }
            return;
        }
        if (TextUtils.isEmpty(option.getUserID()) && TextUtils.isEmpty(option.getGroupID())) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID and userID cannot be null at the same time");
            }
            return;
        }
        if (!TextUtils.isEmpty(option.getUserID()) && !TextUtils.isEmpty(option.getGroupID())) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID and userID cannot set at the same time");
            }
            return;
        }
        if (option.getCount() <= 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "count less than 0");
            }
            return;
        }

        MessageListGetOption messageListGetOption = new MessageListGetOption();
        messageListGetOption.setCount(option.getCount());
        messageListGetOption.setGetTimeBegin(option.getGetTimeBegin());
        messageListGetOption.setGetTimePeriod(option.getGetTimePeriod());

        switch (option.getGetType()) {
            case V2TIMMessageListGetOption.V2TIM_GET_LOCAL_NEWER_MSG:
                messageListGetOption.setToOlderMessage(false);
                messageListGetOption.setGetCloudMessage(false);
                messageListGetOption.setMessageTypeList(option.getMessageTypeList());
                break;
            case V2TIMMessageListGetOption.V2TIM_GET_LOCAL_OLDER_MSG:
                messageListGetOption.setToOlderMessage(true);
                messageListGetOption.setGetCloudMessage(false);
                messageListGetOption.setMessageTypeList(option.getMessageTypeList());
                break;
            case V2TIMMessageListGetOption.V2TIM_GET_CLOUD_NEWER_MSG:
                messageListGetOption.setToOlderMessage(false);
                messageListGetOption.setGetCloudMessage(true);
                break;
            case V2TIMMessageListGetOption.V2TIM_GET_CLOUD_OLDER_MSG:
                messageListGetOption.setToOlderMessage(true);
                messageListGetOption.setGetCloudMessage(true);
                break;
            default:
                if (callback != null) {
                    callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "getType is invalid");
                }
                return;
        }

        if (option.getLastMsg() != null) {
            messageListGetOption.setMessageKey(option.getLastMsg().getMessage().getMessageKey());
        } else {
            if (!TextUtils.isEmpty(option.getGroupID())) {
                if (option.getLastMsgSeq() > 0) {
                    MessageKey messageKey = new MessageKey();
                    messageKey.setSeq(option.getLastMsgSeq());
                    messageListGetOption.setMessageKey(messageKey);
                }
            }
        }

        V2TIMValueCallback<List<Message>> v2TIMValueCallback = new V2TIMValueCallback<List<Message>>() {
            @Override
            public void onSuccess(List<Message> messageList) {
                List<V2TIMMessage> v2TIMMessageList = new ArrayList<>();

                for (Message message : messageList) {
                    V2TIMMessage v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);

                    v2TIMMessageList.add(v2TIMMessage);
                }

                if (callback != null) {
                    callback.onSuccess(v2TIMMessageList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }

        };

        MessageKey messageKey = null;
        if (option.getLastMsg() != null) {
            messageKey = option.getLastMsg().getMessage().getMessageKey();
        }
        if (!TextUtils.isEmpty(option.getUserID())) {
            MessageCenter.getInstance().getC2CHistoryMessageList(option.getUserID(), messageListGetOption, new IMCallback<List<Message>>(v2TIMValueCallback) {
                        @Override
                        public void success(List<Message> data) {
                            super.success(data);
                        }

                        @Override
                        public void fail(int code, String errorMessage) {
                            super.fail(code, errorMessage);
                        }
                    });
        } else {
            MessageCenter.getInstance().getGroupHistoryMessageList(option.getGroupID(), messageListGetOption, new IMCallback<List<Message>>(v2TIMValueCallback) {
                        @Override
                        public void success(List<Message> data) {
                            super.success(data);
                        }

                        @Override
                        public void fail(int code, String errorMessage) {
                            super.fail(code, errorMessage);
                        }
                    });
        }
    }

    @Override
    public void revokeMessage(V2TIMMessage v2TIMMessage, V2TIMCallback callback) {
        if (v2TIMMessage == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "msg is null");
            }
            return;
        }

        if (v2TIMMessage.getStatus() != V2TIMMessage.V2TIM_MSG_STATUS_SEND_SUCC) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "msg status must be V2TIM_MSG_STATUS_SEND_SUCC");
            }
            return;
        }

        MessageCenter.getInstance().revokeMessage(v2TIMMessage.getMessage().getMessageKey(), new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void modifyMessage(final V2TIMMessage msg, final V2TIMCompleteCallback<V2TIMMessage> callback) {
        if (msg == null) {
            if (callback != null) {
                callback.onComplete(BaseConstants.ERR_INVALID_PARAMETERS, "msg is null", null);
            }
            return;
        }

        V2TIMCompleteCallback<Message> v2TIMCompleteCallback = new V2TIMCompleteCallback<Message>() {
            @Override
            public void onComplete(int code, String desc, Message message) {
                V2TIMMessage v2TIMMessage = null;
                if (message != null) {
                    v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);
                }

                if (callback != null) {
                    callback.onComplete(code, desc, v2TIMMessage);
                }
            }
        };

        MessageCenter.getInstance().modifyMessage(msg.getMessage(), new IMCallback(v2TIMCompleteCallback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage, Object data) {
                super.fail(code, errorMessage, data);
            }
        });
    }

    @Override
    public void markC2CMessageAsRead(String userID, V2TIMCallback callback) {
        if (TextUtils.isEmpty(userID)) {
            // 清空所有 c2c 会话的未读消息数
            ConversationManager.getInstance().clearUnreadMessage(true, false, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        } else {
            // 标记指定 C2C 会话已读
            MessageCenter.getInstance().setC2CMessageRead(userID, 0, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        }
    }

    @Override
    public void markGroupMessageAsRead(String groupID, V2TIMCallback callback) {
        if (TextUtils.isEmpty(groupID)) {
            // 清空所有 Group 会话的未读消息数
            ConversationManager.getInstance().clearUnreadMessage(false, true, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        } else {
            // 标记指定的 Group 会话已读
            MessageCenter.getInstance().setGroupMessageRead(groupID, 0, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        }
    }

    @Override
    public void markAllMessageAsRead(V2TIMCallback callback) {
        ConversationManager.getInstance().clearUnreadMessage(true, true, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteMessageFromLocalStorage(V2TIMMessage v2TIMMessage, V2TIMCallback callback) {
        if (v2TIMMessage == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "message is null");
            }
            return;
        }

        MessageCenter.getInstance().deleteLocalMessage(v2TIMMessage.getMessage().getMessageKey(), new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteMessages(List<V2TIMMessage> messages, V2TIMCallback callback) {
        if (messages == null || messages.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "messages is invalid");
            }
            return;
        }

        List<MessageKey> messageKeyList = new ArrayList<>();
        for (V2TIMMessage v2TIMMessage : messages) {
            messageKeyList.add(v2TIMMessage.getMessage().getMessageKey());
        }

        MessageCenter.getInstance().deleteCloudMessageList(messageKeyList, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void clearC2CHistoryMessage(String userID, V2TIMCallback callback) {
        if (TextUtils.isEmpty(userID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "userID is empty");
            }
            return;
        }

        MessageCenter.getInstance().clearC2CHistoryMessage(userID, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void clearGroupHistoryMessage(String groupID, V2TIMCallback callback) {
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
            }
            return;
        }

        MessageCenter.getInstance().clearGroupHistoryMessage(groupID, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public String insertGroupMessageToLocalStorage(final V2TIMMessage v2TIMMessage, String groupID, String sender, final V2TIMValueCallback<V2TIMMessage> callback) {
        if (v2TIMMessage == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "message is null!");
            }
            return "";
        }
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
            }
            return "";
        }
        if (TextUtils.isEmpty(sender)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "sender is empty");
            }
            return "";
        }
        Message message = v2TIMMessage.getMessage();
        message.setMessageType(Message.MESSAGE_TYPE_GROUP);
        message.setSenderUserID(sender);
        message.setGroupID(groupID);

        V2TIMValueCallback<Message> v2TIMValueCallback = new V2TIMValueCallback<Message>() {
            @Override
            public void onSuccess(Message message) {
                if (callback != null) {
                    v2TIMMessage.setMessage(message);
                    callback.onSuccess(v2TIMMessage);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        String messageID = MessageCenter.getInstance().insertLocalMessage(message, new IMCallback(v2TIMValueCallback) {
            @Override
            public void success(Object data) {
                Message message = v2TIMMessage.getMessage();
                message.update((Message)data);
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });

        return messageID;
    }

    @Override
    public String insertC2CMessageToLocalStorage(final V2TIMMessage v2TIMMessage, String userID, String sender, final V2TIMValueCallback<V2TIMMessage> callback) {
        if (v2TIMMessage == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "message is null!");
            }
            return "";
        }
        if (TextUtils.isEmpty(userID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "userID is empty");
            }
            return "";
        }
        if (TextUtils.isEmpty(sender)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "sender is empty");
            }
            return "";
        }
        Message message = v2TIMMessage.getMessage();
        message.setMessageType(Message.MESSAGE_TYPE_C2C);
        message.setSenderUserID(sender);
        message.setReceiverUserID(userID);

        V2TIMValueCallback<Message> v2TIMValueCallback = new V2TIMValueCallback<Message>() {
            @Override
            public void onSuccess(Message message) {
                if (callback != null) {
                    v2TIMMessage.setMessage(message);
                    callback.onSuccess(v2TIMMessage);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        String messageID = MessageCenter.getInstance().insertLocalMessage(message, new IMCallback(v2TIMValueCallback) {
            @Override
            public void success(Object data) {
                Message message = v2TIMMessage.getMessage();
                message.update((Message)data);
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });

        return messageID;
    }

    @Override
    public void findMessages(List<String> messageIDList, final V2TIMValueCallback<List<V2TIMMessage>> callback) {
        if (messageIDList == null || messageIDList.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "messages is empty");
            }
            return;
        }
        V2TIMValueCallback<List<Message>> v2TIMValueCallback = new V2TIMValueCallback<List<Message>>() {
            @Override
            public void onSuccess(List<Message> messageList) {
                List<V2TIMMessage> v2TIMMessageList = new ArrayList<>();

                for (Message message : messageList) {
                    V2TIMMessage v2TIMMessage = new V2TIMMessage();
                    v2TIMMessage.setMessage(message);

                    v2TIMMessageList.add(v2TIMMessage);
                }

                if (callback != null) {
                    callback.onSuccess(v2TIMMessageList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }

        };
        MessageCenter.getInstance().findMessageByMessageId(messageIDList, new IMCallback<List<Message>>(v2TIMValueCallback) {
            @Override
            public void success(List<Message> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void searchLocalMessages(V2TIMMessageSearchParam param,
                                    final V2TIMValueCallback<V2TIMMessageSearchResult> callback) {
        if (param.getSearchTimePosition() < 0 || param.getSearchTimePeriod() < 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "time < 0");
            }
            return;
        }

        if (param.getPageIndex() < 0 || param.getPageSize() < 0) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "pageIndex or pageSize is invalid");
            }
            return;
        }

        V2TIMValueCallback<MessageSearchResult> v2TIMValueCallback = new V2TIMValueCallback<MessageSearchResult>() {
            @Override
            public void onSuccess(MessageSearchResult messageSearchResult) {
                V2TIMMessageSearchResult v2TIMMessageSearchResult = new V2TIMMessageSearchResult();
                v2TIMMessageSearchResult.setMessageSearchResult(messageSearchResult);

                if (callback != null) {
                    callback.onSuccess(v2TIMMessageSearchResult);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }
        };

        MessageCenter.getInstance().findMessageBySearchKey(param.getMessageSearchParam(), new IMCallback<MessageSearchResult>(v2TIMValueCallback) {
            @Override
            public void success(MessageSearchResult data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });

        BaseManager.getInstance().checkTUIComponent(BaseManager.TUI_COMPONENT_SEARCH);
    }

    @Override
    public void sendMessageReadReceipts(final List<V2TIMMessage> messageList, final V2TIMCallback callback) {
        if (messageList == null || 0 == messageList.size()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid messageList");
            }
            return;
        }

        final List<V2TIMMessage> copyMessageList = new ArrayList<>();
        copyMessageList.addAll(messageList);
        List<MessageKey> messageKeyList = new ArrayList<>();
        for (V2TIMMessage v2TIMMessage : copyMessageList) {
            // 只能对接收到的需要已读回执的群消息，发送已读回执
            if (null == v2TIMMessage || true == v2TIMMessage.isSelf() || false == v2TIMMessage.isNeedReadReceipt()) {
                continue;
            }

            // 已经发送了已读回执，不再重复发送已读回执
            Message message = v2TIMMessage.getMessage();
            if (message.isHasSentReceipt()) {
                continue;
            }

            MessageKey messageKey = message.getMessageKey();
            messageKeyList.add(messageKey);
        }

        if (0 == messageKeyList.size()) {
            if (callback != null) {
                callback.onSuccess();
            }
            return;
        }

        V2TIMCallback v2TIMCallback = new V2TIMCallback() {
            @Override
            public void onSuccess() {
                if (callback != null) {
                    callback.onSuccess();
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        MessageCenter.getInstance().sendMessageReceipts(messageKeyList, new IMCallback(v2TIMCallback) {
            @Override
            public void success(Object data) {
                for (V2TIMMessage v2TIMMessage : copyMessageList) {
                    Message message = v2TIMMessage.getMessage();
                    message.setHasSentReceipt(true);
                }

                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getMessageReadReceipts(List<V2TIMMessage> messageList, V2TIMValueCallback<List<V2TIMMessageReceipt>> callback) {
        if (messageList == null || 0 == messageList.size()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message list");
            }
            return;
        }

        final List<V2TIMMessage> copyMessageList = new ArrayList<>();
        copyMessageList.addAll(messageList);

        V2TIMMessage firstMessage = copyMessageList.get(0);
        if (firstMessage.getUserID() != null && !firstMessage.getUserID().isEmpty()) {
            getC2CMessageReadReceipts(copyMessageList, callback);
        } else {
            getGroupMessageReadReceipts(copyMessageList, callback);
        }
    }

    private void getC2CMessageReadReceipts(final List<V2TIMMessage> messageList, final V2TIMValueCallback<List<V2TIMMessageReceipt>> callback) {
        List<String> messageIDList = new ArrayList<>();
        final List<V2TIMMessageReceipt> results = new ArrayList<>();
        for (V2TIMMessage v2TIMMessage : messageList) {
            if (null == v2TIMMessage || null == v2TIMMessage.getUserID() || v2TIMMessage.getUserID().isEmpty() ||
                    false ==v2TIMMessage.isSelf() || V2TIMMessage.V2TIM_MSG_STATUS_SEND_SUCC != v2TIMMessage.getStatus() ||
                    false == v2TIMMessage.isNeedReadReceipt()) {
                continue;
            }

            messageIDList.add(v2TIMMessage.getMsgID());
        }
        if (0 == messageIDList.size()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message list");
            }
            return;
        }

        // 消息的已读状态可能发生了变化，这里本地重新加载下消息
        findMessages(messageIDList, new V2TIMValueCallback<List<V2TIMMessage>>() {
            @Override
            public void onSuccess(List<V2TIMMessage> v2TIMMessages) {
                for (V2TIMMessage localMessage : v2TIMMessages) {
                    Message message = localMessage.getMessage();
                    C2CMessageReceipt receipt = new C2CMessageReceipt();
                    receipt.setUserID(message.getReceiverUserID());
                    receipt.setMessageID(message.getMsgID());
                    receipt.setPeerRead(message.isReceiptPeerRead());

                    V2TIMMessageReceipt v2TIMMessageReceipt = new V2TIMMessageReceipt();
                    v2TIMMessageReceipt.setC2CMessageReceipt(receipt);

                    results.add(v2TIMMessageReceipt);
                }

                if (results.size() > 0) {
                    if (callback != null) {
                        callback.onSuccess(results);
                    }
                } else {
                    if (callback != null) {
                        callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message list");
                    }
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        });

    }

    private void getGroupMessageReadReceipts(final List<V2TIMMessage> messageList, final V2TIMValueCallback<List<V2TIMMessageReceipt>> callback) {
        List<MessageKey> messageKeyList = new ArrayList<>();
        final List<V2TIMMessageReceipt> results = new ArrayList<>();
        for (V2TIMMessage v2TIMMessage : messageList) {
            // 只能针对自己成功发送的需要已读回执的群消息，获取已读回执
            if (null == v2TIMMessage || null == v2TIMMessage.getGroupID() || 0 == v2TIMMessage.getGroupID().length() ||
                    false == v2TIMMessage.isSelf() || V2TIMMessage.V2TIM_MSG_STATUS_SEND_SUCC != v2TIMMessage.getStatus() ||
                    false == v2TIMMessage.isNeedReadReceipt()) {
                continue;
            }

            // 对于已经全部已读的消息，不需要再次请求已读回执了
            Message message = v2TIMMessage.getMessage();
            if (0 == message.getReceiptUnreadCount()) {
                GroupMessageReceipt receipt = new GroupMessageReceipt();
                receipt.setGroupID(message.getGroupID());
                receipt.setMsgID(message.getMsgID());
                receipt.setReadCount(message.getReceiptReadCount());
                receipt.setUnreadCount(message.getReceiptUnreadCount());

                V2TIMMessageReceipt v2Receipt = new V2TIMMessageReceipt();
                v2Receipt.setGroupMessageReceipt(receipt);
                results.add(v2Receipt);
                continue;
            }

            MessageKey messageKey = message.getMessageKey();
            messageKeyList.add(messageKey);
        }

        if (0 == messageKeyList.size()) {
            if (results.size() > 0) {
                if (callback != null) {
                    callback.onSuccess(results);
                }
            } else {
                if (callback != null) {
                    callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid messages");
                }
            }
            return;
        }

        V2TIMValueCallback<List<GroupMessageReceipt>> v2TIMValueCallback = new V2TIMValueCallback<List<GroupMessageReceipt>>() {
            @Override
            public void onSuccess(List<GroupMessageReceipt> groupMessageReceipts) {
                for (GroupMessageReceipt groupReceipt : groupMessageReceipts) {
                    V2TIMMessageReceipt v2Receipt = new V2TIMMessageReceipt();

                    GroupMessageReceipt receipt = new GroupMessageReceipt();
                    receipt.setGroupID(groupReceipt.getGroupID());
                    receipt.setMsgID(groupReceipt.getMsgID());
                    receipt.setReadCount(groupReceipt.getReadCount());
                    receipt.setUnreadCount(groupReceipt.getUnreadCount());

                    v2Receipt.setGroupMessageReceipt(receipt);
                    results.add(v2Receipt);
                }
                if (callback != null) {
                    callback.onSuccess(results);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        MessageCenter.getInstance().getGroupMessageReceipts(messageKeyList, new IMCallback<List<GroupMessageReceipt>>(v2TIMValueCallback) {
            @Override
            public void success(List<GroupMessageReceipt> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    public void getGroupMessageReadMemberList(V2TIMMessage v2TIMMessage, final int filter, long nextSeq, int count, final V2TIMValueCallback<V2TIMGroupMessageReadMemberList> callback) {
        // 只能针对自己成功发送的群消息，获取已读群成员列表
        if (null == v2TIMMessage || null == v2TIMMessage.getGroupID() || 0 == v2TIMMessage.getGroupID().length() || false == v2TIMMessage.isSelf() || V2TIMMessage.V2TIM_MSG_STATUS_SEND_SUCC != v2TIMMessage.getStatus() || false == v2TIMMessage.isNeedReadReceipt()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message");
            }
            return;
        }

        if (count > 100) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid count, maximum support 100");
            }
            return;
        }

        Message message = v2TIMMessage.getMessage();

        V2TIMValueCallback<GroupMessageReadMembers> v2TIMValueCallback = new V2TIMValueCallback<GroupMessageReadMembers>() {
            @Override
            public void onSuccess(GroupMessageReadMembers readMembers) {
                if (callback != null) {
                    V2TIMGroupMessageReadMemberList v2ReadMembers = new V2TIMGroupMessageReadMemberList();
                    v2ReadMembers.setMessageReadMembers(readMembers, filter);
                    callback.onSuccess(v2ReadMembers);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }
        };

        MessageCenter.getInstance().getGroupMessageReadMembers(message, filter, nextSeq, count, new IMCallback<GroupMessageReadMembers>(v2TIMValueCallback) {
            @Override
            public void success(GroupMessageReadMembers data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void setMessageExtensions(V2TIMMessage message, List<V2TIMMessageExtension> extensions,
                                     final V2TIMValueCallback<List<V2TIMMessageExtensionResult>> callback) {
        if (message == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message");
            }
            return;
        }
        if (extensions == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "extensions cannot be null");
            }
            return;
        }
        List<MessageExtension> messageExtensions = new ArrayList<>();
        for (V2TIMMessageExtension v2TIMMessageExtension : extensions) {
            if (TextUtils.isEmpty(v2TIMMessageExtension.getExtensionKey())) {
                if (callback != null) {
                    callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "empty extensionKey");
                }
                return;
            }
            MessageExtension messageExtension = new MessageExtension();
            messageExtension.setExtensionKey(v2TIMMessageExtension.getExtensionKey());
            messageExtension.setExtensionValue(v2TIMMessageExtension.getExtensionValue());
            messageExtensions.add(messageExtension);
        }

        V2TIMValueCallback<List<MessageExtensionResult>> v2TIMValueCallback = new V2TIMValueCallback<List<MessageExtensionResult>>() {
            @Override
            public void onSuccess(List<MessageExtensionResult> messageExtensionResults) {
                if (callback != null) {
                    List<V2TIMMessageExtensionResult> v2TIMMessageExtensionResults = new ArrayList<>();
                    for (MessageExtensionResult result : messageExtensionResults) {
                        V2TIMMessageExtensionResult v2TIMMessageExtensionResult = new V2TIMMessageExtensionResult();
                        v2TIMMessageExtensionResult.setMessageExtensionResult(result);
                        v2TIMMessageExtensionResults.add(v2TIMMessageExtensionResult);
                    }
                    callback.onSuccess(v2TIMMessageExtensionResults);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }
        };

        MessageCenter.getInstance().setMessageExtensions(message.getMessage(), messageExtensions, new IMCallback<List<MessageExtensionResult>>(v2TIMValueCallback) {

            @Override
            public void success(List<MessageExtensionResult> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getMessageExtensions(V2TIMMessage message, final V2TIMValueCallback<List<V2TIMMessageExtension>> callback) {
        if (message == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message");
            }
            return;
        }

        V2TIMValueCallback<List<MessageExtension>> v2TIMValueCallback = new V2TIMValueCallback<List<MessageExtension>>() {
            @Override
            public void onSuccess(List<MessageExtension> extensions) {
                if (callback != null) {
                    List<V2TIMMessageExtension> v2TIMMessageExtensions = new ArrayList<>();
                    for (MessageExtension extension : extensions) {
                        V2TIMMessageExtension v2TIMMessageExtension = new V2TIMMessageExtension();
                        v2TIMMessageExtension.setMessageExtension(extension);
                        v2TIMMessageExtensions.add(v2TIMMessageExtension);
                    }
                    callback.onSuccess(v2TIMMessageExtensions);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }
        };

        MessageCenter.getInstance().getMessageExtensions(message.getMessage(), new IMCallback<List<MessageExtension>>(v2TIMValueCallback) {
            @Override
            public void success(List<MessageExtension> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteMessageExtensions(V2TIMMessage message, List<String> keys, final V2TIMValueCallback<List<V2TIMMessageExtensionResult>> callback) {
        if (message == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid message");
            }
            return;
        }

        V2TIMValueCallback<List<MessageExtensionResult>> v2TIMValueCallback = new V2TIMValueCallback<List<MessageExtensionResult>>() {
            @Override
            public void onSuccess(List<MessageExtensionResult> messageExtensionResults) {
                if (callback != null) {
                    List<V2TIMMessageExtensionResult> v2TIMMessageExtensionResults = new ArrayList<>();
                    for (MessageExtensionResult result : messageExtensionResults) {
                        V2TIMMessageExtensionResult v2TIMMessageExtensionResult = new V2TIMMessageExtensionResult();
                        v2TIMMessageExtensionResult.setMessageExtensionResult(result);
                        v2TIMMessageExtensionResults.add(v2TIMMessageExtensionResult);
                    }
                    callback.onSuccess(v2TIMMessageExtensionResults);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code ,desc);
                }
            }
        };

        MessageCenter.getInstance().deleteMessageExtensions(message.getMessage(), keys, new IMCallback(v2TIMValueCallback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }
}
