/*
 * Tencent is pleased to support the open source community by making wechat-matrix available.
 * Copyright (C) 2021 THL A29 Limited, a Tencent company. All rights reserved.
 * Licensed under the BSD 3-Clause License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://opensource.org/licenses/BSD-3-Clause
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.tencent.matrix.trace.view;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.DashPathEffect;
import android.graphics.Paint;
import android.graphics.Path;
import android.os.Build;
import android.text.TextPaint;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.RequiresApi;

import com.tencent.matrix.trace.R;
import com.tencent.matrix.trace.constants.Constants;

import java.util.LinkedList;

@RequiresApi(Build.VERSION_CODES.N)
public class FloatFrameView extends LinearLayout {

    public TextView fpsView;
    public TextView sceneView;
    public LineChartView chartView;

    public TextView extraInfoView;
    public TextView unknownDelayDurationView;
    public TextView inputHandlingDurationView;
    public TextView animationDurationView;
    public TextView layoutMeasureDurationView;
    public TextView drawDurationView;
    public TextView syncDurationView;
    public TextView commandIssueDurationView;
    public TextView swapBuffersDurationView;
    public TextView gpuDurationView;
    public TextView totalDurationView;

    public TextView sumNormalView;
    public TextView sumMiddleView;
    public TextView sumHighView;
    public TextView sumFrozenView;
    public TextView levelNormalView;
    public TextView levelMiddleView;
    public TextView levelHighView;
    public TextView levelFrozenView;

    public FloatFrameView(Context context) {
        super(context);
        initView(context);
    }

    public FloatFrameView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initView(context);
    }


    private void initView(Context context) {
        setLayoutParams(new ViewGroup.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT));
        LayoutInflater.from(context).inflate(R.layout.float_frame_view, this);
        fpsView = findViewById(R.id.fps_view);
        extraInfoView = findViewById(R.id.extra_info);
        sceneView = findViewById(R.id.scene_view);
        extraInfoView.setText("{other info}");

        unknownDelayDurationView = findViewById(R.id.unknown_delay_duration_tv);
        inputHandlingDurationView = findViewById(R.id.input_handling_duration_tv);
        animationDurationView = findViewById(R.id.animation_duration_tv);
        layoutMeasureDurationView = findViewById(R.id.layout_measure_duration_tv);
        drawDurationView = findViewById(R.id.draw_duration_tv);
        syncDurationView = findViewById(R.id.sync_duration_tv);
        commandIssueDurationView = findViewById(R.id.command_issue_duration_tv);
        swapBuffersDurationView = findViewById(R.id.swap_buffers_duration_tv);
        gpuDurationView = findViewById(R.id.gpu_duration_tv);
        totalDurationView = findViewById(R.id.total_duration_tv);

        sumNormalView = findViewById(R.id.sum_normal);
        sumMiddleView = findViewById(R.id.sum_middle);
        sumHighView = findViewById(R.id.sum_high);
        sumFrozenView = findViewById(R.id.sum_frozen);
        levelNormalView = findViewById(R.id.level_normal);
        levelMiddleView = findViewById(R.id.level_middle);
        levelHighView = findViewById(R.id.level_high);
        levelFrozenView = findViewById(R.id.level_frozen);

        chartView = findViewById(R.id.chart);
    }


    public static class LineChartView extends View {

        private final Paint paint;
        private final TextPaint tipPaint;
        private final Paint levelLinePaint;
        private final Paint tipLinePaint;
        private final static int LINE_COUNT = 50;
        private final LinkedList<LineInfo> lines;
        float linePadding;
        float lineStrokeWidth;
        private Path topPath = new Path();
        private Path middlePath = new Path();
        private float[] topTip = new float[2];
        private float[] middleTip = new float[2];

        private int bestColor = getContext().getResources().getColor(R.color.level_best_color);
        private int normalColor = getContext().getResources().getColor(R.color.level_normal_color);
        private int middleColor = getContext().getResources().getColor(R.color.level_middle_color);
        private int highColor = getContext().getResources().getColor(R.color.level_high_color);
        private int frozenColor = getContext().getResources().getColor(R.color.level_frozen_color);

        private int grayColor = getContext().getResources().getColor(R.color.dark_text);
        float padding = dip2px(getContext(), 8);
        float width;
        float lineContentWidth;
        float height;
        float textSize;

        public LineChartView(Context context, AttributeSet attrs, int defStyleAttr) {
            super(context, attrs, defStyleAttr);
            paint = new Paint();
            tipPaint = new TextPaint(Paint.ANTI_ALIAS_FLAG);
            tipPaint.setTextSize(textSize = dip2px(getContext(), 8));
            tipPaint.setStrokeWidth(dip2px(getContext(), 1));
            tipPaint.setColor(grayColor);

            levelLinePaint = new TextPaint(Paint.ANTI_ALIAS_FLAG);
            levelLinePaint.setStrokeWidth(dip2px(getContext(), 1));
            levelLinePaint.setStyle(Paint.Style.STROKE);

            levelLinePaint.setPathEffect(new DashPathEffect(new float[]{6, 6}, 0));

            tipLinePaint = new Paint(tipPaint);
            tipLinePaint.setStrokeWidth(dip2px(getContext(), 1));
            tipLinePaint.setColor(grayColor);
            tipLinePaint.setStyle(Paint.Style.STROKE);
            tipLinePaint.setPathEffect(new DashPathEffect(new float[]{6, 6}, 0));
            lines = new LinkedList<>();
        }

        public LineChartView(Context context, AttributeSet attrs) {
            this(context, attrs, 0);
        }

        @Override
        protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
            super.onLayout(changed, left, top, right, bottom);
            if (changed) {
                width = getMeasuredWidth();
                height = getMeasuredHeight();

                lineContentWidth = width - 3 * padding;

                lineStrokeWidth = dip2px(getContext(), 1);
                paint.setStrokeWidth(lineStrokeWidth);
                linePadding = lineStrokeWidth * 2;

                float rate = lineContentWidth / 60;
                topTip[0] = 10 * rate + (width - lineContentWidth);
                topTip[1] = LINE_COUNT * linePadding + padding;
                topPath.moveTo(topTip[0], topTip[1]);
                topPath.lineTo(topTip[0], 0);

                middleTip[0] = 30 * rate + (width - lineContentWidth);
                middleTip[1] = LINE_COUNT * linePadding + padding;
                middlePath.moveTo(middleTip[0], middleTip[1]);
                middlePath.lineTo(middleTip[0], 0);
            }
        }

        public void addFps(int fps, int color) {
            LineInfo linePoint = new LineInfo(fps, color);
            if (lines.size() >= LINE_COUNT) {
                lines.removeLast();
            }
            lines.addFirst(linePoint);
            invalidate();
        }

        @SuppressLint("DefaultLocale")
        @Override
        public void draw(Canvas canvas) {
            super.draw(canvas);
            int index = 1;
            int sumFps = 0;
            for (LineInfo lineInfo : lines) {
                sumFps += lineInfo.fps;
                lineInfo.draw(canvas, index);
                if (index % 25 == 0) {
                    Path path = new Path();
                    float pathY = lineInfo.linePoint[1];
                    path.moveTo(0, pathY);
                    path.lineTo(getMeasuredHeight(), pathY);
                    canvas.drawPath(path, tipLinePaint);
                    tipPaint.setColor(grayColor);
                    canvas.drawText(index / 5 + "s", 0, pathY + textSize, tipPaint);
                    if (index > 0) {
                        int aver = sumFps / index;
                        tipPaint.setColor(getColor(aver));
                        canvas.drawText(aver + "FPS", 0, pathY - textSize / 2, tipPaint);
                    }
                }
                index++;
            }
            tipPaint.setColor(grayColor);
            levelLinePaint.setColor(normalColor);
            canvas.drawPath(topPath, levelLinePaint);
            canvas.drawText("50", topTip[0] - textSize / 2, topTip[1] + textSize, tipPaint);
            levelLinePaint.setColor(middleColor);
            canvas.drawPath(middlePath, levelLinePaint);
            canvas.drawText("30", middleTip[0] - textSize / 2, middleTip[1] + textSize, tipPaint);

        }

        public static int dip2px(Context context, float dpValue) {
            final float scale = context.getResources().getDisplayMetrics().density;
            return (int) (dpValue * scale + 0.5f);
        }

        class LineInfo {
            private float[] linePoint = new float[4];

            LineInfo(int fps, int color) {
                this.fps = fps;
                this.color = color;

                linePoint[0] = width; // startX
                linePoint[2] = (60 - fps) * (lineContentWidth) / 60 + (getWidth() - lineContentWidth); // endX

            }

            void draw(Canvas canvas, int index) {
                if (paint.getColor() != color) {
                    paint.setColor(color);
                }
                linePoint[1] = (1 + index) * linePadding; // startY
                linePoint[3] = linePoint[1]; // endY
                canvas.drawLine(linePoint[0], linePoint[1], linePoint[2], linePoint[3], paint);
            }

            int fps;
            int color;
        }

        private int getColor(int fps) {
            int color;
            if (fps > 60 - Constants.DEFAULT_DROPPED_NORMAL) {
                color = bestColor;
            } else if (fps > 60 - Constants.DEFAULT_DROPPED_MIDDLE) {
                color = normalColor;
            } else if (fps > 60 - Constants.DEFAULT_DROPPED_HIGH) {
                color = middleColor;
            } else if (fps > 60 - Constants.DEFAULT_DROPPED_FROZEN) {
                color = highColor;
            } else {
                color = frozenColor;
            }
            return color;
        }
    }


}