/*
 * Decompiled with CFR 0.152.
 */
package shaded.org.apache.tsfile.compress;

import com.github.luben.zstd.Zstd;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import net.jpountz.lz4.LZ4Factory;
import org.tukaani.xz.LZMA2Options;
import org.tukaani.xz.XZInputStream;
import org.tukaani.xz.XZOutputStream;
import org.xerial.snappy.Snappy;
import shaded.org.apache.tsfile.common.conf.TSFileDescriptor;
import shaded.org.apache.tsfile.exception.compress.CompressionTypeNotSupportedException;
import shaded.org.apache.tsfile.exception.compress.GZIPCompressOverflowException;
import shaded.org.apache.tsfile.file.metadata.enums.CompressionType;

public interface ICompressor
extends Serializable {
    public static ICompressor getCompressor(String name) {
        return ICompressor.getCompressor(CompressionType.valueOf(name));
    }

    public static ICompressor getCompressor(CompressionType name) {
        if (name == null) {
            throw new CompressionTypeNotSupportedException("NULL");
        }
        switch (name) {
            case UNCOMPRESSED: {
                return new NoCompressor();
            }
            case SNAPPY: {
                return new SnappyCompressor();
            }
            case LZ4: {
                return new LZ4Compressor();
            }
            case GZIP: {
                return new GZIPCompressor();
            }
            case ZSTD: {
                return new ZstdCompressor();
            }
            case LZMA2: {
                return new LZMA2Compressor();
            }
        }
        throw new CompressionTypeNotSupportedException(name.toString());
    }

    public byte[] compress(byte[] var1) throws IOException;

    public byte[] compress(byte[] var1, int var2, int var3) throws IOException;

    public int compress(byte[] var1, int var2, int var3, byte[] var4) throws IOException;

    public int compress(ByteBuffer var1, ByteBuffer var2) throws IOException;

    public int getMaxBytesForCompression(int var1);

    public CompressionType getType();

    public static class NoCompressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) {
            return data;
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            throw new IOException("No Compressor does not support compression function");
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            throw new IOException("No Compressor does not support compression function");
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            throw new IOException("No Compressor does not support compression function");
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return uncompressedDataSize;
        }

        @Override
        public CompressionType getType() {
            return CompressionType.UNCOMPRESSED;
        }
    }

    public static class SnappyCompressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) throws IOException {
            if (data == null) {
                return new byte[0];
            }
            return Snappy.compress(data);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            byte[] maxCompressed = new byte[this.getMaxBytesForCompression(length)];
            int compressedSize = Snappy.compress(data, offset, length, maxCompressed, 0);
            byte[] compressed = null;
            if (compressedSize < maxCompressed.length) {
                compressed = new byte[compressedSize];
                System.arraycopy(maxCompressed, 0, compressed, 0, compressedSize);
            } else {
                compressed = maxCompressed;
            }
            return compressed;
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            return Snappy.compress(data, offset, length, compressed, 0);
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            return Snappy.compress(data, compressed);
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return Snappy.maxCompressedLength(uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.SNAPPY;
        }
    }

    public static class LZ4Compressor
    implements ICompressor {
        private static final LZ4Factory factory = TSFileDescriptor.getInstance().getConfig().isLz4UseJni() ? LZ4Factory.fastestInstance() : LZ4Factory.safeInstance();
        private static final net.jpountz.lz4.LZ4Compressor compressor = factory.fastCompressor();

        public static LZ4Factory getFactory() {
            return factory;
        }

        @Override
        public byte[] compress(byte[] data) {
            if (data == null) {
                return new byte[0];
            }
            return compressor.compress(data);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            return compressor.compress(data, offset, length);
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) {
            return compressor.compress(data, offset, length, compressed, 0);
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) {
            int startPosition = compressed.position();
            compressor.compress(data, compressed);
            return compressed.position() - startPosition;
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return compressor.maxCompressedLength(uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.LZ4;
        }
    }

    public static class GZIPCompressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) throws IOException {
            if (null == data) {
                return new byte[0];
            }
            return GZIPCompress.compress(data);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            return GZIPCompress.compress(dataBefore);
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            byte[] res = GZIPCompress.compress(dataBefore);
            if (res.length > compressed.length) {
                throw new GZIPCompressOverflowException();
            }
            System.arraycopy(res, 0, compressed, 0, res.length);
            return res.length;
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            int length = data.remaining();
            byte[] dataBefore = new byte[length];
            data.get(dataBefore, 0, length);
            byte[] res = GZIPCompress.compress(dataBefore);
            if (res.length > compressed.capacity()) {
                throw new GZIPCompressOverflowException();
            }
            compressed.put(res);
            return res.length;
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return Math.max(40 + uncompressedDataSize / 2, uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.GZIP;
        }
    }

    public static class ZstdCompressor
    implements ICompressor {
        private int compressionLevel = Zstd.maxCompressionLevel();

        @Override
        public byte[] compress(byte[] data) throws IOException {
            return Zstd.compress(data, this.compressionLevel);
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            if (data == null) {
                return new byte[0];
            }
            byte[] compressedData = new byte[this.getMaxBytesForCompression(length)];
            int compressedSize = this.compress(data, offset, length, compressedData);
            byte[] result = new byte[compressedSize];
            System.arraycopy(compressedData, 0, result, 0, compressedSize);
            return result;
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            return (int)Zstd.compressByteArray(compressed, 0, compressed.length, data, offset, length, this.compressionLevel);
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            return Zstd.compress(compressed, data, this.compressionLevel);
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return (int)Zstd.compressBound(uncompressedDataSize);
        }

        @Override
        public CompressionType getType() {
            return CompressionType.ZSTD;
        }
    }

    public static class LZMA2Compressor
    implements ICompressor {
        @Override
        public byte[] compress(byte[] data) throws IOException {
            if (null == data) {
                return new byte[0];
            }
            byte[] r = LZMA2Compress.compress(data);
            return r;
        }

        @Override
        public byte[] compress(byte[] data, int offset, int length) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            byte[] r = LZMA2Compress.compress(dataBefore);
            return r;
        }

        @Override
        public int compress(byte[] data, int offset, int length, byte[] compressed) throws IOException {
            byte[] dataBefore = new byte[length];
            System.arraycopy(data, offset, dataBefore, 0, length);
            byte[] res = LZMA2Compress.compress(dataBefore);
            System.arraycopy(res, 0, compressed, 0, res.length);
            return res.length;
        }

        @Override
        public int compress(ByteBuffer data, ByteBuffer compressed) throws IOException {
            int length = data.remaining();
            byte[] dataBefore = new byte[length];
            data.get(dataBefore, 0, length);
            byte[] res = LZMA2Compress.compress(dataBefore);
            compressed.put(res);
            return res.length;
        }

        @Override
        public int getMaxBytesForCompression(int uncompressedDataSize) {
            return 100 + uncompressedDataSize;
        }

        @Override
        public CompressionType getType() {
            return CompressionType.LZMA2;
        }
    }

    public static class LZMA2Compress {
        public static byte[] compress(byte[] data) throws IOException {
            LZMA2Options options = new LZMA2Options();
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            XZOutputStream lzma2 = new XZOutputStream((OutputStream)out, options);
            lzma2.write(data);
            lzma2.close();
            byte[] r = out.toByteArray();
            return r;
        }

        public static byte[] uncompress(byte[] data) throws IOException {
            int n;
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ByteArrayInputStream in = new ByteArrayInputStream(data);
            XZInputStream unlzma2 = new XZInputStream(in);
            byte[] buffer = new byte[256];
            while ((n = unlzma2.read(buffer)) > 0) {
                out.write(buffer, 0, n);
            }
            in.close();
            byte[] r = out.toByteArray();
            return r;
        }
    }

    public static class GZIPCompress {
        public static byte[] compress(byte[] data) throws IOException {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            GZIPOutputStream gzip = new GZIPOutputStream(out);
            gzip.write(data);
            gzip.close();
            return out.toByteArray();
        }

        public static byte[] uncompress(byte[] data) throws IOException {
            int n;
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ByteArrayInputStream in = new ByteArrayInputStream(data);
            GZIPInputStream ungzip = new GZIPInputStream(in);
            byte[] buffer = new byte[256];
            while ((n = ungzip.read(buffer)) > 0) {
                out.write(buffer, 0, n);
            }
            in.close();
            return out.toByteArray();
        }
    }
}

