/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.tom_roush.pdfbox.pdfparser;

import android.util.Log;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import com.tom_roush.pdfbox.cos.COSBase;
import com.tom_roush.pdfbox.cos.COSDictionary;
import com.tom_roush.pdfbox.cos.COSDocument;
import com.tom_roush.pdfbox.cos.COSName;
import com.tom_roush.pdfbox.io.IOUtils;
import com.tom_roush.pdfbox.io.RandomAccessBuffer;
import com.tom_roush.pdfbox.io.RandomAccessFile;

public class FDFParser extends COSParser
{
    /**
     * Constructs parser for given file using memory buffer.
     *
     * @param filename the filename of the pdf to be parsed
     *
     * @throws IOException If something went wrong.
     */
    public FDFParser(String filename) throws IOException
    {
        this(new File(filename));
    }

    /**
     * Constructs parser for given file using given buffer for temporary
     * storage.
     *
     * @param file the pdf to be parsed
     *
     * @throws IOException If something went wrong.
     */
    public FDFParser(File file) throws IOException
    {
        super(new RandomAccessFile(file, "r"));
        fileLen = file.length();
        init();
    }

    /**
     * Constructor.
     *
     * @param input input stream representing the pdf.
     * @throws IOException If something went wrong.
     */
    public FDFParser(InputStream input) throws IOException
    {
        super(new RandomAccessBuffer(input));
        fileLen = source.length();
        init();
    }

    /**
     * Tell if the dictionary is a FDF catalog.
     *
     * @param dictionary
     * @return
     */
    @Override
    protected final boolean isCatalog(COSDictionary dictionary)
    {
        return dictionary.containsKey(COSName.FDF);
    }

    private void init()
    {
        String eofLookupRangeStr = System.getProperty(SYSPROP_EOFLOOKUPRANGE);
        if (eofLookupRangeStr != null)
        {
            try
            {
                setEOFLookupRange(Integer.parseInt(eofLookupRangeStr));
            }
            catch (NumberFormatException nfe)
            {
                Log.w("PdfBox-Android", "System property " + SYSPROP_EOFLOOKUPRANGE
                    + " does not contain an integer value, but: '" + eofLookupRangeStr + "'");
            }
        }
        document = new COSDocument();
    }

    /**
     * The initial parse will first parse only the trailer, the xrefstart and all xref tables to have a pointer (offset)
     * to all the pdf's objects. It can handle linearized pdfs, which will have an xref at the end pointing to an xref
     * at the beginning of the file. Last the root object is parsed.
     *
     * @throws IOException If something went wrong.
     */
    private void initialParse() throws IOException
    {
        COSDictionary trailer = null;
        boolean rebuildTrailer = false;
        try
        {
            // parse startxref
            long startXRefOffset = getStartxrefOffset();
            if (startXRefOffset > 0)
            {
                trailer = parseXref(startXRefOffset);
            }
            else if (isLenient())
            {
                rebuildTrailer = true;
            }
        }
        catch (IOException exception)
        {
            if (isLenient())
            {
                rebuildTrailer = true;
            }
            else
            {
                throw exception;
            }
        }
        if (rebuildTrailer)
        {
            trailer = rebuildTrailer();
        }

        COSBase rootObject = parseTrailerValuesDynamically(trailer);

        // resolve all objects
        // A FDF doesn't have a catalog, all FDF fields are within the root object
        if (rootObject instanceof COSDictionary)
        {
            parseDictObjects((COSDictionary) rootObject, (COSName[]) null);
        }
        initialParseDone = true;
    }

    /**
     * This will parse the stream and populate the COSDocument object.
     *
     * @throws IOException If there is an error reading from the stream or corrupt data
     * is found.
     */
    public void parse() throws IOException
    {
        // set to false if all is processed
        boolean exceptionOccurred = true;
        try
        {
            if (!parseFDFHeader())
            {
                throw new IOException( "Error: Header doesn't contain versioninfo" );
            }
            initialParse();
            exceptionOccurred = false;
        }
        finally
        {
            if (exceptionOccurred && document != null)
            {
                IOUtils.closeQuietly(document);
                document = null;
            }
        }
    }
}
