package com.truv.models

import org.json.JSONException
import org.json.JSONObject

data class TruvEventPayload(
    val payload: Payload?,
    val eventType: EventType
) {
    data class Payload(
        val bridgeToken: String?,
        val productType: String?,
        val viewName: String?,
        val employer: TruvEmployer?,
        val publicToken: String?,
        val taskId: String?,
        val providerId: String?,
        val error: TruvError?,
        var url: String?,
        var action: ManualDdsActionType?,
        var depositType: String?,
        var dataSource: String?,
        var docType: String?
    ){
        fun toJson(): JSONObject {
            return JSONObject().apply {
                put("bridgeToken", bridgeToken)
                put("productType", productType)
                put("viewName", viewName)
                put("employer", employer?.toJson())
                put("publicToken", publicToken)
                put("taskId", taskId)
                put("providerId", providerId)
                put("error", error?.toJson())
                put("url", url)
                put("action", action)
                put("depositType", depositType)
                put("dataSource", dataSource)
                put("docType", docType)
            }
        }
    }

    enum class EventType(val event: String) {
        LOAD("LOAD"),
        OPEN("OPEN"),
        SCREEN_VIEW("SCREEN_VIEW"),
        EMPLOYER_SELECTED("EMPLOYER_SELECTED"),
        LINK_CREATED("LINK_CREATED"),
        LOGIN_COMPLETE("LOGIN_COMPLETE"),
        SUCCESS("SUCCESS"),
        ERROR("ERROR"),
        UNSUPPORTED_BROWSER("UNSUPPORTED_BROWSER"),
        OAUTH_OPENED("OAUTH_OPENED"),
        CLOSE("CLOSE"),
        MANUAL_DDS("MANUAL_DDS"),
        DOCUMENTS_UPLOADED("DOCUMENTS_UPLOADED"),
        DOC_UPLOAD_OPENED("DOC_UPLOAD_OPENED")
    }

    enum class ManualDdsActionType(val action: String) {
        EXTERNAL_FLOW("EXTERNAL_FLOW"),
        DOWNLOAD("DOWNLOAD"),
        EMAIL("EMAIL"),
        SHARE("SHARE")
    }

    companion object {

        @Throws(JSONException::class)
        fun fromJson(responseDto: ResponseDto): TruvEventPayload {
            val payloadOut = with(responseDto) {
                val error = payload?.error?.let {
                    TruvError(
                        type = it.type!!,
                        code = TruvError.ErrorCode.valueOf(it.code!!),
                        message = it.message!!
                    )
                }
                val truvEmployer = if (payload?.employer?.name != null) {
                    TruvEmployer(payload.employer.name)
                } else null

                Payload(
                    bridgeToken = payload?.bridgeToken,
                    productType = payload?.productType,
                    viewName = payload?.viewName,
                    employer = truvEmployer,
                    publicToken = payload?.publicToken,
                    taskId = payload?.taskId,
                    providerId = payload?.providerId,
                    error = error,
                    url = payload?.url,
                    action = payload?.action?.let {
                        if (it.isEmpty()) {
                            return@let null
                        }

                        ManualDdsActionType.valueOf(it)
                    },
                    depositType = payload?.depositType,
                    dataSource = payload?.dataSource,
                    docType = payload?.docType
                )
            }

            return TruvEventPayload(
                eventType = EventType.valueOf(responseDto.eventType!!),
                payload = payloadOut
            )
        }
    }
}