package com.truv.webview

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.util.Log
import android.view.View
import android.view.animation.AnimationUtils
import android.webkit.CookieManager
import android.webkit.WebView
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import androidx.annotation.StyleRes
import androidx.core.view.isVisible
import androidx.lifecycle.lifecycleScope
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.truv.R
import com.truv.models.ExternalLoginConfig
import com.truv.models.ResponseDto
import com.truv.network.HttpRequest
import com.truv.webview.models.Cookie
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelAndJoin
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.net.URL
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

data class StorageValue(
    val name: String,
    val value: String,
    val domain: String,
    val source: String,
)

data class ExternalLoginResult(
    val cookies: List<Cookie>,
    val pageUrl: String?,
    val seenUrls: List<String>,
    val storageValues: List<StorageValue>,
)

val TAG = "ExternalWebView"

class ExternalWebViewBottomSheet(
    context: Context,
    @StyleRes styleRes: Int,
    private val onCookie: (ExternalLoginResult) -> Unit,
) : BottomSheetDialog(context, styleRes) {
    var config: ExternalLoginConfig? = null
        set(value) {
            field = value
            if (value?.url != null) {
                findWebView()?.loadUrl(value.url)
                findErrorRetryButton()?.setOnClickListener {
                    findWebView()?.reload()
                }
                findRefresher()?.setOnClickListener {
                    findWebView()?.reload()
                }
                findTitle()?.text = getDomainFromUrl(value.url)
            }
        }

    private val externalWebViewClient by lazy {
        ExternalWebViewClient(context, onLoaded = {
            config?.script?.let { script ->
                lifecycleScope.launch {
                    applyScript(script)
                }
            }
        }, onLoading = { isLoading ->
            Log.d(TAG, "Loading: $isLoading")
            if (isLoading) {
                findErrorLoading()?.isVisible = false
            }
            findWebView()?.isVisible = !isLoading
            findProgressBar()?.isVisible = isLoading
        },
            onLoadingError = { isError ->
                Log.d("ExternalWebView", "Error loading: $isError")
                findWebView()?.isVisible = !isError
                findErrorLoading()?.isVisible = isError
            })
    }

    private var scriptFromUrl: String? = null

    private val bottomSheetCallback = object : BottomSheetBehavior.BottomSheetCallback() {
        private var bottomSheetBehavior: BottomSheetBehavior<FrameLayout>? = null

        fun linkToBehavior(value: BottomSheetBehavior<FrameLayout>) {
            if (bottomSheetBehavior !== value) {
                bottomSheetBehavior?.removeBottomSheetCallback(this)
                bottomSheetBehavior = value
                value.addBottomSheetCallback(this)
            }
        }

        override fun onStateChanged(bottomSheet: View, newState: Int) {
            if (newState == BottomSheetBehavior.STATE_HIDDEN) {
                dismiss()
            }

            if (newState != BottomSheetBehavior.STATE_EXPANDED
                && newState != BottomSheetBehavior.STATE_DRAGGING
                && newState != BottomSheetBehavior.STATE_SETTLING
            ) {
                bottomSheetBehavior?.state = BottomSheetBehavior.STATE_EXPANDED
            }
        }

        override fun onSlide(bottomSheet: View, slideOffset: Float) {
        }
    }

    private fun startProgressAnimation(context: Context) {
        val rotation = AnimationUtils.loadAnimation(context, R.anim.rotate)
        rotation.fillAfter = true
        findProgressBar()?.startAnimation(rotation)
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun initWebView() = with(findWebView()!!) {
        Log.d(TAG, "Init WebView")
        findErrorLoading()?.findViewById<TextView>(R.id.tvErrorMessage)?.text =
            context.getString(R.string.error_message_connection_error)
        findErrorLoading()?.findViewById<TextView>(R.id.tvErrorDescription)?.text =
            context.getString(R.string.error_description)
        findErrorLoading()?.isVisible = false
        settings.javaScriptEnabled = true
        settings.allowContentAccess = true
        settings.domStorageEnabled = true
        setBackgroundColor(Color.WHITE)
        webViewClient = externalWebViewClient
        //https://www.whatismybrowser.com/guides/the-latest-user-agent/chrome
        val userAgent =
            "Mozilla/5.0 (Linux; Android 10; K) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.6099.210 Mobile Safari/537.36"
        settings.userAgentString = userAgent
        addJavascriptInterface(MiddleWareInterface {
//            val responseDto = MiddleWareResponseDto.parse(JSONObject(it))
            lifecycleScope.launch {
                performRequest(it)
            }

        }, "callbackInterface")

        cookiesChecker.start()
    }

    private suspend fun performScriptRequest(scriptUrl: String): String? =
        withContext(Dispatchers.Default) {
            val response = HttpRequest(
                method = "GET",
                url = scriptUrl
            ).response()

            Log.d("ExternalWebView", "performRequest: $response")
            response?.body
        }

    private suspend fun applyScript(script: ResponseDto.Payload.Script) =
        withContext(Dispatchers.Default) {
            try {
                val scriptText = URL(script.url).readText()
                delay(2000)
                withContext(Dispatchers.Main) {
                    evaluateScriptOnWebView(scriptText)
                }
            } catch (e: Exception) {
                Log.e(TAG, "Error applying script", e)
                withContext(Dispatchers.Main) {
                    findWebView()?.isVisible = false
                    findErrorLoading()?.isVisible = true
                }
            }
        }

    private suspend fun performRequest(responseString: String) {
        val response = HttpRequest(
            headers = mapOf(
                "Content-Type" to config?.script?.callbackHeaders?.contentType.orEmpty(),
                "X-Access-Token" to config?.script?.callbackHeaders?.xAccessToken.orEmpty(),
            ),
            body = responseString,
            url = config?.script?.callbackUrl.orEmpty()
        ).response()
        Log.d(TAG, "performRequest: $response")
    }

    private suspend fun loadScriptUrlRequest(): String? {
        if (config?.scriptUrl == null) return null
        return scriptFromUrl ?: run { config?.scriptUrl?.let { performScriptRequest(it) } }
    }

    fun setContentView() {
        Log.d(TAG, "Set content view")
        val contentView = layoutInflater.inflate(R.layout.external_webview_bottom_sheet, null)
        this.setOnShowListener { dialog ->
            val bottomSheet = (dialog as? BottomSheetDialog)?.findViewById<FrameLayout>(
                com.google.android.material.R.id.design_bottom_sheet
            )

            if (bottomSheet != null) {
                val bottomSheetBehavior = BottomSheetBehavior.from(bottomSheet)
                val windowHeight =
                    (contentView.resources.displayMetrics.heightPixels * 0.90).toInt()
                bottomSheetBehavior.peekHeight = windowHeight
                contentView.layoutParams.height = windowHeight
                bottomSheetBehavior.maxHeight = windowHeight

                bottomSheetCallback.linkToBehavior(bottomSheetBehavior)

                bottomSheetBehavior.state = BottomSheetBehavior.STATE_EXPANDED
            }
        }
        super.setContentView(contentView)
        initWebView()
        startProgressAnimation(contentView.context)
    }

    private val cookiesChecker = object {
        var job: Job? = null

        fun start() {
            val oldJob = job
            job = lifecycleScope.launch {
                oldJob?.cancelAndJoin()
                while (isActive) {
                    delay(1000)
                    try {
                        val done = cycle()

                        if (done) {
                            break
                        }
                    } catch (e: Exception) {
                        Log.d(TAG, "error in cycle: $e")
                    }
                }
            }
        }

        fun stop() {
            job?.cancel()
        }

        suspend fun evaluateJs(script: String) = suspendCoroutine<String> { cont ->
            findWebView()?.evaluateJavascript(script) { cont.resume(it) }
        }

        /**
         * @return cycle is completed
         */
        private suspend fun cycle(): Boolean {
            val selectorScript = getSelectorScript() ?: return false

            val result = evaluateJs(selectorScript)
            Log.d(TAG, "WebView evaluate status: $result")
            if (result == "false") {
                return false
            }

            val seenURLs = externalWebViewClient.getSeenPages().toList()
            val allRequestsURLs = externalWebViewClient.getAllRequestURLs().toList()
            val isLoading = externalWebViewClient.isWebViewLoading()
            Log.d(
                TAG,
                "Collecting cookies from seen urls: ${allRequestsURLs.joinToString(", ")}"
            )

            if (isLoading) {
                Log.d(
                    TAG,
                    "WebView is still in loading state"
                )
            }

            val dashboardUrl = if (isLoading) {
                seenURLs.lastOrNull()
            } else {
                findWebView()?.url
            }

            Log.d(TAG, "Got dashboard_url from seen urls: $dashboardUrl")

            val allCookies = getAllCookies(allRequestsURLs)

            Log.d(TAG, "All cookies: $allCookies")

            val webView = findWebView() ?: return false
            val localStorageValues = externalWebViewClient.getStorageValues("localStorage", webView)
            val sessionStorageValues = externalWebViewClient.getStorageValues("sessionStorage", webView)
            Log.d(TAG, "localStorage: $localStorageValues, sessionStorage: $sessionStorageValues")
            val storageValues = localStorageValues.map { StorageValue(it.key, it.value, it.domain, "localStorage") } +
                    sessionStorageValues.map { StorageValue(it.key, it.value, it.domain, "sessionStorage") }

            onCookie(
                ExternalLoginResult(
                    cookies = allCookies,
                    pageUrl = dashboardUrl,
                    seenUrls = seenURLs.toList(),
                    storageValues = storageValues
                )
            )

            dismiss()

            return true
        }

        private fun getAllCookies(urls: List<String>): List<Cookie> {
            return urls.fold(listOf<Cookie>()) { acc, url ->
                val cookies = kotlin.runCatching { CookieManager.getInstance().getCookie(url) }.getOrNull() ?: return@fold acc
                val cookieStrings =
                    cookies.split(";".toRegex()).dropLastWhile { it.isEmpty() }
                        .toTypedArray()
                val domain = ".${URL(url).host}"

                val list = cookieStrings.mapNotNull { cookieString ->
                    val equalIndex = cookieString.indexOf('=')
                    if (equalIndex == -1) {
                        return@mapNotNull null
                    }

                    return@mapNotNull Cookie(
                        name = cookieString.substring(0, equalIndex).trim(),
                        value = cookieString.substring(equalIndex + 1).trim(),
                        domain = domain,
                        path = "/",
                        secure = false,
                        httpOnly = false,
                    )
                }

                return@fold acc.plus(list)
            }
                .distinctBy { Pair(it.domain, it.name) }
        }

        private suspend fun getSelectorScript(): String? {
            scriptFromUrl = loadScriptUrlRequest()

            // For debug: use this string for success by error on ADP site
            // "window.document.evaluate(\"//sdf-alert[@status='error']\", document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue != null";

            return when {
                scriptFromUrl != null -> return scriptFromUrl
                config?.selector != null -> "window.document.evaluate(\"${config?.selector}\", document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue != null"
                else -> {
                    Log.e(TAG, "No selector or scriptUrl provided")
                    null
                }
            }
        }
    }

    private suspend fun evaluateScriptOnWebView(script: String) = withContext(Dispatchers.Main) {
        try {
            if (findWebView()?.isVisible == true) {
                findWebView()?.evaluateJavascript(script, null)
            } else {
                Log.d(TAG, "WebView is not visible")
            }
        } catch (ex: Exception) {
            Log.e(TAG, "Error applying script", ex)
        }

    }

    override fun dismiss() {
        cookiesChecker.stop()
        super.dismiss()
    }

    private fun findErrorLoading(): View? = findViewById(R.id.errorLoadingLayout)
    private fun findErrorRetryButton(): View? = findViewById(R.id.btnTryAgain)
    private fun findWebView(): WebView? = findViewById(R.id.webview)
    private fun findProgressBar(): View? = findViewById(R.id.progress_bar)
    private fun findTitle(): TextView? = findViewById(R.id.webview_title)
    private fun findRefresher(): ImageView? = findViewById(R.id.refresher)
}