package com.truv.webview

import android.content.Context
import android.graphics.Bitmap
import android.os.Build
import android.util.Log
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import android.webkit.WebViewClient
import org.json.JSONArray
import org.json.JSONObject
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine


class ExternalWebViewClient(
    private val context: Context,
    private val onLoading: (Boolean) -> Unit = {},
    private val onLoaded: () -> Unit = {},
    private val onLoadingError: (Boolean) -> Unit = {},
    private val openExternalLinkInAppBrowser: ((String) -> Unit)? = null,
) : WebViewClient() {
    private val seenURLs = mutableSetOf<String>()
    private val allRequests = mutableSetOf<String>()
    private var isLoading = false

    override fun onPageStarted(view: WebView, url: String, favicon: Bitmap?) {
        isLoading = true
        seenURLs.add(url)
        allRequests.add(url)
        super.onPageStarted(view, url, favicon)
    }

    override fun onPageFinished(view: WebView?, url: String?) {
        isLoading = false
        super.onPageFinished(view, url)
    }
    
    override fun shouldInterceptRequest(
        view: WebView?,
        request: WebResourceRequest
    ): WebResourceResponse? {
        if (request.requestHeaders.containsKey("X-Requested-With")) {
            request.requestHeaders.remove("X-Requested-With")
        }

        allRequests.add(request.url.toString())

        return super.shouldInterceptRequest(view, request)
    }

    override fun shouldOverrideUrlLoading(
        view: WebView?,
        request: WebResourceRequest?
    ): Boolean {
        Log.d("ExternalWebViewClient", "URL: ${request?.url}")
        onLoading(true)
        request?.let {
            openExternalLinkInAppBrowser?.invoke(it.url.toString()) ?: run {
                view?.loadUrl(it.url.toString())
            }
        }

        seenURLs.add(request?.url.toString())
        allRequests.add(request?.url.toString())

        return true
    }

    override fun onPageCommitVisible(view: WebView?, url: String?) {
        super.onPageCommitVisible(view, url)
        onLoading(false)
        onLoaded()
    }

    override fun doUpdateVisitedHistory(view: WebView?, url: String?, isReload: Boolean) {
        Log.d("ExternalWebViewClient", "doUpdateVisitedHistory: $url")
        if (url != null) {
            seenURLs.add(url)
            allRequests.add(url)
        }
        super.doUpdateVisitedHistory(view, url, isReload)
    }

    override fun onReceivedError(
        view: WebView,
        request: WebResourceRequest,
        error: WebResourceError
    ) {
        super.onReceivedError(view, request, error)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            if (error.errorCode == WebViewClient.ERROR_CONNECT ||
                error.errorCode == WebViewClient.ERROR_TIMEOUT ||
                error.errorCode == WebViewClient.ERROR_HOST_LOOKUP
            ) {
                onLoadingError(true)
            } else{
                onLoadingError(false)
            }
        }
    }

    data class StorageItem(
        val key: String,
        val value: String,
        val domain: String
    )

    suspend fun getStorageValues(type: String, view: WebView): List<StorageItem> = suspendCoroutine { continuation ->
        view.evaluateJavascript(
            """
            (function() {
              const items = [];
              for (let i = 0; i < $type.length; i++) {
                const key = $type.key(i);
                items.push({
                  key: key,
                  value: $type.getItem(key),
                  domain: window.location.hostname
                });
              }
              return items;
            })();
            """
        ) { value ->
            Log.d(TAG, "Storage: $type, $value")
            val values = kotlin.runCatching {
                if (value == "null") {
                    emptyList<StorageItem>()
                } else {
                    try {
                        val jsonArray = JSONArray(value)
                        Log.d(TAG, "jsonArray for $type: $jsonArray")
                        val items = mutableListOf<StorageItem>()
                        for (i in 0 until jsonArray.length()) {
                            val item = jsonArray.getJSONObject(i)
                            items.add(
                                StorageItem(
                                    key = item.getString("key"),
                                    value = item.getString("value"),
                                    domain = item.getString("domain")
                                )
                            )
                        }
                        items
                    } catch (e: Exception) {
                        Log.d(TAG, "error: $e")
                        throw e
                    }
                }
            }.getOrDefault(emptyList<StorageItem>())
            continuation.resume(values)
        }
    }

    fun getSeenPages(): Set<String> {
        return seenURLs.toSet()
    }

    fun getAllRequestURLs(): Set<String> {
        return allRequests.toSet()
    }

    fun isWebViewLoading(): Boolean {
        return isLoading
    }
}