package com.truv.webview

import android.util.Log
import android.webkit.JavascriptInterface
import com.truv.models.ExternalLoginConfig
import com.truv.models.ResponseDto
import com.truv.models.TruvEventPayload
import com.truv.models.TruvSuccessPayload
import org.json.JSONException
import org.json.JSONObject

class WebAppInterface(
    private val eventListeners: Set<TruvEventsListener>,
    private val onShowExternalWebView: ((ExternalLoginConfig) -> Unit)
) {

    @JavascriptInterface
    fun onSuccess(payload: String) {
        Log.d(TAG, "onSuccess invoked $payload")

        try {
            val successPayload = TruvSuccessPayload.fromJson(payload)
            eventListeners.forEach { it.onSuccess(successPayload) }
        } catch (e: JSONException) {
            Log.e(TAG, "Json exception at onSuccess invoked $payload", e)
            eventListeners.forEach { it.onError() }
        }
    }

    @JavascriptInterface
    fun onEvent(event: String) {
        Log.d(TAG, "onEvent invoked $event")

        try {
            val json = JSONObject(event)
            val responseDto = ResponseDto.parse(json)

            when (responseDto.eventType) {
                "START_HEADERS_COLLECTION" -> {
                    // Not implemented yet
                }
                "START_EXTERNAL_LOGIN" -> {
                    val payload = responseDto.payload

                    val externalLoginConfig =
                        ExternalLoginConfig(
                            url = payload?.url,
                            selector = payload?.isLoggedIn?.selector,
                            scriptUrl = payload?.isLoggedIn?.scriptUrl,
                            script = payload?.script
                        )

                    onShowExternalWebView.invoke(externalLoginConfig)
                }
                "OAUTH_OPENED" -> {
                    // Not implemented yet
                    sendEvent(responseDto)
                }
                else -> {
                    sendEvent(responseDto)
                }
            }
        } catch (e: JSONException) {
            Log.e(TAG, "Json exception at onEvent invoked $event", e)
        }
    }

    fun sendEvent(responseDto: ResponseDto) {
        val eventPayload = TruvEventPayload.fromJson(responseDto)
        eventListeners.forEach { it.onEvent(eventPayload) }
    }

    @JavascriptInterface
    fun onClose() {
        Log.d(TAG, "onClose invoked")
        eventListeners.forEach { it.onClose() }
    }

    @JavascriptInterface
    fun onLoad() {
        Log.d(TAG, "onLoad invoked")
        eventListeners.forEach { it.onLoad() }
    }

    @JavascriptInterface
    fun onError() {
        Log.d(TAG, "onError invoked")
        eventListeners.forEach { it.onError() }
    }

    companion object {
        const val TAG = "TruvBridgeView"
    }

}