/**
 * Copyright (C) 2009 Kent Tong <freemant2000@yahoo.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * Free Software Foundation version 3.
 *
 * program is distributed in the hope that it will be useful,
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ttdev.wicketpagetest;

import org.apache.wicket.protocol.http.WebApplication;
import org.openqa.selenium.WebDriver;
import org.testng.annotations.AfterSuite;
import org.testng.annotations.BeforeSuite;

/**
 * To unit test your Wicket pages, before running the tests, you should call
 * {@link #beforePageTests()}. Then it will launch Jetty, your webapp and a
 * Selenium client before running any tests in that suite. To shut them down
 * afterwards, run {@link #afterPageTests()}.
 * 
 * If you're using TestNG, use the {@link WebPageTestContext} subclass.
 * 
 * In your test cases, you can obtain the global Selenium client by
 * {@link #getSelenium()}. However, usually you should obtain a
 * {@link WicketSelenium} instead which contains special support for HTML pages
 * generated by Wicket.
 * <p>
 * If you'd like to customize the behaviors of {@link WebPageTestBasicContext},
 * you can create a {@link Configuration} object and then pass it to
 * {@link #beforePageTests(Configuration)}. To do that, just create a wrapper
 * class with a {@link BeforeSuite} method and a {@link AfterSuite} method.
 * These methods should call {@link #beforePageTests(Configuration)} and
 * {@link #afterPageTests()} respectively.
 * 
 * @see Configuration
 * 
 * @author Kent Tong
 * 
 */
public class WebPageTestBasicContext {
	private static WebPageTestBasicContext instance;

	private WebDriver selenium;
	private WicketSeleniumDriver wicketSelenium;
	private WicketAppJettyLauncher jettyLauncher;
	private Configuration cfg;

	public static void beforePageTests() throws Exception {
		beforePageTests(new Configuration());
	}

	public static void beforePageTests(Configuration cfg) throws Exception {
		instance = new WebPageTestBasicContext();
		instance.start(cfg);
	}

	public static Configuration getConfiguration() {
		return instance.cfg;
	}

	public static WicketSeleniumDriver getWicketSelenium() {
		return instance.wicketSelenium;
	}

	private void start(Configuration cfg) {
		this.cfg = cfg;
		startWebAppInJetty();
		startSelenium();
	}

	private void startWebAppInJetty() {
		jettyLauncher = new WicketAppJettyLauncher();
		jettyLauncher.startAppInJetty(cfg);
	}

	private void startSelenium() {
		selenium = cfg.getSelenium();
		wicketSelenium = new WicketSeleniumDriver(cfg, selenium);
	}

	public static void afterPageTests() throws Exception {
		instance.stop();
	}

	private void stop() {
		stopSelenium();
		stopJetty();
	}

	private void stopJetty() {
		if (jettyLauncher != null) {
			jettyLauncher.stopJetty();
		}
	}

	private void stopSelenium() {
		if (selenium != null) {
			selenium.quit();
		}
	}

	public WebApplication getApplication() {
		return jettyLauncher.getApplication();
	}

	public static WebDriver getSelenium() {
		return instance.selenium;
	}

	public static WebApplication getWebApplication() {
		return instance.getApplication();
	}

}
