//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Contains aggregated information about a {@link Message}'s delivery statuses across all {@link Participant participants}
 * of a {@link Conversation}.
 * <p>
 * At any moment during delivering message to a {@link Participant} the message can have zero or more of following
 * delivery statuses:
 *
 * <ul><li>
 * Message considered as <b>sent</b> to a participant, if the nearest upstream carrier accepted the message.
 * </li><li>
 * Message considered as <b>delivered</b> to a participant, if Twilio has received confirmation of message
 * delivery from the upstream carrier, and, where available, the destination handset.
 * </li><li>
 * Message considered as <b>undelivered</b> to a participant, if Twilio has received a delivery receipt
 * indicating that the message was not delivered. This can happen for many reasons including carrier content
 * filtering and the availability of the destination handset.
 * </li><li>
 * Message considered as <b>read</b> by a participant, if the message has been delivered and opened by the
 * recipient in the conversation. The recipient must have enabled read receipts.
 * </li><li>
 * Message considered as <b>failed</b> to be delivered to a participant if the message could not be sent.
 * This can happen for various reasons including queue overflows, account suspensions and media
 * errors (in the case of MMS for instance).
 *</li></ul>
 *
 * {@link AggregatedDeliveryReceipt} class contains aggregated value {@link DeliveryAmount} for each delivery status.
 * The {@link DeliveryAmount} displays amount of participants which have the status for the message.
 */
public class AggregatedDeliveryReceipt {

    /** Amount of participants which have the status for the message. */
    public enum DeliveryAmount {
        /**
         * The amount for the delivery statuses is 0.
         * So no delivery event has been received with that status for the specified message.
         */
        NONE(0),
        /**
         * Amount of the delivery statuses is at least 1. So at least one delivery event has been received.
         */
        SOME(1),
        /**
         * Amount of the delivery statuses equals the maximum number of delivery events expected for that message.
         * The maximum number of delivery events expected for that message is returned by {@link #getTotal}.
         */
        ALL(2);

        private final int mValue;
        private DeliveryAmount(int value) { this.mValue = value; }

        /** @return The associated integer value. */
        public int getValue() { return mValue; }
        /** @return The {@link DeliveryAmount} by associated integer value. */
        public static DeliveryAmount fromInt(int value)
        {
            for (DeliveryAmount amount : DeliveryAmount.values()) {
                if (amount.getValue() == value)
                    return amount;
            }
            throw new IllegalStateException("Invalid value " + value + " for AggregatedDeliveryReceipt.DeliveryAmount");
        }
    }

    private final int mTotal;
    private final DeliveryAmount mRead;
    private final DeliveryAmount mUndelivered;
    private final DeliveryAmount mDelivered;
    private final DeliveryAmount mFailed;
    private final DeliveryAmount mSent;

    /** @internal */
    AggregatedDeliveryReceipt(int total,
                              DeliveryAmount read,
                              DeliveryAmount undelivered,
                              DeliveryAmount delivered,
                              DeliveryAmount failed,
                              DeliveryAmount sent) {
        mTotal = total;
        mRead = read;
        mUndelivered = undelivered;
        mDelivered = delivered;
        mFailed = failed;
        mSent = sent;
    }

    /**
     * Returns maximum number of delivery events expected for the message.
     */
    public int getTotal() {
        return mTotal;
    }

    /**
     * Message considered as <b>sent</b> to a participant, if the nearest upstream carrier accepted the message.
     *
     * @return {@link DeliveryAmount} of participants that have <b>sent</b> delivery status for the message.
     */
    public DeliveryAmount getSent() {
        return mSent;
    }

    /**
     * Message considered as <b>delivered</b> to a participant, if Twilio has received confirmation of message
     * delivery from the upstream carrier, and, where available, the destination handset.
     *
     * @return {@link DeliveryAmount} of participants that have <b>delivered</b> delivery status for the message.
     */
    public DeliveryAmount getDelivered() {
        return mDelivered;
    }

    /**
     * Message considered as <b>undelivered</b> to a participant, if Twilio has received a delivery receipt
     * indicating that the message was not delivered. This can happen for many reasons including carrier content
     * filtering and the availability of the destination handset.
     *
     * @return {@link DeliveryAmount} of participants that have <b>undelivered</b> delivery status for the message.
     */
    public DeliveryAmount getUndelivered() {
        return mUndelivered;
    }

    /**
     * Message considered as <b>read</b> by a participant, if the message has been delivered and opened by the
     * recipient in the conversation. The recipient must have enabled read receipts.
     *
     * @return {@link DeliveryAmount} of participants that have <b>read</b> delivery status for the message.
     */
    public DeliveryAmount getRead() {
        return mRead;
    }

    /**
     * Message considered as <b>failed</b> to be delivered to a participant if the message could not be sent.
     * This can happen for various reasons including queue overflows, account suspensions and media
     * errors (in the case of MMS for instance). Twilio does not charge you for failed messages.
     *
     * @return {@link DeliveryAmount} of participants that have <b>failed</b> delivery status for the message.
     */
    public DeliveryAmount getFailed() {
        return mFailed;
    }

    /** @hide */
    @NonNull
    @Override
    public String toString() {
        return "AggregatedDeliveryReceipt(" +
                "\ntotal: " + mTotal +
                "\nread: " + mRead +
                "\nundelivered: " + mUndelivered +
                "\ndelivered: " + mDelivered +
                "\nfailed: " + mFailed +
                "\nsent: " + mSent +
                "\n)";
    }

    /** @hide */
    @Override
    public int hashCode() {
        int result = mTotal;
        result = 31 * result + mRead.hashCode();
        result = 31 * result + mUndelivered.hashCode();
        result = 31 * result + mDelivered.hashCode();
        result = 31 * result + mFailed.hashCode();
        result = 31 * result + mSent.hashCode();
        return result;
    }

    /** @hide */
    @Override
    public boolean equals(@Nullable Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof AggregatedDeliveryReceipt)) {
            return false;
        }

        AggregatedDeliveryReceipt other = (AggregatedDeliveryReceipt) obj;
        return mTotal == other.mTotal
                && mRead.equals(other.mRead)
                && mUndelivered.equals(other.mUndelivered)
                && mDelivered.equals(other.mDelivered)
                && mFailed.equals(other.mFailed)
                && mSent.equals(other.mSent);
    }
}
