package com.twilio.conversations;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.twilio.util.TwilioLogger;
import kotlin.jvm.JvmClassMappingKt;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

/**
 * Attributes representation, can be set to Conversation, User, Participant or Message.
 */
public class Attributes {

    private static final TwilioLogger logger = TwilioLogger.getLogger(
        JvmClassMappingKt.getKotlinClass(Attributes.class));

    /** Default attributes value. */
    public static final Attributes DEFAULT = new Attributes(new JSONObject());

    @NonNull
    private final Type mType;
    @NonNull
    private final String mJson;

    /**
     * Create NULL attributes.
     */
    public Attributes() {
        mType = Type.NULL;
        mJson = JSONObject.NULL.toString();
    }

    /**
     * Initialize attributes with a {@link JSONObject}.
     */
    public Attributes(@NonNull JSONObject object) {
        if (object == null) {
            throw new IllegalArgumentException("object must not be null");
        }
        mType = Type.OBJECT;
        mJson = object.toString();
    }

    /**
     * Initialize attributes with a {@link JSONArray}.
     */
    public Attributes(@NonNull JSONArray array) {
        if (array == null) {
            throw new IllegalArgumentException("array must not be null");
        }
        mType = Type.ARRAY;
        mJson = array.toString();
    }

    /**
     * Initialize attributes with a {@link String}.
     */
    public Attributes(@NonNull String string) {
        if (string == null) {
            throw new IllegalArgumentException("string must not be null");
        }
        mType = Type.STRING;
        mJson = JSONObject.quote(string);
    }

    /**
     * Initialize attributes with a {@link Number}.
     */
    public Attributes(@NonNull Number number) {
        if (number == null) {
            throw new IllegalArgumentException("number must not be null");
        }
        mType = Type.NUMBER;
        try {
            mJson = JSONObject.numberToString(number);
        } catch (JSONException e) {
            throw new IllegalArgumentException("Cannot serialize number: " + number, e);
        }
    }

    /**
     * Initialize attributes with a boolean value.
     */
    public Attributes(boolean value) {
        mType = Type.BOOLEAN;
        mJson = String.valueOf(value);
    }

    /**
     * Returns {@link Type} of attributes
     */
    @NonNull
    public Type getType() {
        return mType;
    }

    /**
     * Returns {@link JSONObject} stored in attributes if {@link #getType} of attributes is {@link Type#OBJECT},
     * otherwise returns <tt>null</tt>.
     */
    @Nullable
    public JSONObject getJSONObject() {
        if (mType != Type.OBJECT) {
            return null;
        }

        try {
            return new JSONObject(mJson);
        } catch (Exception e) {
            logger.e("Error creating JSONObject: " + mJson, e);
            return null;
        }
    }

    /**
     * Returns {@link JSONArray} stored in attributes if {@link #getType} of attributes is {@link Type#ARRAY},
     * otherwise returns <tt>null</tt>.
     */
    @Nullable
    public JSONArray getJSONArray() {
        if (mType != Type.ARRAY) {
            return null;
        }

        try {
            return new JSONArray(mJson);
        } catch (Exception e) {
            logger.e("Error creating JSONArray: " + mJson, e);
            return null;
        }
    }

    /**
     * Returns {@link String} stored in attributes if {@link #getType} of attributes is {@link Type#STRING},
     * otherwise returns <tt>null</tt>.
     */
    @Nullable
    public String getString() {
        if (mType != Type.STRING) {
            return null;
        }

        try {
            return (String) new JSONTokener(mJson).nextValue();
        } catch (Exception e) {
            logger.e("Error creating String: " + mJson, e);
            return null;
        }
    }

    /**
     * Returns {@link Number} stored in attributes if {@link #getType} of attributes is {@link Type#NUMBER},
     * otherwise returns <tt>null</tt>.
     */
    @Nullable
    public Number getNumber() {
        if (mType != Type.NUMBER) {
            return null;
        }

        try {
            return (Number) new JSONTokener(mJson).nextValue();
        } catch (Exception e) {
            logger.e("Error creating Number: " + mJson, e);
            return null;
        }
    }

    /**
     * Returns {@link Boolean} stored in attributes if {@link #getType} of attributes is {@link Type#BOOLEAN},
     * otherwise returns <tt>null</tt>.
     */
    @Nullable
    public Boolean getBoolean() {
        if (mType != Type.BOOLEAN) {
            return null;
        }

        try {
            return (Boolean) new JSONTokener(mJson).nextValue();
        } catch (Exception e) {
            logger.e("Error creating Boolean: " + mJson, e);
            return null;
        }
    }

    /** @hide */
    @NonNull
    @Override
    public String toString() {
        return mJson;
    }

    /** @hide */
    @Override
    public int hashCode() {
        return 31 * mType.hashCode() + mJson.hashCode();
    }

    /** @hide */
    @Override
    public boolean equals(@Nullable Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof Attributes)) {
            return false;
        }

        Attributes other = (Attributes) obj;
        return mType == other.mType && mJson.equals(other.mJson);
    }

    /**
     * Parses string received from JNI into {@link Attributes} object.
     *
     * @throws JSONException if cannot parse input string.
     */
    @NonNull
    static Attributes parse(String attributes) throws JSONException {
        Object value = new JSONTokener(attributes).nextValue();

        if (value instanceof JSONObject) {
            return new Attributes((JSONObject) value);
        }

        if (value instanceof JSONArray) {
            return new Attributes((JSONArray) value);
        }

        if (JSONObject.NULL.equals(value)) {
            return new Attributes();
        }

        if (value instanceof Boolean) {
            return new Attributes((Boolean) value);
        }

        if (value instanceof Number) {
            return new Attributes((Number) value);
        }

        if (value instanceof String) {
            return new Attributes((String) value);
        }

        throw new JSONException("Unknown JSON value type: " + value);
    }

    /**
     * Type of data stored in {@link Attributes}.
     */
    enum Type {
        /**
         * {@link Attributes} contains {@link JSONObject}. So {@link #getJSONObject} returns not-null value.
         */
        OBJECT,
        /**
         * {@link Attributes} contains {@link JSONArray}. So {@link #getJSONArray} returns not-null value.
         */
        ARRAY,
        /**
         * {@link Attributes} contains {@link String}. So {@link #getString} returns not-null value.
         */
        STRING,
        /**
         * {@link Attributes} contains {@link Number}. So {@link #getNumber} returns not-null value.
         */
        NUMBER,
        /**
         * {@link Attributes} contains {@link Boolean} value. So {@link #getBoolean} returns not-null value.
         */
        BOOLEAN,
        /**
         * {@link Attributes} contains no data. So either method {@link #getJSONObject}, {@link #getJSONArray},
         * {@link #getString}, {@link #getNumber} and {@link #getBoolean} returns <tt>null</tt>.
         */
        NULL
    }
}
