//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.twilio.conversations.content.ContentTemplate;
import com.twilio.conversations.content.ContentTemplateVariable;

import org.jetbrains.annotations.NotNull;

import java.io.InputStream;
import java.util.Date;
import java.util.List;

/**
 * Container for conversation object.
 * <p>
 * To obtain a Conversation call {@link ConversationsClient#createConversation}, {@link ConversationsClient#getConversation} or get it
 * from collections returned by {@link ConversationsClient#getMyConversations}.
 * <p>
 * Conversation is a relatively heavy-weight object. It receives state updates from the back-end
 * in real time.
 */
public interface Conversation
{
    /**
     * User's notification level on a conversation.
     */
    enum NotificationLevel {
        /** User will receive notifications for the conversation if joined, nothing if unjoined. */
        DEFAULT(0),
        /** User will not receive notifications for the conversation. */
        MUTED(1);

        private final int value;
        private NotificationLevel(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link NotificationLevel} by associated integer value. */
        public static NotificationLevel fromInt(int value)
        {
            for (NotificationLevel t : NotificationLevel.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for NotificationLevel");
        }
    }

    /**
     * Represents the various statuses of the user with respect to the conversation.
     */
    enum ConversationStatus {
        /** User has joined this conversation. */
        JOINED(1),
        /** User has NOT been joined this conversation. */
        NOT_PARTICIPATING(2);

        private final int value;
        private ConversationStatus(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link ConversationStatus} by associated integer value. */
        public static ConversationStatus fromInt(int value)
        {
            for (ConversationStatus t : ConversationStatus.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for ConversationStatus");
        }
    }

    /**
     * Represents the various states of the conversation.
     *
     * States of the conversation could change in the following sequence: ACTIVE <-> INACTIVE -> CLOSED
     * with timers (automatically flip to "inactive" after some period of inactivity), triggers (new message,
     * new participant -- flip to "active" back on activity), or API call.
     */
    enum ConversationState {
        /**
         * Conversation state is unknown.
         */
        UNDEFINED(0),
        /**
         * Conversation state is active, i.e. some events have happened in the conversation
         * recently (new message received, new participant added etc).
         */
        ACTIVE(1),
        /**
         * Conversation state is inactive, i.e. no new events have happened in the conversation
         * recently (new message received, new participant added etc). So the conversation has been moved
         * to inactive state by timeout.
         */
        INACTIVE(2),
        /**
         * When conversation moves to CLOSED state SDKs receives update that the user left given conversation.
         * So SDKs will not receive any updates on it in the future. Any operations with closed conversation
         * are not permitted.
         */
        CLOSED(3);

        private final int value;
        private ConversationState(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link ConversationState} by associated integer value. */
        public static ConversationState fromInt(int value)
        {
            for (ConversationState t : ConversationState.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for ConversationState");
        }
    }

    /**
     * Indicates synchronization status for conversation.
     * <p>
     * Attempts to query information that has not been synchronized yet will result in return of
     * null value for that property.
     */
    enum SynchronizationStatus {
        /** Local copy, does not exist in cloud. */
        NONE(0),
        /** Conversation SID, not synchronized with cloud. */
        IDENTIFIER(1),
        /** Conversation metadata: friendly name, conversation SID, attributes, unique name. */
        METADATA(2),
        /** Conversation collections: participants, messages can be fetched. */
        ALL(3),
        /** Conversation synchronization failed. */
        FAILED(4);

        private final int value;
        private SynchronizationStatus(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link SynchronizationStatus} by associated integer value. */
        public static SynchronizationStatus fromInt(int value)
        {
            for (SynchronizationStatus t : SynchronizationStatus.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for SynchronizationStatus");
        }
        /**
         * Check if conversation synchronization is at least at the given state.
         * If either requested status or current conversation status is #FAILED this function will return false.
         * @param  s Requested synchronization level.
         * @return   true if current synch status is at least at the requested value, or more, and is not #FAILED.
         */
        public boolean isAtLeast(SynchronizationStatus s) {
            if (s == FAILED || this == FAILED) return false;
            return getValue() >= s.getValue();
        }
    }

    /**
     * Indicates reason for conversation update.
     */
    enum UpdateReason
    {
        /** Conversation status changed. */
        STATUS(1),
        /** Conversation last read message changed. */
        LAST_READ_MESSAGE_INDEX(2),
        /** Conversation unique name changed. */
        UNIQUE_NAME(3),
        /** Conversation friendly name changed. */
        FRIENDLY_NAME(4),
        /** Conversation attributes changed. */
        ATTRIBUTES(5),
        /** Last message in conversation changed. This update does not trigger when message itself
         * changes, there's Message.UpdateReason event for that. However, if a new message is added
         * or last conversation message is deleted this event will be triggered.
         * @see #getLastMessageDate and #getLastMessageIndex to get updated last message.
         */
        LAST_MESSAGE(6),
        /** Notification level changed.
         * @see NotificationLevel
         * @see #setNotificationLevel
         */
        NOTIFICATION_LEVEL(7),
        /**
         * Conversation state changed.
         * @see ConversationState
         */
        STATE(8);

        private final int value;
        private UpdateReason(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link UpdateReason} by associated integer value. */
        public static UpdateReason fromInt(int value)
        {
            for (UpdateReason t : UpdateReason.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for UpdateReason");
        }
    }

    /**@name Getters */
    /**@{*/

    /**
     * Get unique identifier for this conversation.
     * <p>
     * This identifier can be used to get this Conversation again using
     * {@link ConversationsClient#getConversation}.
     * <p>
     * The conversation SID is persistent and globally unique.
     *
     * @return Conversation SID.
     */
    String getSid();

    /**
     * Get friendly name of the conversation.
     * <p>
     * Friendly name is a free-form text string, it is not unique and could be used
     * for user-friendly conversation name display in the UI.
     *
     * @return Conversation friendly name.
     * @see #setFriendlyName
     */
    String getFriendlyName();

    /**
     * Get unique name of the conversation.
     * <p>
     * Unique name is similar to SID but can be specified by the user.
     *
     * @return Conversation unique name or null if it was not set.
     * @see #setUniqueName
     */
    String getUniqueName();

    /**
     * @return Limits for attachments per each message in the conversation.
     * @see ConversationLimits
     */
    ConversationLimits getLimits();

    /**
     * Get custom attributes associated with the Conversation.
     * <p>
     * Attributes are stored as a JSON format object, of arbitrary internal structure.
     * Conversation attributes are limited in size to 32Kb.
     *
     * @return {@link Attributes} object with attributes.
     * @see #setAttributes
     */
    @NonNull
    Attributes getAttributes();

    /**
     * The current user's notification level on this conversation. This property reflects whether the
     * user will receive push notifications for activity on this conversation.
     * @return Current notification level.
     * @see #setNotificationLevel
     * @see NotificationLevel
     */
    NotificationLevel getNotificationLevel();

    /**
     * Get the current synchronization status for conversation.
     *
     * @return Current synchronization status for the conversation
     */
    SynchronizationStatus getSynchronizationStatus();

    /**
     * Get the current user's participation status on this conversation.
     *
     * @return Conversation participation status.
     */
    ConversationStatus getStatus();

    /**
     * Get activity state of this conversation.
     *
     * @return The current {@link ConversationState} for this conversation.
     */
    ConversationState getState();

    /**
     * Get update date of the {@link ConversationState}
     *
     * @return Date when {@link ConversationState} was last updated.
     */
    @Nullable
    Date getStateDateUpdatedAsDate();

    /**
     * Get creator of the conversation.
     *
     * @return Identity of the conversation's creator.
     */
    String getCreatedBy();

    /**
     * Get creation date of the conversation as
     * an <a href="http://www.iso.org/iso/home/standards/iso8601.htm">ISO 8601</a> string.
     *
     * @return Date when conversation was created as a string in ISO 8601 format.
     */
    String getDateCreated();

    /**
     * Get creation date of the conversation.
     *
     * @return Date when conversation was created or null if date string could not be parsed.
     */
    Date getDateCreatedAsDate();

    /**
     * Get update date of the conversation as
     * an <a href="http://www.iso.org/iso/home/standards/iso8601.htm">ISO 8601</a> string.
     * <p>
     * Update date changes when conversation attributes, friendly name or unique name are
     * modified. It will not change in response to messages posted or participants added or removed.
     *
     * @return Date when conversation was last updated as a string in ISO 8601 format.
     */
    String getDateUpdated();

    /**
     * Get update date of the conversation.
     * <p>
     * Update date changes when conversation attributes, friendly name or unique name are
     * modified. It will not change in response to messages posted or participants added or removed.
     *
     * @return Date when conversation was last updated or null if date string could not be parsed.
     */
    Date getDateUpdatedAsDate();

    /**@}*/
    /**@name Setters */
    /**@{*/

    /**
     * Update the friendly name for this conversation.
     *
     * @param friendlyName New friendly name.
     * @param listener Listener that will receive callback with the status of update operation.
     * @see #getFriendlyName
     */
    void setFriendlyName(String friendlyName, StatusListener listener);

    /**
     * Update the unique name for this conversation.
     * <p>
     * Unique name is unique within Service Instance. You will receive an error if you try to set
     * a name that is not unique.
     *
     * @param uniqueName    New unique name for this conversation.
     * @param listener      Listener that will receive callback with the result.
     * @see #getUniqueName
     */
    void setUniqueName(String uniqueName, StatusListener listener);

    /**
     * Set attributes associated with this conversation.
     * <p>
     * Attributes are stored as a JSON format object, of arbitrary internal structure.
     * Conversation attributes are limited in size to 32Kb.
     *
     * @param attributes The new developer-provided {@link Attributes} for this conversation.
     * @param listener Listener that will receive callback with the result.
     * @see #getAttributes
     */
    void setAttributes(@NotNull Attributes attributes, StatusListener listener);

    /**
     * Set the user's notification level for the conversation.  This property determines whether the
     * user will receive push notifications for activity on this conversation.
     *
     * @param notificationLevel The new notification level for the current user on this conversation.
     * @param listener Listener that will receive callback with the status of update operation.
     * @see #getNotificationLevel
     */
    void setNotificationLevel(NotificationLevel notificationLevel, StatusListener listener);

    /**@}*/
    /**@name Listeners */
    /**@{*/

    /**
     * Add a new {@link ConversationListener} for this Conversation.
     * <p>
     * Conversation listeners receive real time updates about changes in the conversation state,
     * messages and participants.
     * <p>
     * This listener will be called on the originating thread if it has a Looper, otherwise
     * on the main UI thread.
     *
     * @param listener A conversation listener.
     * @see #removeListener
     * @see #removeAllListeners
     */
    void addListener(ConversationListener listener);

    /**
     * Remove ConversationListener for this Conversation.
     *
     * @param listener Listener to remove.
     * @see #addListener
     * @see #removeAllListeners
     */
    void removeListener(ConversationListener listener);

    /**
     * Remove all ConversationListeners for this Conversation.
     *
     * @see #addListener
     * @see #removeListener
     */
    void removeAllListeners();

    /**@}*/
    /**@name Actions */
    /**@{*/

    /**
     * Join the current user to this conversation.
     * <p>
     * Joining the conversation is a prerequisite for sending and receiving messages in the conversation.
     * You can join the conversation or you could be added to it by another conversation participant.
     *
     * @param listener Listener that will receive callback with the result.
     * @see #leave
     */
    void join(StatusListener listener);

    /**
     * Leave this conversation.
     *
     * @param listener Listener that will receive callback with the result.
     * @see #join
     */
    void leave(StatusListener listener);

    /**
     * Destroy the current conversation.
     * <p>
     * Note: this will delete the Conversation and all associated metadata from the service instance.
     * <b>Participants in the conversation and all conversation messages, including posted media will be lost.</b>
     * <p>
     * There is no undo for this operation!
     *
     * @param listener Listener that will receive callback with the result.
     */
    void destroy(StatusListener listener);

    /**
     * Indicate that Participant is typing in this conversation.
     * <p>
     * You should call this method to indicate that a local user is entering a message into
     * current conversation. The typing state is forwarded to users subscribed to this
     * conversation through {@link ConversationListener#onTypingStarted} and
     * {@link ConversationListener#onTypingEnded} callbacks.
     * <p>
     * After approximately 5 seconds after the last {@link #typing} call the SDK will emit
     * {@link ConversationListener#onTypingEnded} signal.
     * <p>
     * One common way to implement this indicator is to call {@link #typing} repeatedly
     * in response to key input events.
     */
    void typing();

    /**@}*/
    /**@name Counts */
    /**@{*/

    /**
     * Get total number of messages in the conversation.
     * <p>
     * This method is semi-realtime. This means that this data will be eventually correct,
     * but will also possibly be incorrect for a few seconds. The Conversations system does not
     * provide real time events for counter values changes.
     * <p>
     * So this is quite useful for any UI badges, but is not recommended
     * to build any core application logic based on these counters being accurate in real time.
     * <p>
     * This function performs an async call to service to obtain up-to-date message count.
     * The retrieved value is then cached for 5 seconds so there is no reason to call this
     * function more often than once in 5 seconds.
     *
     * @param listener Listener to receive results of the query.
     * @see #getUnreadMessagesCount
     */
    void getMessagesCount(CallbackListener<Long> listener);

    /**
     * Get number of unread messages in the conversation.
     * <p>
     * Note: if the last read index has not been yet set for current user as the participant of this conversation
     * then unread messages count is considered uninitialized. In this case null is returned.
     * See {@link #setLastReadMessageIndex}.
     * <p>
     * This method is semi-realtime. This means that this data will be eventually correct,
     * but will also possibly be incorrect for a few seconds. The Conversations system does not
     * provide real time events for counter values changes.
     * <p>
     * So this is quite useful for any “unread messages count” badges, but is not recommended
     * to build any core application logic based on these counters being accurate in real time.
     * <p>
     * This function performs an async call to service to obtain up-to-date message count.
     * The retrieved value is then cached for 5 seconds so there is no reason to call this
     * function more often than once in 5 seconds.
     * <p>
     * Use this method to obtain number of unread messages together with
     * {@link #setLastReadMessageIndex} instead of relying on
     * Message indices which may have gaps. See {@link Message#getMessageIndex} for details.
     *
     * @param listener Listener to receive results of the query.
     * @see #getMessagesCount
     */
    void getUnreadMessagesCount(CallbackListener<Long> listener);

    /**
     * Get total number of participants in the conversation roster.
     * <p>
     * This method is semi-realtime. This means that this data will be eventually correct,
     * but will also possibly be incorrect for a few seconds. The Conversations system does not
     * provide real time events for counter values changes.
     * <p>
     * So this is quite useful for any UI badges, but is not recommended
     * to build any core application logic based on these counters being accurate in real time.
     * <p>
     * This function performs an async call to service to obtain up-to-date participant count.
     * The retrieved value is then cached for 5 seconds so there is no reason to call this
     * function more often than once in 5 seconds.
     *
     * @param listener Listener to receive results of the query.
     */
    void getParticipantsCount(CallbackListener<Long> listener);

    /**@}*/
    /**@name Messages */
    /**@{*/

    /**
     * Prepares a new message to send into the conversation.
     * <p>
     * <b>Example (Kotlin):</b>
     * <pre><code>
     val unsentMessage = conversation.prepareMessage {
         body = "Take a look at this!"

         addMedia("image/png", "cat.png", FileInputStream(filePath)) {
            onStarted {
                println("Uploading cat.png started")
            }
            onCompleted { mediaSid ->
                println("cat.png has been uploaded, mediaSid: $mediaSid")
            }
         }
     }

     val message = unsentMessage.send()

     // or with single extension function
     val message2 = conversation.sendMessage {
        body = "Hello!"
     }

     * </code></pre>
     * <p>
     * <b>Example (Java):</b>
     * <pre><code>
     conversation.prepareMessage()
             .setBody("Take a look at this!")
             .addMedia("image/png", "cat.png", new FileInputStream(filePath), new MediaUploadListener() {
                 @Override
                 public void onStarted() {
                     Timber.d("Upload started");
                 }

                 @Override
                 public void onProgress(long bytesSent) {
                     Timber.d("Uploaded " + bytes + " bytes of cat.png");
                 }

                 @Override
                 public void onCompleted(@NotNull String mediaSid) {
                     Timber.d("Upload completed");
                 }

                 @Override
                 public void onFailed(@NotNull ErrorInfo errorInfo) {
                     Timber.d("Upload failed: " + errorInfo);
                 }
             })
             .buildAndSend(new CallbackListener<Message>() {
                 @Override
                 public void onSuccess(Message result) {
                     Timber.d("Message sent");
                 }

                 @Override
                 public void onError(ErrorInfo errorInfo) {
                     Timber.d("Error send message: " + errorInfo);
                 }
             });
     * </code></pre>
     * @return {@link MessageBuilder} for building new message and sending it into the conversation.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    MessageBuilder prepareMessage();

    /**
     * Removes a message from the conversation.
     *
     * @param message   The message to remove.
     * @param listener  Status listener to report result of the operation.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void removeMessage(Message message, StatusListener listener);

    /**
     * Get last message date in the conversation.
     * @see UpdateReason#LAST_MESSAGE status update
     * @return Date of last message in the conversation or null if no last message or no access to conversation.
     */
    Date getLastMessageDate();

    /**
     * Get last message's index in the conversation.
     * @see UpdateReason#LAST_MESSAGE status update
     * @return Index of last message in the conversation or null if no last message or no access to conversation.
     */
    Long getLastMessageIndex();

    /**
     * Return user last read message index for the conversation.
     *
     * @return read message index. Null if not set
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    Long getLastReadMessageIndex();

    /**
     * Set user last read message index for the conversation.
     *
     * @param lastReadMessageIndex read message index
     * @param listener Status listener to report result of the operation that receives current number of unread messages.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void setLastReadMessageIndex(long lastReadMessageIndex, CallbackListener<Long> listener);

    /**
     * Increase user last read message index for the conversation.
     * Index is ignored if it is smaller than user current index.
     *
     * @param lastReadMessageIndex read message index
     * @param listener Status listener to report result of the operation that receives current number of unread messages.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void advanceLastReadMessageIndex(long lastReadMessageIndex, CallbackListener<Long> listener);

    /**
     * Mark all messages in conversation as read.
     * This method set last read message index to last message index in conversation.
     *
     * @param listener   Status listener to report result of the operation that receives current number of unread messages.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void setAllMessagesRead(CallbackListener<Long> listener);

    /**
     * Mark all messages in conversation as unread.
     * This method set last read message index before the first message index in conversation.
     *
     * @param listener   Status listener to report result of the operation that receives current number of unread messages.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void setAllMessagesUnread(CallbackListener<Long> listener);

    /**
     * Fetch at most count messages including and prior to the specified index.
     *
     * @param index start index
     * @param count count of messages to load
     * @param listener operation callback listener
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void getMessagesBefore(long index, int count, @NonNull CallbackListener<List<Message>> listener);

    /**
     * Fetch at most count messages including and subsequent to the specified index.
     *
     * @param index start index
     * @param count count of messages to load
     * @param listener operation callback listener
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void getMessagesAfter(long index, int count, @NonNull CallbackListener<List<Message>> listener);

    /**
     * Load last messages in conversation.
     *
     * @param count count of messages to load
     * @param listener operation callback listener
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void getLastMessages(int count, @NonNull CallbackListener<List<Message>> listener);

    /**
     * Get message object by known index.
     *
     * @param  index Message index.
     * @param  listener Listener for Message object at index.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void getMessageByIndex(long index, @NonNull CallbackListener<Message> listener);

    /**@}*/
    /**@name Participants */
    /**@{*/

    /**
     * Add the participant with the specified username to this conversation.
     * <p>
     * If the participant is already present in the conversation roster an error will be returned.
     *
     * @param identity The username of the participant to add to this conversation.
     * @param attributes The developer-provided {@link Attributes} for participant or null to use default empty attributes.
     * @param listener  StatusListener to report status of the action.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void addParticipantByIdentity(String identity, Attributes attributes, StatusListener listener);

    /**
     * Add specified non chat participant to this conversation (sms, whatsapp participants etc).
     * <p>
     * If the participant is already present in the conversation roster an error will be returned.
     *
     * @param address The participant address to add to this conversation (phone number for sms and whatsapp participants)
     * @param proxyAddress Proxy address (Twilio phone number for sms and whatsapp participants).
     *                     See conversations <a href="https://www.twilio.com/docs/conversations/quickstart">quickstart</a> for more info.
     * @param attributes The developer-provided {@link Attributes} for participant or null to use default empty attributes.
     * @param listener  StatusListener to report status of the action.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void addParticipantByAddress(String address, String proxyAddress, Attributes attributes, StatusListener listener);

    /**
     * Remove specified participant from this conversation.
     *
     * @param participant    The participant to remove from this conversation.
     * @param listener  StatusListener to report status.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void removeParticipant(Participant participant, StatusListener listener);

    /**
     * Remove the participant with the specified identity from this conversation.
     *
     * @param identity  The identity of the participant to remove from this conversation.
     * @param listener  StatusListener to report status.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    void removeParticipantByIdentity(String identity, StatusListener listener);

    /**
     * Obtain an array of participants of this conversation.
     *
     * @return An array of Participant objects representing the membership of the conversation.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    List<Participant> getParticipantsList();

    /**
     * Get a conversation participant by identity.
     *
     * @param  identity The identity of the participant to look up.
     * @return Participant object in current conversation, or null if not found.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    Participant getParticipantByIdentity(String identity);

    /**
     * Get a conversation participant by sid.
     *
     * @param  sid The sid of the participant to look up.
     * @return Participant object in current conversation, or null if not found.
     * @throws IllegalStateException If conversation isn't synchronized properly.
     */
    Participant getParticipantBySid(String sid);

    /**@}*/

    /**
     * Builder for preparing new message.
     *
     * @see #prepareMessage()
     */
    interface MessageBuilder {

        /**
         * Set new message body.
         *
         * If content SID is set by the {@link #setContentTemplate} method, then the body field
         * is ignored and the body of the sent message is filled with resolved content from the
         * {@link ContentTemplate} instead.
         *
         * @param body Message body.
         * @return Self for chaining.
         */
        MessageBuilder setBody(String body);

        /**
         * Set new message subject.
         * <p>
         * Subject is a part of the message which will be sent to email participants
         * of the conversation (if any) as subject field of email.
         * <p>
         * If no email participants in the conversation - other chat participants will receive this
         * field as part of the message anyway.
         *
         * @param subject Message subject.
         * @return Self for chaining.
         */
        MessageBuilder setSubject(String subject);

        /**
         * Set new message attributes.
         *
         * @param  attributes {@link Attributes} for the message.
         * @return            Self for chaining.
         */
        MessageBuilder setAttributes(@NotNull Attributes attributes);

        /**
         * Set new message email body.
         * <p>
         * This body will be sent to email participants of the conversation
         *
         * @param emailBody content of the body.
         * @param contentType type of the body, see {@link ConversationLimits#emailBodiesAllowedContentTypes}.
         * @return Self for chaining.
         */
        MessageBuilder setEmailBody(@NotNull String emailBody, @NotNull String contentType);

        /**
         * Set new message email body.
         * <p>
         * This body will be sent to email participants of the conversation
         *
         * @param inputStream content of the body.
         * @param contentType type of the body, see {@link ConversationLimits#emailBodiesAllowedContentTypes}.
         * @param uploadListener {@link MediaUploadListener} for monitoring upload process.
         * @return Self for chaining.
         */
        MessageBuilder setEmailBody(@NotNull InputStream inputStream, @NotNull String contentType, @Nullable MediaUploadListener uploadListener);

        /**
         * Set new message email history.
         * <p>
         * This history will be sent to email participants of the conversation
         *
         * @param emailHistory content of the email history.
         * @param contentType type of the history, see {@link ConversationLimits#emailHistoriesAllowedContentTypes}.
         * @return Self for chaining.
         */
        MessageBuilder setEmailHistory(@NotNull String emailHistory, @NotNull String contentType);

        /**
         * Set new message email history.
         * <p>
         * This history will be sent to email participants of the conversation
         *
         * @param inputStream content of the email history.
         * @param contentType type of the history, see {@link ConversationLimits#emailHistoriesAllowedContentTypes}.
         * @param uploadListener {@link MediaUploadListener} for monitoring upload process.
         * @return Self for chaining.
         */
        MessageBuilder setEmailHistory(@NotNull InputStream inputStream, @NotNull String contentType, @Nullable MediaUploadListener uploadListener);

        /**
         * Adds attachment for the new message.
         *
         * @param inputStream content of attachment.
         * @param contentType type of the attachment.
         * @param filename name of the attachment.
         * @param uploadListener {@link MediaUploadListener} for monitoring upload process.
         * @return Self for chaining.
         */
        MessageBuilder addMedia(@NotNull InputStream inputStream, @NotNull String contentType, @Nullable String filename, @Nullable MediaUploadListener uploadListener);

        /**
         * Adds {@link ContentTemplate} SID for the message. This overload uses default variables values.
         * Use {@link #setContentTemplate(String, List)} to specify custom variable values.
         *
         * Adding non-null content SID converts message to a rich message. In this case the body field
         * is ignored and the body of the sent message is filled with resolved content from the
         * {@link ContentTemplate} instead.
         *
         * Use {@link ConversationsClient#getContentTemplates} to request all available {@link ContentTemplate}s.
         *
         * @param contentTemplateSid SID of the {@link ContentTemplate}
         * @return Self for chaining.
         */
        MessageBuilder setContentTemplate(String contentTemplateSid);

        /**
         * Adds {@link ContentTemplate} SID for the message. This overload receives custom variables values.
         * Use {@link #setContentTemplate(String)} to send rich message with default variable values.
         *
         * Adding non-null content SID converts message to a rich message. In this case the body field
         * is ignored and the body of the sent message is filled with resolved content from the
         * {@link ContentTemplate} instead.
         *
         * Use {@link ConversationsClient#getContentTemplates} to request all available {@link ContentTemplate}s.
         *
         * @param contentTemplateSid SID of the {@link ContentTemplate}
         * @param variables Custom variables to resolve the template.
         * @return Self for chaining.
         */
        MessageBuilder setContentTemplate(String contentTemplateSid, List<ContentTemplateVariable> variables);

        /**
         * Builds new {@link UnsentMessage}.
         *
         * @return {@link UnsentMessage} which is ready for send.
         */
        UnsentMessage build();

        /**
         * Builds new {@link UnsentMessage} and sends it immediately.
         *
         * @param listener Listener to report result of the operation that receives sent Message object.
         * @return {@link CancellationToken} which allows to cancel network request.
         */
        CancellationToken buildAndSend(@Nullable CallbackListener<Message> listener);
    }

    /**
     * Unsent message which is ready for sending.
     *
     * @see MessageBuilder
     */
    interface UnsentMessage {

        /**
         * Send the prepared message to the conversation.
         *
         * @param listener Listener to report result of the operation that receives sent Message object.
         * @return {@link CancellationToken} which allows to cancel network request.
         */
        CancellationToken send(@Nullable CallbackListener<Message> listener);
    }
}
