//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

/**
 * Listener interface to report conversation operations such as participant join/leave and messages
 * add/delete/update.
 * <p>
 * You should create this listener and attach it to conversation using {@link Conversation#addListener} if you
 * want to receive status updates on conversation changes.
 * <p>
 * This listener's callbacks will be called on the originating thread if it has a Looper, otherwise
 * on the main UI thread.
 */
/*!
 * To invoke listener callbacks on a specific thread, wrap your call in a Handler invocation with
 * a specific thread `someOtherThread` e.g.
 *
 * ```java
 * Handler handler = new Handler(someOtherThreadLooper);
 * handler.post(new Runnable() {
 *     @Override
 *     public void run() {
 *         client.getConversations().getUserConversationsList(myListener);
 *     }
 * });
 * ```
 *
 * Then `myListener` will be invoked on `someOtherThread`. Note that your thread has to have an event
 * Looper to be able to run callbacks.
 */
public interface ConversationListener {

    /**@name Messages */
    /**@{*/

    /**
     * Called when a Message is added to the conversation the current user is subscribed to.
     * <p>
     * You could obtain the Conversation where it was added by using {@link Message#getConversation}
     * or {@link Message#getConversationSid}.
     *
     * @param message The message that has been added.
     * @see #onMessageUpdated
     * @see #onMessageDeleted
     */
    void onMessageAdded(Message message);

    /**
     * Called when a Message is changed in the conversation the current user is subscribed to.
     * <p>
     * You could obtain the Conversation where it was changed by using {@link Message#getConversation}
     * or {@link Message#getConversationSid}.
     * <p>
     * Message change events include {@link Message#getBody body} updates
     * and {@link Message#getAttributes attribute} updates.
     *
     * @param message The message that changed.
     * @param reason Reason for the update.
     * @see #onMessageAdded
     * @see #onMessageDeleted
     */
    void onMessageUpdated(Message message, Message.UpdateReason reason);

    /**
     * Called when a Message is deleted from the conversation the current user is subscribed to.
     * <p>
     * You could obtain the Conversation where it was deleted from by using {@link Message#getConversation}
     * or {@link Message#getConversationSid}.
     *
     * @param message The message that got deleted.
     * @see #onMessageAdded
     * @see #onMessageUpdated
     */
    void onMessageDeleted(Message message);

    /**@}*/
    /**@name Participants */
    /**@{*/

    /**
     * Called when a Participant is added to the conversation the current user is subscribed to.
     * <p>
     * You could obtain the Conversation it was added to by using {@link Participant#getConversation}.
     *
     * @param participant The participant that joined.
     * @see #onParticipantUpdated
     * @see #onParticipantDeleted
     */
    void onParticipantAdded(Participant participant);

    /**
     * Called when a Participant is changed in the conversation the current user is subscribed to.
     * <p>
     * You could obtain the Conversation where it was changed by using {@link Participant#getConversation}.
     * <p>
     * Participant change events include {@link Participant#getLastReadMessageIndex read horizon}
     * updates, {@link User#isOnline reachability status} updates.
     *
     * @param participant The participant that changed.
     * @param reason Reason for the update.
     * @see #onParticipantAdded
     * @see #onParticipantDeleted
     */
    void onParticipantUpdated(Participant participant, Participant.UpdateReason reason);

    /**
     * Called when a Participant is deleted from the conversation the current user is subscribed to.
     * <p>
     * You could obtain the Conversation it was deleted from by using {@link Participant#getConversation}.
     *
     * @param participant The participant that was deleted.
     * @see #onParticipantAdded
     * @see #onParticipantUpdated
     */
    void onParticipantDeleted(Participant participant);

    /**@}*/
    /**@name Typing indicator */
    /**@{*/

    /**
     * Called when the Participant starts typing in a Conversation.
     *
     * @param conversation Conversation on which the event happened.
     * @param participant The participant that started typing.
     * @see #onTypingEnded
     */
    void onTypingStarted(Conversation conversation, Participant participant);

    /**
     * Called when the Participant stops typing in a Conversation.
     * <p>
     * Typing indicator has a timeout after user stops typing to avoid triggering it too often.
     * Expect about 5 seconds delay between stopping typing and receiving typing ended event.
     *
     * @param conversation Conversation on which the event happened.
     * @param participant The participant that stopped typing.
     * @see #onTypingStarted
     */
    void onTypingEnded(Conversation conversation, Participant participant);

    /**@}*/
    /**@name Conversation status */
    /**@{*/

    /**
     * Called when conversation synchronization status changed.
     *
     * @param conversation Conversation for which synchronization status has changed.
     * @see Conversation.SynchronizationStatus
     */
    void onSynchronizationChanged(Conversation conversation);

    /**@}*/
}
