//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.twilio.conversations.content.ContentTemplate;
import com.twilio.util.TwilioLogger;

import java.util.List;
import java.util.Map;

/**
 * Conversations client - main entry point for the Conversations SDK.
 */
public interface ConversationsClient {

    /**
     * Set verbosity level for log messages to be printed to android logcat.
     * <p>
     * Default log level is {@link LogLevel#SILENT}.
     *
     * @param level Verbosity level. See {@link LogLevel} for supported options.
     */
    static void setLogLevel(LogLevel level) {
        ConversationsClientImpl.setLogLevel(level.mValue);
    }

    /**@name Lifecycle */
    /**@{*/

    /**
     * Creates a new Conversations client instance with a token manager and client properties.
     * <p>
     * Callback listener is invoked with reference to ConversationsClient when client
     * initialization is completed.
     *
     * @param context
     *            The Application Context from your Android application. Make
     *            sure you don't pass an Activity Context. You can retrieve the
     *            Application Context by calling getApplicationContext() on your
     *            Activity. Cannot be null.
     * @param token            Access token for Conversations service.
     * @param properties       ConversationsClient initialization properties
     * @param listener         Callback listener that will receive reference to
     *                         the newly initialized ConversationsClient.
     */
    static void create(@NonNull Context context,
                       @NonNull String token,
                       @NonNull Properties properties,
                       @NonNull CallbackListener<ConversationsClient> listener) {

        ConversationsClientImpl.create(context, token, properties, listener);
    }

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * Returns the version of the Conversations SDK.
     *
     * @return The version of the SDK.
     */
    static String getSdkVersion() {
        return ConversationsClientImpl.getSdkVersion();
    }

    /**
     * Get properties for current client
     *
     * @return properties for client
     */
    Properties getProperties();

    /**@}*/
    /**@name Lifecycle */
    /**@{*/

    /**
     * Method to update the authentication token for this client.
     *
     * @param token     A new access token for this Client.
     * @param listener  Listener that will receive callback with the result.
     */
    void updateToken(String token, StatusListener listener);

    /**
     * Cleanly shuts down the messaging client when you are done with it.
     * It will dispose() the client after shutdown, so it could not be reused.
     */
    void shutdown();

    /**@}*/
    /**@name Listeners */
    /**@{*/

    /**
     * Method to add listener for this Client.
     *
     * @param listener the listener to add.
     */
    void addListener(@NonNull ConversationsClientListener listener);

    /**
     * Method to remove listener from this Client.
     *
     * @param listener the listener to remove.
     */
    void removeListener(@NonNull ConversationsClientListener listener);

    /**
     * Method to remove all listeners from this Client.
     */
    void removeAllListeners();

    /**@}*/
    /**@name Push notifications */
    /**@{*/

    /**
     * Register Firebase Messaging token for push notification updates.
     *
     * @param token The registration token an Android application needs to register with FCM
     *              connection servers before it can receive messages.
     * @param listener Listener that will receive callback with the result.
     */
    void registerFCMToken(FCMToken token, StatusListener listener);

    /**
     * Unregister from Firebase Messaging updates.
     *
     * @param token
     *      The {@link FCMToken} to unregister.
     *      Must be the same token that was passed in {@link #registerFCMToken} before.
     * @param listener Listener that will receive callback with the result.
     */
    void unregisterFCMToken(FCMToken token, StatusListener listener);

    /**
     * Queue the incoming notification with the conversations library for processing.
     *
     * You must call this function if you want the conversations library to call your
     * {@link ConversationsClientListener#onAddedToConversationNotification},
     * {@link ConversationsClientListener#onRemovedFromConversationNotification},
     * or {@link ConversationsClientListener#onNewMessageNotification} callbacks.
     *
     * @param notification Notification received from FCM.
     */
    void handleNotification(NotificationPayload notification);

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * @return Current transport state.
     */
    ConnectionState getConnectionState();

    /**
     * Get user identity for current user
     *
     * @return User identity for current user
     */
    String getMyIdentity();

    /**
     * Get list of <b>all</b> Conversation participants with a given identity.
     * <p>
     * The effect of this function is to find and return all Participant instances across multiple
     * conversations with the given identity.
     *
     * @param  identity Participant identity to look up.
     * @return          List of Participants across Conversations with given identity.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    List<Participant> getParticipantsByIdentity(String identity);

    /**@}*/
    /**@name Users */
    /**@{*/

    /**
     * Get user based on user identity and subscribe to real-time updates for this user.
     * There's a limit on the number of simultaneously subscribed objects in the SDK.
     * This is to reduce consumed memory and network traffic.
     *
     * @param identity User identity to query.
     * @param listener Listener to receive resulting subscribed User object.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     * @see #getSubscribedUsers
     */
    void getAndSubscribeUser(String identity, CallbackListener<User> listener);

    /**
     * Get a list of currently subscribed User objects.
     * <p>
     * These objects receive status updates in real-time.
     * When you subcribe to too many users simultaneously, the oldest subscribed users will be
     * automatically unsubscribed.
     *
     * @return List of subscribed User objects.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    List<User> getSubscribedUsers();

    /**
     * Get logged in User object.
     * <p>
     * Returns the User object for your currently logged in User. You can query and update
     * this object at will.
     *
     * @return Own User object.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    User getMyUser();

    /**@}*/
    /**@name Conversations */
    /**@{*/

    /**
     * Create a conversation with friendly name.
     * <p>
     * This operation creates a new conversation entity on the backend.
     *
     * @param friendlyName  Friendly name of the new conversation.
     * @param listener      Listener that receives the status of create conversation action.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    void createConversation(String friendlyName, CallbackListener<Conversation> listener);

    /**
     * Get ConversationBuilder to create conversation with options.
     * <p>
     * A pattern to build conversation with options is:
     * <p>
     * <b>Example:</b>
     * <pre><code>
     conversationBuilder()
         .withFriendlyName("Conversation")
         .build(listener);
     * </code><pre>
     * @return A ConversationBuilder used to create conversation with options.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    ConversationBuilder conversationBuilder();

    /**
     * Retrieves a conversation with the specified SID or unique name.
     *
     * @param conversationSidOrUniqueName Identifier for the conversation.
     * @param listener Listener to receive the conversation.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    void getConversation(String conversationSidOrUniqueName, CallbackListener<Conversation> listener);

    /**
     * Request list of user's joined conversations.
     *
     * @return A list of Conversations that the user is joined to.
     * @throws IllegalStateException If ConversationClient isn't synchronized properly.
     */
    List<Conversation> getMyConversations();

    /**
     * Get reachability service status.
     *
     * @return true if reachability info is available, false otherwise.
     */
    boolean isReachabilityEnabled();

    /**@}*/

    /**
     * Get content URLs for all media attachments in the given set using single network request.
     *
     * @param media    List of media attachments to query for content URL.
     * @param listener Listener to receive Map of mediaSID to content temporary URL.
     * @return {@link CancellationToken} which allows to cancel network request.
     */
    @NonNull
    CancellationToken getTemporaryContentUrlsForMedia(@NonNull List<Media> media,
                                                      @NonNull CallbackListener<Map<String, String>> listener);

    /**
     * Get content URLs for all media attachments in the given set using single network request.
     *
     * @param mediaSids List of media sids to query for content URL.
     * @param listener  Listener to receive Map of mediaSID to content temporary URL.
     * @return {@link CancellationToken} which allows to cancel network request.
     */
    @NonNull
    CancellationToken getTemporaryContentUrlsForMediaSids(@NonNull List<String> mediaSids,
                                                          @NonNull CallbackListener<Map<String, String>> listener);

    /**
     * Request rich content templates.
     *
     * Rich content templates can be created using console or REST API.
     *
     * @param listener  Listener to receive rich content templates.
     * @return {@link CancellationToken} which allows to cancel network request.
     */
    @NonNull
    CancellationToken getContentTemplates(@NonNull CallbackListener<List<ContentTemplate>> listener);

    /**
     * Represents client initialization status.
     */
    enum SynchronizationStatus {
        /** Initialization is started. */
        STARTED(0),
        /** Conversations list initialization completed. */
        CONVERSATIONS_COMPLETED(1),
        /** Initialization completed. */
        COMPLETED(2),
        /** Initialization failed. */
        FAILED(3);

        private final int value;
        private SynchronizationStatus(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link SynchronizationStatus} by associated integer value. */
        public static SynchronizationStatus fromInt(int value)
        {
            for (SynchronizationStatus t : ConversationsClient.SynchronizationStatus.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for SynchronizationStatus");
        }
    }

    /**
     * Represents underlying twilsock connection state.
     */
    enum ConnectionState {
        /** Transport is trying to connect and register or trying to recover. */
        CONNECTING(0),
        /** Transport is working. */
        CONNECTED(1),
        /** Transport is not working. */
        DISCONNECTED(2),
        /** Transport was not enabled because authentication token is invalid or not authorized. */
        DENIED(3),
        /** Error in connecting or sending transport message. Possibly due to offline. */
        ERROR(4),
        /** Server has rejected enabling transport and customer action is required. */
        FATAL_ERROR(5);

        private final int value;
        private ConnectionState(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link ConnectionState} by associated integer value. */
        public static ConnectionState fromInt(int value)
        {
            for (ConnectionState t : ConversationsClient.ConnectionState.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for ConnectionState");
        }
    }

    /**
     * Helper to create new conversation with provided data.
     */
    interface ConversationBuilder {
        /**
         * Set conversation friendly name.
         * @param  friendlyName Friendly name for conversation to create.
         * @return              Self for chaining.
         */
        ConversationBuilder withFriendlyName(String friendlyName);

        /**
         * Set conversation unique name.
         * @param  uniqueName User-specified unique name for conversation.
         * @return            Self for chaining.
         */
        ConversationBuilder withUniqueName(String uniqueName);

        /**
         * Set user-specified custom conversation attributes.
         * @param  attributes {@link Attributes} object with custom conversation attributes.
         * @return            Self for chaining.
         */
        ConversationBuilder withAttributes(@NonNull Attributes attributes);

        /**
         * Method to create conversation with options.
         * <p>
         * Set options using {@link #withFriendlyName},
         * {@link #withUniqueName} and {@link #withAttributes} before calling this function.
         *
         * @param listener Listener that receives the status of create conversation action.
         */
        void build(CallbackListener<Conversation> listener);
    }

    /**
     * Properties for client initialization configuration
     */
    interface Properties {
        /** Minimum valid command timeout value that could be passed into {@link Builder#setCommandTimeout}. */
        int MIN_COMMAND_TIMEOUT = 10000;
        /** Default command timeout value that could be passed into {@link Builder#setCommandTimeout}. */
        int DEFAULT_COMMAND_TIMEOUT = 10000;

        /**
         * Creates a new {@link Builder} instance for building {@link Properties}.
         *
         * @return New {@link Builder} instance.
         */
        static Builder newBuilder() {
            return new ConversationsClientImpl.PropertiesImpl.BuilderImpl();
        }

        /**
         * Twilio server region to connect to.
         * Instances exist in specific regions, so this should only be changed if needed.
         *
         * @return Region such as `us1` or `ie1`.
         */
        String getRegion();

        /**
         * If useProxy flag is `true` {@link ConversationsClient} will try to read and
         * apply proxy settings in the following order:
         * <ol>
         *   <li>
         *   If there is no proxysettings.properties file in the assets folder of your app
         *      then android system proxy settings will be applied.
         *   <li>
         *   If the proxysettings.properties file exists in the assets folder of your app
         *      then proxy configuration will be read from it and android system settings will be ignored.
         *   <li>
         *   If proxy settings cannot be read either from the proxysettings.properties file and from
         *      android system settings {@link ConversationsClient} will use direct connection.
         * </ol>
         * If this flag is `false` all proxy settings will be ignored and direct connection will be used.
         * <p>
         * The default value is `false`.
         * <p>
         * In current version connection via proxy server is not supported for media. So media uploading
         * always uses direct connection.
         * <p>
         * <b>Example of the proxysettings.properties file:</b>
         * <pre><code>
         host=192.168.8.108
         port=8080
         user=myUser
         password=myPassword
         * </code><pre>
         */
        boolean useProxy();

        /**
         * Defer certificate trust decisions to Android OS, overriding the default of
         * certificate pinning for Twilio back-end connections.
         * <p>
         * Twilio client SDKs utilize certificate pinning to prevent man-in-the-middle attacks
         * against your connections to our services. Customers in certain very specific
         * environments may need to opt-out of this if custom certificate authorities must
         * be allowed to intentionally intercept communications for security or policy reasons.
         * <p>
         * Setting this property to `true` for a Conversations Client instance will defer to Android to
         * establish whether or not a given connection is providing valid and trusted TLS certificates.
         * <p>
         * Keeping this property at its default value of `false` allows the Twilio client SDK
         * to determine trust when communicating with our servers.
         * <p>
         * The default value is `false`.
         */
        boolean getDeferCA();

        /**
         * Timeout in milliseconds for commands which SDK sends over network.
         *
         * @return Timeout in milliseconds for commands which SDK sends over network
         * (i.e. {@link Conversation.UnsentMessage#send}, {@link Conversation#join} etc).
         * <p>
         * {@link StatusListener#onError} will be called when timeout is reached.
         * <p>
         * In case of bad connectivity SDK retries to send command until timeout is reached.
         * Timeout could occur earlier than specified time if there is no enough time to make one more attempt.
         * <p>
         * The default value is {@link #DEFAULT_COMMAND_TIMEOUT}.
         */
        int getCommandTimeout();

        /**
         * Builder class for client properties
         */
        interface Builder {
            /**
             * Select Twilio server region to connect to.
             * Conversations instances exist in specific regions, so this should only be changed if needed.
             * Defaults to `us1`.
             *
             * @param  region Region such as `us1` or `ie1`.
             * @return        Self for chaining.
             */
            Builder setRegion(String region);

            /**
             * If useProxy flag is `true` {@link ConversationsClient} will try to read and
             * apply proxy settings in the following order:
             * <ol>
             *   <li>
             *   If there is no proxysettings.properties file in the assets folder of your app
             *      then android system proxy settings will be applied.
             *   <li>
             *   If the proxysettings.properties file exists in the assets folder of your app
             *      then proxy configuration will be read from it and android system settings will be ignored.
             *   <li>
             *   If proxy settings cannot be read either from the proxysettings.properties file and from
             *      android system settings {@link ConversationsClient} will use direct connection.
             * </ol>
             * If this flag is `false` all proxy settings will be ignored and direct connection will be used.
             * <p>
             * The default value is `false`.
             * <p>
             * In current version connection via proxy server is not supported for media. So media uploading
             * always uses direct connection.
             * <p>
             * <b>Example of the proxysettings.properties file:</b>
             * <pre><code>
             host=192.168.8.108
             port=8080
             user=myUser
             password=myPassword
             * </code><pre>
             * @param  useProxy `true` for using proxy server, `false` for using direct connection.
             * @return Self for chaining.
             */

            Builder setUseProxy(boolean useProxy);

            /**
             * Defer certificate trust decisions to Android OS, overriding the default of
             * certificate pinning for Twilio back-end connections.
             * <p>
             * Twilio client SDKs utilize certificate pinning to prevent man-in-the-middle attacks
             * against your connections to our services. Customers in certain very specific
             * environments may need to opt-out of this if custom certificate authorities must
             * be allowed to intentionally intercept communications for security or policy reasons.
             * <p>
             * Setting this property to `true` for a Conversations Client instance will defer to Android to
             * establish whether or not a given connection is providing valid and trusted TLS certificates.
             * <p>
             * Keeping this property at its default value of `false` allows the Twilio client SDK
             * to determine trust when communicating with our servers.
             * <p>
             * The default value is `false`.
             *
             * @param  defer True to use Android OS certficate trust store, false to use bundled with the SDK.
             * @return            Self for chaining.
             */
            Builder setDeferCertificateTrustToPlatform(boolean defer);

            /**
             * Set timeout for commands which SDK sends over network (i.e. {@link Conversation.UnsentMessage#send},
             * {@link Conversation#join} etc). {@link StatusListener#onError} will be called when timeout is reached.
             * <p>
             * In case of bad connectivity SDK retries to send command until timeout is reached.
             * Timeout could occur earlier than specified time if there is no enough time to make one more attempt.
             * <p>
             * At the moment when timeout happened the request could be already transmitted to the backend or not.
             * In case when it has been transmitted, after timeout it doesn't rollback any actions made by request,
             * just ignores the response.
             * <p>
             * The default value is {@link #DEFAULT_COMMAND_TIMEOUT}.
             *
             * @param commandTimeout timeout in milliseconds. Must be greater than or equal to {@link #MIN_COMMAND_TIMEOUT}
             * @return Self for chaining.
             * @throws IllegalArgumentException if passed value less than {@link #MIN_COMMAND_TIMEOUT}
             */
            Builder setCommandTimeout(int commandTimeout);

            /**
             * Create Properties object from this Builder.
             * @return New Properties object to use with Client construction.
             */
            ConversationsClient.Properties createProperties();
        }
    }

    /**
     * Log level constants for passing to {@link #setLogLevel}.
     */
    enum LogLevel {
        /** Show low-level tracing messages as well as all Debug log messages. */
        VERBOSE(TwilioLogger.VERBOSE),
        /** Show low-level debugging messages as well as all Info log messages. */
        DEBUG(TwilioLogger.DEBUG),
        /** Show informational messages as well as all Warning log messages. */
        INFO(TwilioLogger.INFO),
        /** Show warnings as well as all Critical log messages. */
        WARN(TwilioLogger.WARN),
        /** Show critical log messages as well as all Fatal log messages. */
        ERROR(TwilioLogger.ERROR),
        /** Show fatal errors only. */
        ASSERT(TwilioLogger.ASSERT),
        /** Show no log messages. */
        SILENT(TwilioLogger.SILENT);

        final int mValue;
        LogLevel(int value) { mValue = value; }

        /** @return The associated integer value. */
        public int getValue() { return mValue; }
        /** @return The {@link LogLevel} by associated integer value. */
        public static LogLevel fromInt(int value)
        {
            for (LogLevel item : values()) {
                if (item.getValue() == value)
                    return item;
            }
            throw new IllegalStateException("Invalid value " + value + " for LogLevel");
        }
    }

    /**
     * Class which represents token received by application from the Firebase Cloud Messaging service.
     */
    class FCMToken {

        private final String token;

        /**
         * @param token Token received by application from the Firebase Cloud Messaging service.
         */
        public FCMToken(String token) {
            if (TextUtils.isEmpty(token)) {
                throw new IllegalArgumentException("Token must not be empty");
            }
            this.token = token;
        }

        public String getToken() {
            return token;
        }
    }
}
