//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import com.twilio.util.ErrorInfo;

/**
 * This interface defines ConversationsClient callback methods.
 */
public interface ConversationsClientListener {

    /**@name Conversation events */
    /**@{*/

    /**
     * Called when the current user has a conversation added to their conversation list,
     * conversation status is not specified.
     *
     * @param conversation The conversation that was added.
     */
    void onConversationAdded(Conversation conversation);

    /**
     * Called when the conversation is updated.
     * Conversation synchronization updates are delivered via different callback.
     *
     * @param conversation The conversation that updated.
     * @param reason  Which information in the conversation has updated.
     */
    void onConversationUpdated(Conversation conversation, Conversation.UpdateReason reason);

    /**
     * Called when current participant is removed from the conversation or conversation is closed or deleted.
     *
     * @param conversation The conversation that got deleted.
     */
    void onConversationDeleted(Conversation conversation);

    /**
     * Called when conversation synchronization status changed.
     * Use {@link Conversation#getSynchronizationStatus} to obtain new conversation status.
     *
     * @param conversation Conversation whose synchronization status has changed.
     */
    void onConversationSynchronizationChange(Conversation conversation);

    /**@}*/
    /**@name Conversations client events */
    /**@{*/

    /**
     * Called when an error condition occurs.
     *
     * @param errorInfo {@link ErrorInfo} object containing error info
     */
    void onError(ErrorInfo errorInfo);

    /**@}*/
    /**@name User events */
    /**@{*/

    /**
     * Called when user info is updated for currently loaded users
     *
     * @param user User info object.
     * @param reason Reason for update.
     */
    void onUserUpdated(User user, User.UpdateReason reason);

    /**
     * Called when a user is subscribed to and will receive realtime state updates.
     * @param user User that has been subscribed.
     */
    void onUserSubscribed(User user);

    /**
     * Called when a user is unsubscribed from and will not receive realtime state updates anymore.
     * @param user User that has been unsubscribed.
     */
    void onUserUnsubscribed(User user);

    /**@}*/
    /**@name Conversations client events */
    /**@{*/

    /**
     * Called when client synchronization status changes
     *
     * @param status New client synchronization status.
     */
    void onClientSynchronization(ConversationsClient.SynchronizationStatus status);

    /**@}*/
    /**@name Push notifications */
    /**@{*/

    /**
     * Called when client receives a push notification for new message.
     *
     * @param conversationSid Sid of the related conversation.
     * @param messageSid Sid of the related message.
     * @param messageIndex Index of the related message.
     */
    void onNewMessageNotification(String conversationSid, String messageSid, long messageIndex);

    /**
     * Called when client receives a push notification for added to conversation event.
     *
     * @param conversationSid Sid of the related conversation.
     */
    void onAddedToConversationNotification(String conversationSid);

    /**
     * Called when client receives a push notification for removed from conversation event.
     *
     * @param conversationSid Sid of the related conversation.
     */
    void onRemovedFromConversationNotification(String conversationSid);

    /**
     * Called when you are successfully registered for push notifications.
     */
    void onNotificationSubscribed();

    /**
     * Called when registering for push notifications fails.
     * @param errorInfo An error indicating the failure.
     */
    void onNotificationFailed(ErrorInfo errorInfo);

    /**@}*/
    /**@name Conversations client events */
    /**@{*/

    /**
     * Called when client connnection state has changed.
     *
     * @param state New connection state.
     */
    void onConnectionStateChange(ConversationsClient.ConnectionState state);

    /**
     * Called when token has expired.
     *
     * In response, ConversationsClient should generate a new token and call updateToken()
     * as soon as possible.
     */
    void onTokenExpired();

    /**
     * Called when token is about to expire soon.
     *
     * In response, ConversationsClient should generate a new token and call updateToken()
     * as soon as possible.
     *
     * This event is not guaranteed to be delivered if {@link ConversationsClient} is created with
     * token ttl less than 5 minutes.
     */
    void onTokenAboutToExpire();
}
