//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Date;

/**
 * Contains detailed information about a {@link Message}'s delivery statuses to one of {@link Participant participants}
 */
public class DetailedDeliveryReceipt {

    /** Delivery status of a message. */
    public enum DeliveryStatus {
        /**  
         * Message considered as <b>read</b> by a participant, if the message has been delivered and opened by the
         * recipient in the conversation. The recipient must have enabled read receipts.
         */
        READ(0),

        /** 
         * Message considered as <b>undelivered</b> to a participant, if Twilio has received a delivery receipt
         * indicating that the message was not delivered. This can happen for many reasons including carrier content
         * filtering and the availability of the destination handset.
         */
        UNDELIVERED(1),

        /** 
         * Message considered as <b>delivered</b> to a participant, if Twilio has received confirmation of message
         * delivery from the upstream carrier, and, where available, the destination handset.
         */
        DELIVERED(2),

        /**  
         * Message considered as <b>failed</b> to be delivered to a participant if the message could not be sent.
         * This can happen for various reasons including queue overflows, account suspensions and media
         * errors (in the case of MMS for instance).
         */
        FAILED(3),

        /** 
         * Message considered as <b>sent</b> to a participant, if the nearest upstream carrier accepted the message.
         */
        SENT(4),

        /**
         * Message considered as <b>queued</b>, if it is waiting in a queue to be sent to an upstream carrier.
         */
        QUEUED(5);

        private final int mValue;
        private DeliveryStatus(int value) { this.mValue = value; }

        /** @return The associated integer value. */
        public int getValue() { return mValue; }
        /** @return The {@link DeliveryStatus} by associated integer value. */
        public static DeliveryStatus fromInt(int value)
        {
            for (DeliveryStatus status : values()) {
                if (status.getValue() == value)
                    return status;
            }
            throw new IllegalStateException("Invalid value " + value + " for DeliveryReceipt.DeliveryStatus");
        }
    }

    private final String mSid;
    private final String mMessageSid;
    private final String mConversationSid;
    private final String mChannelMessageSid;
    private final String mParticipantSid;
    private final DeliveryStatus mStatus;
    private final int mErrorCode;
    private final Date mDateCreated;
    private final Date mDateUpdated;

    /** @internal */
    DetailedDeliveryReceipt(String sid,
                            String messageSid,
                            String conversationSid,
                            String channelMessageSid,
                            String participantSid,
                            DeliveryStatus status,
                            int errorCode,
                            Date dateCreated,
                            Date dateUpdated) {
        mSid = sid;
        mMessageSid = messageSid;
        mConversationSid = conversationSid;
        mChannelMessageSid = channelMessageSid;
        mParticipantSid = participantSid;
        mStatus = status;
        mErrorCode = errorCode;
        mDateCreated = dateCreated;
        mDateUpdated = dateUpdated;
    }

    /**
     * Get unique identifier for this {@link DetailedDeliveryReceipt}.
     *
     * @return The {@link DetailedDeliveryReceipt} SID.
     */
    public String getSid() {
        return mSid;
    }

    /**
     * Get unique identifier of {@link Message} which this {@link DetailedDeliveryReceipt} is for.
     *
     * @return The {@link Message} SID.
     */
    public String getMessageSid() {
        return mMessageSid;
    }

    /**
     * Get unique identifier of {@link Conversation} which this {@link DetailedDeliveryReceipt} is for.
     *
     * @return The {@link Conversation} SID.
     */
    public String getConversationSid() {
        return mConversationSid;
    }

    /**
     * A messaging channel-specific identifier for the message delivered to participant
     * e.g. SMxx for text messages and MMxx for media messages etc.
     *
     * @return SID of the ‘channel’ message.
     */
    public String getChannelMessageSid() {
        return mChannelMessageSid;
    }

    /**
     * Get unique identifier of {@link Participant} which this {@link DetailedDeliveryReceipt} is for.
     *
     * @return The {@link Participant} SID.
     */
    public String getParticipantSid() {
        return mParticipantSid;
    }

    /**
     * Get {@link DeliveryStatus} of the {@link Message} to the {@link Participant}.
     *
     * @return The {@link DeliveryStatus}.
     */
    public DeliveryStatus getStatus() {
        return mStatus;
    }

    /**
     * Get error code which points to a detailed description of the reason why the {@link Message} 
     * cannot be delivered to the {@link Participant}.
     *
     * @return The error code of delivery operation.
     */
    public int getErrorCode() {
        return mErrorCode;
    }

    /**
     * The creation date for this {@link DetailedDeliveryReceipt}.
     *
     * @return The timestamp when this {@link DetailedDeliveryReceipt} was created.
     */
    public Date getDateCreatedAsDate() {
        return mDateCreated;
    }

    /**
     * The timestamp when this {@link DetailedDeliveryReceipt} was updated last time.
     *
     * @return Most recent date when this {@link DetailedDeliveryReceipt} was updated or null.
     */
    public Date getDateUpdatedAsDate() {
        return mDateUpdated;
    }

    /** @hide */
    @Override
    public int hashCode() {
        int result = mSid != null ? mSid.hashCode() : 0;
        result = 31 * result + (mMessageSid != null ? mMessageSid.hashCode() : 0);
        result = 31 * result + (mConversationSid != null ? mConversationSid.hashCode() : 0);
        result = 31 * result + (mChannelMessageSid != null ? mChannelMessageSid.hashCode() : 0);
        result = 31 * result + (mParticipantSid != null ? mParticipantSid.hashCode() : 0);
        result = 31 * result + mErrorCode;
        result = 31 * result + (mDateCreated != null ? mDateCreated.hashCode() : 0);
        result = 31 * result + (mDateUpdated != null ? mDateUpdated.hashCode() : 0);
        return result;
    }

    /** @hide */
    @Override
    public boolean equals(@Nullable Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof DetailedDeliveryReceipt)) {
            return false;
        }

        DetailedDeliveryReceipt other = (DetailedDeliveryReceipt) obj;
        return (mSid == other.mSid || (mSid != null && mSid.equals(other.mSid)))
                && (mMessageSid == other.mMessageSid || (mMessageSid != null && mMessageSid.equals(other.mMessageSid)))
                && (mConversationSid == other.mConversationSid || (mConversationSid != null && mConversationSid.equals(other.mConversationSid)))
                && (mChannelMessageSid == other.mChannelMessageSid || (mChannelMessageSid != null && mChannelMessageSid.equals(other.mChannelMessageSid)))
                && (mParticipantSid == other.mParticipantSid || (mParticipantSid != null && mParticipantSid.equals(other.mParticipantSid)))
                && (mErrorCode == other.mErrorCode)
                && (mDateCreated == other.mDateCreated || (mDateCreated != null && mDateCreated.equals(other.mDateCreated)))
                && (mDateUpdated == other.mDateUpdated || (mDateUpdated != null && mDateUpdated.equals(other.mDateUpdated)));
    }

    /** @hide */
    @NonNull
    @Override
    public String toString() {
        return "DeliveryReceipt(" +
                "\nsid: " + mSid +
                "\nmessageSid: " + mMessageSid +
                "\nconversationSid: " + mConversationSid +
                "\nchannelSid: " + mChannelMessageSid +
                "\nparticipantSid: " + mParticipantSid +
                "\nstatus: " + mStatus +
                "\nerrorCode: " + mErrorCode +
                "\ndateCreated: " + mDateCreated +
                "\ndateUpdated: " + mDateUpdated +
                "\n)";
    }
}
