//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.twilio.conversations.content.ContentData;
import com.twilio.conversations.content.ContentTemplate;

import org.jetbrains.annotations.NotNull;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Representation of a Conversations Message object.
 */
public interface Message
{
    /**
     * Indicates reason for message update.
     */
    enum UpdateReason
    {
        /** Message body has been updated. */
        BODY(0),
        /** Message attributes have been updated. */
        ATTRIBUTES(1),
        /** Message aggregated delivery receipt has been updated. */
        DELIVERY_RECEIPT(2),
        /** Message subject has been updated. */
        SUBJECT(3);

        private final int value;
        private UpdateReason(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link UpdateReason} by associated integer value. */
        public static UpdateReason fromInt(int value)
        {
            for (UpdateReason t : UpdateReason.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for Message.UpdateReason");
        }
    }

    /**@name Getters */
    /**@{*/

    /**
     * Returns the identifier for this message.
     *
     * @return The message SID.
     */
    String getSid();

    /**
     * The global identity of the author of this message.
     *
     * @return The global identity.
     */
    String getAuthor();

    /**
     * The subject for this message.
     *
     * @return The message subject or null if message has no subject.
     */
    @Nullable
    String getSubject();

    /**
     * The creation date for this message.
     *
     * @return The timestamp when this message was created.
     */
    String getDateCreated();

    /**
     * The creation timestamp for this message.
     *
     * @return Date when this message was created or null if date string could not be parsed.
     */
    Date getDateCreatedAsDate();

    /**
     * The timestamp when this message was updated last time.
     *
     * @return Most recent timestamp when this message was updated.
     */
    String getDateUpdated();

    /**
     * The {@link Date} when this message was updated last time.
     *
     * @return Most recent {@link Date} when this message was updated or null.
     */
    Date getDateUpdatedAsDate();

    /**
     * Returns the identity of the user who updated the message.
     */
    @Nullable
    String getLastUpdatedBy();

    /**
     * The body for this message.
     *
     *  For an email the body will contain the preview for the email.
     *  To get full email body call {@link #getEmailBody}.
     *
     * @return The message body.
     */
    String getBody();

    /**
     * Returns unique identifier for {@link ContentTemplate} for this message.
     */
    @Nullable
    String getContentSid();

    /**
     * Returns {@link ContentData} for this message when {@link #getContentSid()} is not `null`, otherwise calls
     * {@link CallbackListener#onError}.
     *
     */
    CancellationToken getContentData(@NonNull CallbackListener<ContentData> listener);

    /**@}*/
    /**@name Setters */
    /**@{*/

    /**
     * Updates the body for a message.
     *
     * @param body The body for the message.
     * @param listener Listener that will receive callback of the result.
     */
    void updateBody(String body, StatusListener listener);

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * Returns the conversation SID of the conversation this message belongs to.
     *
     * @return Conversation SID.
     */
    String getConversationSid();

    /**
     * Returns the parent conversation this message belongs to.
     *
     * @return Conversation.
     */
    Conversation getConversation();

    /**
     * Returns the participant SID of the participant this message was sent by.
     *
     * @return Participant SID.
     */
    String getParticipantSid();

    /**
     * Returns the participant this message was sent by.
     *
     * @return Participant.
     */
    Participant getParticipant();

    /**
     * Returns the index number for this message.
     * <p>
     * By design of the conversations system the message indices may have arbitrary gaps between them,
     * that does not necessarily mean they were deleted or otherwise modified - just that
     * messages may have non-contiguous indices even if they are sent immediately one after another.
     * <p>
     * Trying to use indices for some calculations is going to be unreliable.
     * <p>
     * To calculate the number of unread messages it is better to use the read horizon API.
     * See {@link Conversation#getUnreadMessagesCount} for details.
     *
     * @return the message index.
     */
    long getMessageIndex();

    /**
     * Retrieve attributes associated with this message.
     *
     * @return the message {@link Attributes}.
     */
    @NonNull
    Attributes getAttributes();

    /**@}*/
    /**@name Setters */
    /**@{*/

    /**
     * Set attributes associated with this message.
     *
     * @param attributes The new developer-provided {@link Attributes} for this message.
     * @param listener Listener that will receive callback with the result.
     */
    void setAttributes(@NonNull Attributes attributes, StatusListener listener);

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * Returns {@link AggregatedDeliveryReceipt} for the message.
     */
    AggregatedDeliveryReceipt getAggregatedDeliveryReceipt();

    /**
     * Returns list of {@link DetailedDeliveryReceipt} per each {@link Participant} for the {@link Message}.
     */
    void getDetailedDeliveryReceiptList(CallbackListener<List<DetailedDeliveryReceipt>> listener);

    /**@}*/
    /**@name Media */
    /**@{*/

    /**
     * Returns a (possibly empty) List of media matching a specific set of categories.
     *
     * @param categories Set of categories to match.
     * @return List of media descriptors matching given categories.
     */
    @NotNull
    List<Media> getMediaByCategories(@NonNull Set<MediaCategory> categories);

    /**
     * Returns all media attachments, except email body/history attachments.
     */
    @NotNull
    List<Media> getAttachedMedia();

    /**
     * Returns email body attachment of specific content type.
     *
     * @param contentType Type to match.
     * @return Email body attachment or null if message has no email body for the content type.
     */
    @Nullable
    Media getEmailBody(@NonNull String contentType);

    /**
     * Returns email body attachment for the "text/plain" content type.
     *
     * @return Email body attachment or null if message has no email body for the "text/plain" type.
     */
    @Nullable
    Media getEmailBody();

    /**
     * Returns email history attachment of specific content type.
     *
     * @param contentType Type to match.
     * @return Email history attachment or null if message has no email history for the content type.
     */
    @Nullable
    Media getEmailHistory(@NonNull String contentType);

    /**
     * Returns email history attachment for the "text/plain" content type.
     *
     * @return Email history attachment or null if message has no email body for the "text/plain" type.
     */
    @Nullable
    Media getEmailHistory();

    /**
     * Get content URLs for all media attachments in the given set using single network request.
     *
     * NB. It is a convenience method originated in the conversations client.
     *
     * @param media    List of media attachments to query for content URL.
     * @param listener Listener to receive Map of mediaSID to content temporary URL.
     * @return {@link CancellationToken} which allows to cancel network request.
     */
    @NonNull
    CancellationToken getTemporaryContentUrlsForMedia(@NonNull List<Media> media,
                                                      @NonNull CallbackListener<Map<String, String>> listener);

    /**
     * Get content URLs for all media attachments in the given set using single network request.
     *
     * NB. It is a convenience method originated in the conversations client.
     *
     * @param mediaSids List of media sids to query for content URL.
     * @param listener  Listener to receive Map of mediaSID to content temporary URL.
     * @return {@link CancellationToken} which allows to cancel network request.
     */
    @NonNull
    CancellationToken getTemporaryContentUrlsForMediaSids(@NonNull List<String> mediaSids,
                                                          @NonNull CallbackListener<Map<String, String>> listener);

    /**
     * Get content URLs for all attached media returned by {@link #getAttachedMedia()} using single network request.
     *
     * @param listener  Listener to receive Map of mediaSID to content temporary URL.
     * @return {@link CancellationToken} which allows to cancel network request.
     */
    @NonNull
    CancellationToken getTemporaryContentUrlsForAttachedMedia(@NonNull CallbackListener<Map<String, String>> listener);

    /**@}*/
}
