//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import android.os.Bundle;
import com.twilio.util.TwilioLogger;
import java.util.Map;
import kotlin.jvm.JvmClassMappingKt;
import org.jetbrains.annotations.NotNull;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Helper accessor for notification data payload bundle as received from Twilio Notifications.
 * Use in <code>FirebaseMessagingService.onMessageReceived</code> implementation to retrieve Twilio-generated
 * payload fields safely.
 * <p>
 * Do not access the fields in the bundle directly - they could be modified without notice, always
 * use this helper to access them.
 */
public class NotificationPayload
{
    private static final TwilioLogger logger = TwilioLogger.getLogger(
        JvmClassMappingKt.getKotlinClass(NotificationPayload.class));

    // Using -1 as invalid message_index as 0 is valid value
    // See: https://docs.google.com/document/d/1t_gd-JY4SVIR7w24qh-Cs0BtWDZrkzTYhIniFGgrRuE/edit#

    /** Value returned by {@link #getMessageIndex} if notification contains no message index. */
    public static final long INVALID_MESSAGE_INDEX = -1;

    /**
     * Represents payload type.
     * <p>
     * Payload types are defined by Twilio Conversations. When you receive a new push notification
     * you could check if it is coming from Twilio Conversations.
     */
    public enum Type {
        /** Not a Twilio payload. */
        UNKNOWN(0),
        /** Notification arrives when a new message is added to a conversation. */
        NEW_MESSAGE(1),
        /** Notification arrives when current user is added to a conversation. */
        ADDED_TO_CONVERSATION(3),
        /** Notification arrives when current user is removed from a conversation. */
        REMOVED_FROM_CONVERSATION(4);

        private final int value;
        private Type(int value) { this.value = value; }
        /** @internal */
        public static Type fromString(String in) {
            if (in.contentEquals("twilio.conversations.new_message"))
                return NEW_MESSAGE;
            if (in.contentEquals("twilio.conversations.added_to_conversation"))
                return ADDED_TO_CONVERSATION;
            if (in.contentEquals("twilio.conversations.removed_from_conversation"))
                return REMOVED_FROM_CONVERSATION;
            logger.w("Unsupported notification type: " + in);
            return UNKNOWN;
        }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link Type} by associated integer value. */
        public static Type fromInt(int value)
        {
            for (Type item : values()) {
                if (item.getValue() == value)
                    return item;
            }
            logger.e("Invalid value " + value + " for NotificationPayload.Type");
            throw new IllegalStateException("Invalid value " + value + " for NotificationPayload.Type");
        }
    }

    /**
     * Create notification payload from the received map.
     * <p>
     * <b>Example:</b>
     * <pre><code>
     public class FCMListenerService extends FirebaseMessagingService {
        @Override
        public void onMessageReceived(RemoteMessage remoteMessage) {
            NotificationPayload payload = new NotificationPayload(remoteMessage.getData());
            conversationsClient.handleNotification(payload);
            // Do the processing here
        }
     }
     * </code><pre>
     * @param  remoteMessage Android FCM RemoteMessage with push notification data as received
     *                       from the system.
     */
    public NotificationPayload(Map<String, String> remoteMessage) {
        JSONObject obj = new JSONObject(remoteMessage);
        Bundle data = new Bundle();

        copyStringToBundle(obj, "twi_message_type", data);
        copyStringToBundle(obj, "twi_body", data);
        copyStringToBundle(obj, "twi_sound", data);
        copyStringToBundle(obj, "twi_message_id", data); // unused
        copyStringToBundle(obj, "conversation_sid", data);
        copyStringToBundle(obj, "conversation_title", data);
        copyStringToBundle(obj, "author", data);
        copyStringToBundle(obj, "message_sid", data);
        copyNonNegativeNumberToBundle(obj, "message_index", data);

        copyNonNegativeNumberToBundle(obj, "media_count", data);
        String media = optString(obj, "media");
        if (media != null && !media.isEmpty()) {
            try {
                JSONObject objMedia = new JSONObject(media);
                copyStringToBundle(objMedia, "sid", data, "media_sid");
                copyStringToBundle(objMedia, "filename", data, "media_filename");
                copyStringToBundle(objMedia, "content_type", data, "media_content_type");
                copyNonNegativeNumberToBundle(objMedia, "size", data, "media_size");
            } catch (JSONException e) {
                logger.i("Media notification parsing exception: " + e.toString());
            }
        }

        payload = data;
    }

    /**@name Getters */
    /**@{*/

    /**
     * Get notification type.
     *
     * @return received notification type as a Type enum.
     */
    public Type getType() {
        if (payload.containsKey("twi_message_type")) {
            return Type.fromString(payload.getString("twi_message_type"));
        }
        return Type.UNKNOWN;
    }

    /**
     * Get notification body.
     *
     * @return Notification payload as configured by the push notification template in service.
     */
    public String getBody() {
        return payload.getString("twi_body", "");
    }

    /**
     * Get notification sound.
     *
     * @return Sound name as configured by the push notification configuration in service,
     *         empty string if not configured.
     */
    public String getSound() {
        return payload.getString("twi_sound", "");
    }

    /**
     * Get notification conversation title.
     *
     * @return Conversation title if available, empty string otherwise.
     */
    public String getConversationTitle() {
        return payload.getString("conversation_title", "");
    }

    /**
     * Get notification conversation SID.
     *
     * @return Conversation SID if available, empty string otherwise.
     */
    public String getConversationSid() {
        return payload.getString("conversation_sid", "");
    }
 
    /**
     * Get message author. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Message author for NEW_MESSAGE notification type, empty string otherwise.
     */
    public String getAuthor() {
        return payload.getString("author", "");
    }

    /**
     * Get message SID. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Message SID for NEW_MESSAGE notification type, empty string otherwise.
     */
    public String getMessageSid() {
        return payload.getString("message_sid", "");
    }

    /**
     * Get message index. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Message index for NEW_MESSAGE notification type, {@link #INVALID_MESSAGE_INDEX} otherwise.
     */
    public long getMessageIndex() {
        return payload.getLong("message_index", INVALID_MESSAGE_INDEX);
    }

    /**
     * Get number of media attachments. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Number of attached media.
     */
    public long getMediaCount() {
        return payload.getLong("media_count", 0);
    }

    /**
     * Get media size. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Media attachment size if media was attached, zero otherwise.
     * In case of multiple attachments, return the size of the first one.
     */
    public long getMediaSize() {
        return payload.getLong("media_size", 0);
    }

    /**
     * Get media SID. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Media SID if media was attached, empty string otherwise.
     * In case of multiple attachments, return SID of the first one.
     */
    public String getMediaSid() {
        return payload.getString("media_sid", "");
    }

    /**
     * Get media filename. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Media filename if media was attached, empty string otherwise.
     * In case of multiple attachments, return filename of the first one.
     */
    public String getMediaFilename() {
        return payload.getString("media_filename", "");
    }

    /**
     * Get media content type. <i>Applicable for NEW_MESSAGE notification type only.</i>
     *
     * @return Media content type if media was attached, empty string otherwise.
     * In case of multiple attachments, return content type of the first one.
     */
    public String getMediaContentType() {
        return payload.getString("media_content_type", "");
    }

    /**@}*/

    Bundle payload;

    private static String optString(JSONObject srcJson, @NotNull String key) {
        // JSONObject returns "null" for null.
        // Fix that issue there.
        return srcJson.isNull(key) ? null : srcJson.optString(key);
    }

    private static void copyStringToBundle(JSONObject srcJson, @NotNull String key, Bundle destBundle) {
        copyStringToBundle(srcJson, key, destBundle, key);
    }

    private static void copyStringToBundle(JSONObject srcJson, @NotNull String srcKey, Bundle destBundle, @NotNull String destKey) {
        String value = optString(srcJson, srcKey);
        if (value != null)
            destBundle.putString(destKey, value);
    }

    private static void copyNonNegativeNumberToBundle(JSONObject srcJson, @NotNull String key, Bundle destBundle) {
        copyNonNegativeNumberToBundle(srcJson, key, destBundle, key);
    }

    private static void copyNonNegativeNumberToBundle(JSONObject srcJson, @NotNull String srcKey, Bundle destBundle, @NotNull String destKey) {
        long value = srcJson.optLong(srcKey, -1);
        if (value >= 0)
            destBundle.putLong(destKey, value);
    }
}
