//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import androidx.annotation.NonNull;

import java.util.Date;

/**
 * Representation of a Conversation Participant object.
 */
public interface Participant
{
    /**
     * Indicates reason for participant info update.
     */
    enum UpdateReason
    {
        /**
         * Participant last read message index has changed.
         * <p>
         * This update event is fired when participant's read horizon changes. This usually
         * indicates that some messages were read by that participant.
         */
        LAST_READ_MESSAGE_INDEX(0),
        /**
         * Participant last read message timestamp has changed.
         * <p>
         * This update event is fired when participant's read horizon changes (or just set to the same position again).
         * This usually indicates that some messages were read by that participant.
         */
        LAST_READ_TIMESTAMP(1),
        /**
         * Participant attributes have changed.
         * <p>
         * This update event is fired when participant's attributes change.
         */
        ATTRIBUTES(2);

        private final int value;
        private UpdateReason(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link UpdateReason} by associated integer value. */
        public static UpdateReason fromInt(int value)
        {
            for (UpdateReason t : UpdateReason.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for UpdateReason");
        }
    }

    /**@name Getters */
    /**@{*/

    /**
     * Returns unique identifier of a participant on a conversation.
     *
     * @return The unique id of a participant.
     */
    String getSid();

    /**
     * Return participant's last read message index for this conversation
     *
     * @return read message index. Null if not set
     * @see #getLastReadTimestamp
     */
    Long getLastReadMessageIndex();

    /**
     * Return participant's last read message timestamp for this conversation
     *
     * @return read message timestamp. Empty string if read index is not set
     * @see #getLastReadMessageIndex
     */
    String getLastReadTimestamp();

    /**
     * Get creation date of the participant as
     * an <a href="http://www.iso.org/iso/home/standards/iso8601.htm">ISO 8601</a> string.
     *
     * @return Date when participant was created as a string in ISO 8601 format.
     */
    String getDateCreated();

    /**
     * Get creation date of the participant.
     *
     * @return Date when participant was created or null if date string could not be parsed.
     */
    Date getDateCreatedAsDate();

    /**
     * Get update date of the participant as
     * an <a href="http://www.iso.org/iso/home/standards/iso8601.htm">ISO 8601</a> string.
     *
     * @return Date when participant was last updated as a string in ISO 8601 format.
     */
    String getDateUpdated();

    /**
     * Get update date of the participant.
     *
     * @return Date when participant was last updated or null if date string could not be parsed.
     */
    Date getDateUpdatedAsDate();

    /**
     * Return conversation this participant belongs to.
     *
     * @return Conversation for this participant.
     */
    Conversation getConversation();

    /**
     * Return user identity for current participant
     *
     * @return User identity of current participant.
     */
    String getIdentity();

    /**
     * Return channel of current participant (chat, sms, whatsapp, etc...)
     *
     * @return Return channel of current participant (chat, sms, whatsapp, etc...)
     */
    String getChannel();

    /**
     * Custom attributes associated with this participant.
     *
     * @return the participant {@link Attributes}.
     */
    @NonNull
    Attributes getAttributes();

    /**@}*/
    /**@name Setters */
    /**@{*/

    /**
     * Set attributes associated with this participant.
     *
     * @param attributes The new developer-provided {@link Attributes} for this participant.
     * @param listener Listener that will receive callback with the result.
     */
    void setAttributes(@NonNull Attributes attributes, StatusListener listener);

    /**@}*/
    /**@name Actions */
    /**@{*/

    /**
     * Return subscribed user object for current participant
     *
     * @param listener Listener to receive the resulting User or error.
     */
    void getAndSubscribeUser(CallbackListener<User> listener);

    /**
     * Remove this participant from conversation
     *
     * @param listener StatusListener to report status.
     */
    void remove(StatusListener listener);

    /**@}*/
}
