//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations;

import androidx.annotation.NonNull;

/**
 * User info object
 */
public interface User
{

    /**
     * Indicates reason for user info update.
     */
    enum UpdateReason
    {
        /** Friendly name changed. */
        FRIENDLY_NAME(0),
        /** Attributes changed. */
        ATTRIBUTES(1),
        /** Online status changed. */
        REACHABILITY_ONLINE(2),
        /** Notifications reachability changed. */
        REACHABILITY_NOTIFIABLE(3);

        private final int value;
        private UpdateReason(int value) { this.value = value; }

        /** @return The associated integer value. */
        public int getValue() { return value; }
        /** @return The {@link UpdateReason} by associated integer value. */
        public static UpdateReason fromInt(int value)
        {
            for (UpdateReason t : UpdateReason.values()) {
                if (t.getValue() == value)
                    return t;
            }
            throw new IllegalStateException("Invalid value " + value + " for UpdateReason");
        }
    }

    /**@name Getters */
    /**@{*/

    /**
     * Method that returns the friendlyName from the user info.
     *
     * @return the friendlyName.
     */
    String getFriendlyName();

    /**@}*/
    /**@name Setters */
    /**@{*/

    /**
     * Updates the friendly name for this user info.
     *
     * @param friendlyName updated friendly name.
     * @param listener Listener that will receive callback with the result.
     */
    void setFriendlyName(String friendlyName, StatusListener listener);

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * Custom attributes associated with this user.
     *
     * @return the user {@link Attributes}.
     */
    @NonNull
    Attributes getAttributes();

    /**@}*/
    /**@name Setters */
    /**@{*/

    /**
     * Set attributes associated with this user.
     *
     * @param attributes The new developer-provided {@link Attributes} for this user.
     * @param listener Listener that will receive callback with the result.
     */
    void setAttributes(@NonNull Attributes attributes, StatusListener listener);

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * Returns the identity of the user.
     *
     * @return the identity.
     */
    String getIdentity();

    /**
     * Return user's online status, if available.
     *
     * Note that if {@link ConversationsClient#isReachabilityEnabled} indicates reachability is disabled,
     * this will return false.
     *
     * @return true if user is online, false otherwise.
     */
    boolean isOnline();

    /**
     * Return user's push reachability.
     *
     * Note that if {@link ConversationsClient#isReachabilityEnabled} indicates reachability is disabled,
     * this will return false.
     *
     * @return true if user is reachable via push, false otherwise.
     */
    boolean isNotifiable();

    /**@}*/
    /**@name Actions */
    /**@{*/

    /**
     * Unsubscribe from this user, freeing up resources.
     * User will no longer receive real-time status updates.
     */
    void unsubscribe();

    /**@}*/
    /**@name Getters */
    /**@{*/

    /**
     * Check if this user receives real-time status updates.
     *
     * @return true if user is subscribed, false otherwise.
     */
    boolean isSubscribed();

    /**@}*/
}
