//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations.content

import com.twilio.conversations.ConversationsClient
import java.util.*
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

/**
 * Rich content template.
 *
 * Use [ConversationsClient.getContentTemplates] to request all content templates for current account.
 */
@Serializable
data class ContentTemplate(
    /** The server-assigned unique identifier for ContentTemplate. */
    @SerialName("sid") val sid: String,

    /** String name used to describe the Content. Not visible to the recipient. */
    @SerialName("friendly_name") val friendlyName: String,

    /** The timestamp when this template was created. */
    @Serializable(with = DateAsTextSerializer::class)
    @SerialName("date_created") val dateCreated: Date,

    /** The timestamp when this template was updated last time. */
    @Serializable(with = DateAsTextSerializer::class)
    @SerialName("date_updated") val dateUpdated: Date,

    /** Declare variables and define default placeholder values for variables included in content. */
    @Serializable(with = ContentVariablesListAsTextSerializer::class)
    @SerialName("variables") val variables: List<ContentTemplateVariable>,

    /** Variants of the content. See [ContentData] definition. */
    @Serializable(with = ContentVariantsSerializer::class)
    @SerialName("variants") val variants: Map<String, ContentData>,
)

/**
 * Represents rich content template variable.
 * @see ContentTemplate.variables
 */
@Serializable
data class ContentTemplateVariable(
    /** Variable name. */
    val name: String,

    /** Variable value. */
    val value: String,
)

/**
 * Represents one data type in rich content template.
 * For more details see: https://www.twilio.com/docs/content-api/content-types-overview
 */
sealed interface ContentData {
    /**
     * Full data as stringified JSON. Can be used for new content types and fields which
     * are not supported in the current version of Conversations SDK.
     */
    val rawData: String

    /** Contains only plain text-based content. Represents the twilio/text content type. */
    @Serializable
    data class Text(
        /** The text of the message you want to send. */
        val body: String,

        override val rawData: String = "",
    ) : ContentData

    /**
     * Used to send file attachments, or to send long text via MMS in the US and Canada.
     * Represents the twilio/media content type.
     */
    @Serializable
    data class Media(
        /** The text of the message you want to send. */
        val body: String? = null,

        /** URLs of the media you want to send. */
        val media: List<String>,

        override val rawData: String = "",
    ) : ContentData

    /**
     * Contains a location pin and an optional label, which can be used to enhance delivery
     * notifications or connect recipients to physical experiences you offer.
     * Represents the twilio/location content type.
     */
    @Serializable
    data class Location(
        /** The longitude value of the location pin you want to send. */
        val longitude: Double,

        /** The latitude value of the location pin you want to send. */
        val latitude: Double,

        /** Label to be displayed to the end user alongside the location pin. */
        val label: String? = null,

        override val rawData: String = "",
    ) : ContentData

    /**
     * Let recipients tap, rather than type, to respond to the message.
     * Represents the twilio/quick-reply content type.
     */
    @Serializable
    data class QuickReply(
        /** The text of the message you want to send. This is included as a regular text message. */
        val body: String,

        /** Up to 3 buttons can be created for Quick Reply buttons. See [Reply]. */
        @SerialName("actions")
        val replies: List<Reply>,

        override val rawData: String = "",
    ) : ContentData

    /** Shows a button that sends back a predefined text. Used in [QuickReply.replies]. */
    @Serializable
    data class Reply(
        /**
         * Display value for the action.
         * This is the message that will be sent back when the user taps on the button.
         */
        val title: String,

        /** Postback payload. This field is not visible to the end user. */
        val id: String? = null
    )

    /**
     * Buttons let recipients tap to trigger actions such as launching a website or making a phone call.
     * Represents the twilio/call-to-action content type.
     */
    @Serializable
    data class CallToAction(
        /** The text of the message you want to send. This is included as a regular text message. */
        val body: String,

        /** Buttons that recipients can tap on to act on the message. */
        val actions: List<Action>,

        override val rawData: String = "",
    ) : ContentData

    /**
     * Includes a menu of up to 10 options, which offers a simple way for users to make a selection.
     * Represents the twilio/list-picker content type.
     */
    @Serializable
    data class ListPicker(
        /** The text of the message you want to send. This is rendered as the body of the message. */
        val body: String,

        /** Display value for the primary button. */
        val button: String,

        /** [ListItem] objects displayed in the list. */
        val items: List<ListItem>,

        override val rawData: String = "",
    ) : ContentData

    /** Represents one item in the [ListPicker]. */
    @Serializable
    data class ListItem(
        /** Unique item identifier. Not visible to the recipient. */
        val id: String,

        /** Display value for the item. */
        val item: String,

        /** Description of the item. */
        val description: String? = null,
    )

    /**
     * Includes a menu of up to 10 options, which offers a simple way for users to make a selection.
     * Represents the twilio/card content type.
     */
    @Serializable
    data class DataCard(
        /** Title of card. */
        val title: String,

        /** Subtitle of card. */
        val subtitle: String? = null,

        /** URLs of the media to send with the message. */
        val media: List<String> = emptyList(),

        /** Buttons that recipients can tap on to act on the message. */
        val actions: List<Action> = emptyList(),

        override val rawData: String = "",
    ) : ContentData

    /** The following parameters are used as an array in the [CallToAction.actions] and [DataCard.actions]. */
    @Serializable(with = ContentActionSerializer::class)
    sealed interface Action {
        /**
         * Full data as stringified JSON. Can be used for new action types and fields which
         * are not supported in the current version of Conversations SDK.
         */
        val rawData: String

        /** Shows a button that redirects recipient to predefined URL. */
        @Serializable
        data class Url(
            /** Display value for the action. */
            val title: String,

            /** URL to direct to when the recipient taps the button. */
            val url: String,

            override val rawData: String = ""
        ) : Action

        /** Shows a button that calls a phone number. */
        @Serializable
        data class Phone(
            /** Display value for the action. */
            val title: String,

            /** Phone number to call when the recipient taps the button. */
            val phone: String,

            override val rawData: String = ""
        ) : Action

        /** Shows a button that sends back a predefined text. */
        @Serializable
        data class Reply(
            /**
             * Display value for the action.
             * This is the message that will be sent back when the user taps on the button.
             */
            val title: String,

            /** Postback payload. This field is not visible to the end user. */
            val id: String? = null,

            /** Index for the action. */
            val index: Long,

            override val rawData: String = ""
        ) : Action

        /** Used for unknown action types, which aren't present in the current version of Conversations SDK. */
        data class Other(override val rawData: String = "") : Action
    }

    /** Used for unknown content types, which aren't present in the current version of Conversations SDK. */
    data class ContentDataRaw(override val rawData: String) : ContentData
}
