//
//  Twilio Conversations Client
//
//  Copyright © Twilio, Inc. All rights reserved.
//
package com.twilio.conversations.extensions

import android.content.Context
import com.twilio.conversations.Attributes
import com.twilio.conversations.CallbackListener
import com.twilio.conversations.Conversation
import com.twilio.conversations.ConversationListener
import com.twilio.conversations.ConversationsClient
import com.twilio.conversations.ConversationsClient.FCMToken
import com.twilio.conversations.ConversationsClientListener
import com.twilio.conversations.DetailedDeliveryReceipt
import com.twilio.conversations.Media
import com.twilio.conversations.MediaUploadListener
import com.twilio.conversations.MediaUploadListenerBuilder
import com.twilio.conversations.Message
import com.twilio.conversations.Participant
import com.twilio.conversations.StatusListener
import com.twilio.conversations.User
import com.twilio.conversations.content.ContentData
import com.twilio.conversations.content.ContentTemplate
import com.twilio.conversations.content.ContentTemplateVariable
import com.twilio.conversations.extensions.ChannelType.Chat
import com.twilio.conversations.extensions.ChannelType.Other
import com.twilio.conversations.extensions.ChannelType.Sms
import com.twilio.conversations.extensions.ChannelType.Unset
import com.twilio.conversations.extensions.ChannelType.Whatsapp
import com.twilio.util.ErrorInfo
import com.twilio.util.ErrorInfo.Companion.CONVERSATION_NOT_SYNCHRONIZED
import com.twilio.util.TwilioException
import java.io.InputStream
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.suspendCancellableCoroutine

suspend fun createConversationsClient(
        context: Context,
        token: String,
        properties: ConversationsClient.Properties = ConversationsClient.Properties.newBuilder().createProperties()
) = suspendCancellableCoroutine<ConversationsClient> { continuation ->

    ConversationsClient.create(context, token, properties, object : CallbackListener<ConversationsClient> {

        override fun onSuccess(result: ConversationsClient) = continuation.resume(result)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun createAndSyncConversationsClient(
        context: Context,
        token: String,
        synchronizationStatus: ConversationsClient.SynchronizationStatus = ConversationsClient.SynchronizationStatus.COMPLETED,
        properties: ConversationsClient.Properties = ConversationsClient.Properties.newBuilder().createProperties()): ConversationsClient {

    val client = createConversationsClient(context, token, properties)
    client.waitForSynchronization(synchronizationStatus)
    return client
}

private suspend fun ConversationsClient.waitForSynchronization(
        synchronizationStatus: ConversationsClient.SynchronizationStatus = ConversationsClient.SynchronizationStatus.COMPLETED) {

    val completable = CompletableDeferred<Unit>()

    val listener = addListener(
            onClientSynchronization = { status ->
                if (status == synchronizationStatus) {
                    completable.complete(Unit)
                }
            }
    )

    try {
        completable.await()
    } finally {
        removeListener(listener)
    }
}

suspend fun ConversationsClient.registerFCMToken(token: FCMToken) = suspendCancellableCoroutine<Unit> { continuation ->
    registerFCMToken(token, object : StatusListener {

        override fun onSuccess() {
            synchronized(continuation) {
                if (!continuation.isActive) return
                continuation.resume(Unit)
            }
        }

        override fun onError(errorInfo: ErrorInfo) {
            synchronized(continuation) {
                if (!continuation.isActive) return
                continuation.resumeWithException(TwilioException(errorInfo))
            }
        }
    })
}

suspend fun ConversationsClient.createConversation(friendlyName: String = ""): Conversation = suspendCancellableCoroutine { continuation ->
    createConversation(friendlyName, object : CallbackListener<Conversation> {

        override fun onSuccess(result: Conversation) = continuation.resume(result)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.waitForSynchronization(
        synchronizationStatus: Conversation.SynchronizationStatus = Conversation.SynchronizationStatus.ALL) {

    val complete = CompletableDeferred<Unit>()

    val listener = addListener(
            onSynchronizationChanged = { conversation ->
                synchronized<Unit>(complete) {
                    if (complete.isCompleted) return@addListener

                    if (conversation.synchronizationStatus == Conversation.SynchronizationStatus.FAILED) {
                        val errorInfo = ErrorInfo(
                                CONVERSATION_NOT_SYNCHRONIZED, "Conversation synchronization failed: ${conversation.sid}}")

                        complete.completeExceptionally(TwilioException(errorInfo))
                    } else if (conversation.synchronizationStatus.isAtLeast(synchronizationStatus)) {
                        complete.complete(Unit)
                    }
                }
            }
    )

    try {
        complete.await()
    } finally {
        removeListener(listener)
    }
}

suspend fun ConversationsClient.ConversationBuilder.build(): Conversation = suspendCancellableCoroutine { continuation ->
    build(object : CallbackListener<Conversation> {

        override fun onSuccess(result: Conversation) = continuation.resume(result)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun ConversationsClient.getConversation(sidOrUniqueName: String): Conversation = suspendCancellableCoroutine { continuation ->
    getConversation(sidOrUniqueName, object : CallbackListener<Conversation> {

        override fun onSuccess(result: Conversation) = continuation.resume(result)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun ConversationsClient.getTemporaryContentUrlsForMedia(media: List<Media>): Map<String, String> = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getTemporaryContentUrlsForMedia(
        media, CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun ConversationsClient.getTemporaryContentUrlsForMediaSids(mediaSids: List<String>): Map<String, String> = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getTemporaryContentUrlsForMediaSids(
        mediaSids, CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun ConversationsClient.getContentTemplates(): List<ContentTemplate> = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getContentTemplates(
        CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun Conversation.destroy(): Unit = suspendCancellableCoroutine { continuation ->
    destroy(object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.join(): Unit = suspendCancellableCoroutine { continuation ->
    join(object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.joinAndSync(synchronizationStatus: Conversation.SynchronizationStatus = Conversation.SynchronizationStatus.ALL) {
    join()
    waitForSynchronization(synchronizationStatus)
}

suspend fun Conversation.leave(): Unit = suspendCancellableCoroutine { continuation ->
    leave(object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.getUnreadMessagesCount(): Long? = suspendCancellableCoroutine { continuation ->
    getUnreadMessagesCount(object : CallbackListener<Long?> {

        override fun onSuccess(count: Long?) = continuation.resume(count)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.setAttributes(attributes: Attributes): Unit = suspendCancellableCoroutine { continuation ->
    setAttributes(attributes, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Message.setAttributes(attributes: Attributes): Unit = suspendCancellableCoroutine { continuation ->
    setAttributes(attributes, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Participant.setAttributes(attributes: Attributes): Unit = suspendCancellableCoroutine { continuation ->
    setAttributes(attributes, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Participant.getAndSubscribeUser(): User = suspendCancellableCoroutine { continuation ->
    getAndSubscribeUser(object : CallbackListener<User> {

        override fun onSuccess(user: User) = continuation.resume(user)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Participant.remove(): Unit = suspendCancellableCoroutine { continuation ->
    remove(object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

sealed class ChannelType(internal val value: String?) {
    object Unset : ChannelType(null)
    object Chat : ChannelType("chat")
    object Sms : ChannelType("sms")
    object Whatsapp : ChannelType("whatsapp")
    data class Other(val channel: String) : ChannelType(channel)
}

// In order to avoid dependency on kotlin-reflect
private val channelTypes = listOf(Unset, Chat, Sms, Whatsapp).associateBy { it.value }

val Participant.channelType: ChannelType
    get() = channelTypes[channel] ?: Other(channel)

suspend fun User.setAttributes(attributes: Attributes): Unit = suspendCancellableCoroutine { continuation ->
    setAttributes(attributes, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun User.setFriendlyName(friendlyName: String?): Unit = suspendCancellableCoroutine { continuation ->
    setFriendlyName(friendlyName, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.getLastMessages(count: Int): MutableList<Message> = suspendCancellableCoroutine { continuation ->
    getLastMessages(count, object : CallbackListener<MutableList<Message>> {

        override fun onSuccess(messageList: MutableList<Message>) = continuation.resume(messageList)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.getMessagesBefore(index: Long, count: Int): MutableList<Message> = suspendCancellableCoroutine { continuation ->
    getMessagesBefore(index, count, object : CallbackListener<MutableList<Message>> {

        override fun onSuccess(messageList: MutableList<Message>) = continuation.resume(messageList)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.getMessagesAfter(index: Long, count: Int): MutableList<Message> = suspendCancellableCoroutine { continuation ->
    getMessagesAfter(index, count, object : CallbackListener<MutableList<Message>> {

        override fun onSuccess(messageList: MutableList<Message>) = continuation.resume(messageList)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.getMessageByIndex(index: Long): Message = suspendCancellableCoroutine { continuation ->
    getMessageByIndex(index, object : CallbackListener<Message> {

        override fun onSuccess(message: Message) = continuation.resume(message)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

class MessageBuilder(private val builder: Conversation.MessageBuilder) {

    var body: String? = null

    var subject: String? = null

    var attributes: Attributes = Attributes.DEFAULT

    var contentTemplateSid: String? = null

    var contentTemplateVariables: List<ContentTemplateVariable>? = null

    fun addMedia(inputStream: InputStream, contentType: String, filename: String? = null, listener: MediaUploadListener? = null) {
        builder.addMedia(inputStream, contentType, filename, listener)
    }

    inline fun addMedia(inputStream: InputStream, contentType: String, filename: String? = null, block: MediaUploadListenerBuilder.() -> Unit = {}) {
        val listener = MediaUploadListenerBuilder().apply(block).build()
        addMedia(inputStream, contentType, filename, listener)
    }

    fun addMedia(inputStream: InputStream, contentType: String, filename: String? = null) =
        addMedia(inputStream, contentType, filename, null)

    fun setEmailBody(emailBody: String, contentType: String) {
        builder.setEmailBody(emailBody, contentType)
    }

    fun setEmailBody(inputStream: InputStream, contentType: String, listener: MediaUploadListener? = null) {
        builder.setEmailBody(inputStream, contentType, listener)
    }


    inline fun setEmailBody(inputStream: InputStream, contentType: String, block: MediaUploadListenerBuilder.() -> Unit = {}) {
        val listener = MediaUploadListenerBuilder().apply(block).build()
        setEmailBody(inputStream, contentType, listener)
    }

    fun setEmailBody(inputStream: InputStream, contentType: String) =
        setEmailBody(inputStream, contentType, null)

    fun setEmailHistory(emailHistory: String, contentType: String) {
        builder.setEmailHistory(emailHistory, contentType)
    }

    fun setEmailHistory(inputStream: InputStream, contentType: String, listener: MediaUploadListener? = null) {
        builder.setEmailHistory(inputStream, contentType, listener)
    }

    inline fun setEmailHistory(inputStream: InputStream, contentType: String, block: MediaUploadListenerBuilder.() -> Unit = {}) {
        val listener = MediaUploadListenerBuilder().apply(block).build()
        setEmailHistory(inputStream, contentType, listener)
    }

    fun setEmailHistory(inputStream: InputStream, contentType: String) =
        setEmailHistory(inputStream, contentType, null)

    @PublishedApi
    internal fun build(): Conversation.UnsentMessage = builder
        .setBody(body)
        .setSubject(subject)
        .setAttributes(attributes)
        .setContentTemplate(contentTemplateSid, contentTemplateVariables)
        .build()
}

inline fun Conversation.prepareMessage(block: MessageBuilder.() -> Unit): Conversation.UnsentMessage =
    MessageBuilder(prepareMessage()).apply(block).build()

suspend inline fun Conversation.sendMessage(block: MessageBuilder.() -> Unit): Message = prepareMessage(block).send()

suspend fun Conversation.setLastReadMessageIndex(lastReadMessageIndex: Long): Long = suspendCancellableCoroutine { continuation ->
    setLastReadMessageIndex(lastReadMessageIndex, object : CallbackListener<Long> {

        override fun onSuccess(unreadMessagesCount: Long) = continuation.resume(unreadMessagesCount)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.advanceLastReadMessageIndex(lastReadMessageIndex: Long): Long = suspendCancellableCoroutine { continuation ->
    advanceLastReadMessageIndex(lastReadMessageIndex, object : CallbackListener<Long> {

        override fun onSuccess(unreadMessagesCount: Long) = continuation.resume(unreadMessagesCount)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.setAllMessagesRead(): Long = suspendCancellableCoroutine { continuation ->
    setAllMessagesRead(object : CallbackListener<Long> {

        override fun onSuccess(unreadMessagesCount: Long) = continuation.resume(unreadMessagesCount)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.setAllMessagesUnread(): Long? = suspendCancellableCoroutine { continuation ->
    setAllMessagesUnread(object : CallbackListener<Long?> {

        override fun onSuccess(unreadMessagesCount: Long?) = continuation.resume(unreadMessagesCount)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Message.updateMessageBody(body: String): Unit = suspendCancellableCoroutine { continuation ->
    updateBody(body, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Message.getDetailedDeliveryReceiptList(): List<DetailedDeliveryReceipt> = suspendCancellableCoroutine { continuation ->
    getDetailedDeliveryReceiptList(object : CallbackListener<List<DetailedDeliveryReceipt>> {

        override fun onSuccess(result: List<DetailedDeliveryReceipt>) = continuation.resume(result)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Message.getTemporaryContentUrlsForMedia(media: List<Media>): Map<String, String> = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getTemporaryContentUrlsForMedia(
        media, CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun Message.getTemporaryContentUrlsForMediaSids(mediaSids: List<String>): Map<String, String> = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getTemporaryContentUrlsForMediaSids(
        mediaSids, CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun Message.getTemporaryContentUrlsForAttachedMedia(): Map<String, String> = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getTemporaryContentUrlsForAttachedMedia(
        CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun Message.getContentData(): ContentData = suspendCancellableCoroutine { continuation ->
    val cancellationToken = getContentData(
        CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellationToken.cancel() }
}

suspend fun Conversation.addParticipantByIdentity(identity: String, attributes: Attributes = Attributes.DEFAULT): Unit = suspendCancellableCoroutine { continuation ->
    addParticipantByIdentity(identity, attributes, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.addParticipantByAddress(address: String, proxyAddress: String, attributes: Attributes = Attributes.DEFAULT): Unit = suspendCancellableCoroutine { continuation ->
    addParticipantByAddress(address, proxyAddress, attributes, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.removeParticipantByIdentity(identity: String): Unit = suspendCancellableCoroutine { continuation ->
    removeParticipantByIdentity(identity, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.removeParticipant(participant: Participant): Unit = suspendCancellableCoroutine { continuation ->
    removeParticipant(participant, object : StatusListener {

        override fun onSuccess() = continuation.resume(Unit)

        override fun onError(errorInfo: ErrorInfo) = continuation.resumeWithException(TwilioException(errorInfo))
    })
}

suspend fun Conversation.MessageBuilder.buildAndSend() = suspendCancellableCoroutine<Message> { continuation ->
    val cancellation = buildAndSend(
        CallbackListener<Message>(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellation.cancel() }
}

suspend fun Conversation.UnsentMessage.send() = suspendCancellableCoroutine<Message> { continuation ->
    val cancellation = send(
        CallbackListener<Message>(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellation.cancel() }
}

suspend fun Media.getTemporaryContentUrl() = suspendCancellableCoroutine<String> { continuation ->
    val cancellation = getTemporaryContentUrl(
        CallbackListener(
            onSuccess = { continuation.resume(it) },
            onError = { continuation.resumeWithException(TwilioException(it)) },
        )
    )

    continuation.invokeOnCancellation { cancellation.cancel() }
}

inline fun ConversationsClient.addListener(
    crossinline onConversationAdded: (conversation: Conversation) -> Unit = {},
    crossinline onConversationUpdated: (conversation: Conversation, reason: Conversation.UpdateReason) -> Unit = { _, _ -> },
    crossinline onConversationDeleted: (conversation: Conversation) -> Unit = {},
    crossinline onConversationSynchronizationChange: (conversation: Conversation) -> Unit = {},
    crossinline onError: (errorInfo: ErrorInfo) -> Unit = {},
    crossinline onUserUpdated: (user: User, reason: User.UpdateReason) -> Unit = { _, _ -> },
    crossinline onUserSubscribed: (user: User) -> Unit = {},
    crossinline onUserUnsubscribed: (user: User) -> Unit = {},
    crossinline onClientSynchronization: (status: ConversationsClient.SynchronizationStatus) -> Unit = {},
    crossinline onNewMessageNotification: (conversationSid: String, messageSid: String, messageIndex: Long) -> Unit = { _, _, _ -> },
    crossinline onAddedToConversationNotification: (conversationSid: String) -> Unit = {},
    crossinline onRemovedFromConversationNotification: (conversationSid: String) -> Unit = {},
    crossinline onNotificationSubscribed: () -> Unit = {},
    crossinline onNotificationFailed: (errorInfo: ErrorInfo) -> Unit = {},
    crossinline onConnectionStateChange: (state: ConversationsClient.ConnectionState) -> Unit = {},
    crossinline onTokenExpired: () -> Unit = {},
    crossinline onTokenAboutToExpire: () -> Unit = {}): ConversationsClientListener {

    val listener = ConversationsClientListener(
            onConversationAdded,
            onConversationUpdated,
            onConversationDeleted,
            onConversationSynchronizationChange,
            onError,
            onUserUpdated,
            onUserSubscribed,
            onUserUnsubscribed,
            onClientSynchronization,
            onNewMessageNotification,
            onAddedToConversationNotification,
            onRemovedFromConversationNotification,
            onNotificationSubscribed,
            onNotificationFailed,
            onConnectionStateChange,
            onTokenExpired,
            onTokenAboutToExpire)

    addListener(listener)
    return listener
}

inline fun ConversationsClientListener(
    crossinline onConversationAdded: (conversation: Conversation) -> Unit = {},
    crossinline onConversationUpdated: (conversation: Conversation, reason: Conversation.UpdateReason) -> Unit = { _, _ -> },
    crossinline onConversationDeleted: (conversation: Conversation) -> Unit = {},
    crossinline onConversationSynchronizationChange: (conversation: Conversation) -> Unit = {},
    crossinline onError: (errorInfo: ErrorInfo) -> Unit = {},
    crossinline onUserUpdated: (user: User, reason: User.UpdateReason) -> Unit = { _, _ -> },
    crossinline onUserSubscribed: (user: User) -> Unit = {},
    crossinline onUserUnsubscribed: (user: User) -> Unit = {},
    crossinline onClientSynchronization: (status: ConversationsClient.SynchronizationStatus) -> Unit = {},
    crossinline onNewMessageNotification: (conversationSid: String, messageSid: String, messageIndex: Long) -> Unit = { _, _, _ -> },
    crossinline onAddedToConversationNotification: (conversationSid: String) -> Unit = {},
    crossinline onRemovedFromConversationNotification: (conversationSid: String) -> Unit = {},
    crossinline onNotificationSubscribed: () -> Unit = {},
    crossinline onNotificationFailed: (errorInfo: ErrorInfo) -> Unit = {},
    crossinline onConnectionStateChange: (state: ConversationsClient.ConnectionState) -> Unit = {},
    crossinline onTokenExpired: () -> Unit = {},
    crossinline onTokenAboutToExpire: () -> Unit = {}
): ConversationsClientListener = object : ConversationsClientListener {

        override fun onConversationAdded(conversation: Conversation) = onConversationAdded(conversation)

        override fun onConversationUpdated(conversation: Conversation, reason: Conversation.UpdateReason) = onConversationUpdated(conversation, reason)

        override fun onConversationDeleted(conversation: Conversation) = onConversationDeleted(conversation)

        override fun onConversationSynchronizationChange(conversation: Conversation) = onConversationSynchronizationChange(conversation)

        override fun onError(errorInfo: ErrorInfo) = onError(errorInfo)

        override fun onUserUpdated(user: User, reason: User.UpdateReason) = onUserUpdated(user, reason)

        override fun onUserSubscribed(user: User) = onUserSubscribed(user)

        override fun onUserUnsubscribed(user: User) = onUserUnsubscribed(user)

        override fun onClientSynchronization(status: ConversationsClient.SynchronizationStatus) = onClientSynchronization(status)

        override fun onNewMessageNotification(conversationSid: String, messageSid: String, messageIndex: Long) = onNewMessageNotification(conversationSid, messageSid, messageIndex)

        override fun onAddedToConversationNotification(conversationSid: String) = onAddedToConversationNotification(conversationSid)

        override fun onRemovedFromConversationNotification(conversationSid: String) = onRemovedFromConversationNotification(conversationSid)

        override fun onNotificationSubscribed() = onNotificationSubscribed()

        override fun onNotificationFailed(errorInfo: ErrorInfo) = onNotificationFailed(errorInfo)

        override fun onConnectionStateChange(state: ConversationsClient.ConnectionState) = onConnectionStateChange(state)

        override fun onTokenExpired() = onTokenExpired()

        override fun onTokenAboutToExpire() = onTokenAboutToExpire()
}

inline fun Conversation.addListener(
    crossinline onMessageAdded: (message: Message) -> Unit = {},
    crossinline onMessageUpdated: (message: Message, reason: Message.UpdateReason) -> Unit = { _, _ -> },
    crossinline onMessageDeleted: (message: Message) -> Unit = {},
    crossinline onParticipantAdded: (participant: Participant) -> Unit = {},
    crossinline onParticipantUpdated: (participant: Participant, reason: Participant.UpdateReason) -> Unit = { _, _ -> },
    crossinline onParticipantDeleted: (participant: Participant) -> Unit = {},
    crossinline onTypingStarted: (conversation: Conversation, participant: Participant) -> Unit = { _, _ -> },
    crossinline onTypingEnded: (conversation: Conversation, participant: Participant) -> Unit = { _, _ -> },
    crossinline onSynchronizationChanged: (conversation: Conversation) -> Unit = {}): ConversationListener {

    val listener = ConversationListener(
            onMessageAdded,
            onMessageUpdated,
            onMessageDeleted,
            onParticipantAdded,
            onParticipantUpdated,
            onParticipantDeleted,
            onTypingStarted,
            onTypingEnded,
            onSynchronizationChanged
    )
    addListener(listener)
    return listener
}

inline fun ConversationListener(
    crossinline onMessageAdded: (message: Message) -> Unit = {},
    crossinline onMessageUpdated: (message: Message, reason: Message.UpdateReason) -> Unit = { _, _ -> },
    crossinline onMessageDeleted: (message: Message) -> Unit = {},
    crossinline onParticipantAdded: (participant: Participant) -> Unit = {},
    crossinline onParticipantUpdated: (participant: Participant, reason: Participant.UpdateReason) -> Unit = { _, _ -> },
    crossinline onParticipantDeleted: (participant: Participant) -> Unit = {},
    crossinline onTypingStarted: (conversation: Conversation, participant: Participant) -> Unit = { _, _ -> },
    crossinline onTypingEnded: (conversation: Conversation, participant: Participant) -> Unit = { _, _ -> },
    crossinline onSynchronizationChanged: (conversation: Conversation) -> Unit = {}
): ConversationListener = object : ConversationListener {

    override fun onMessageAdded(message: Message) = onMessageAdded(message)

    override fun onMessageUpdated(message: Message, reason: Message.UpdateReason) = onMessageUpdated(message, reason)

    override fun onMessageDeleted(message: Message) = onMessageDeleted(message)

    override fun onParticipantAdded(participant: Participant) = onParticipantAdded(participant)

    override fun onParticipantUpdated(participant: Participant, reason: Participant.UpdateReason) = onParticipantUpdated(participant, reason)

    override fun onParticipantDeleted(participant: Participant) = onParticipantDeleted(participant)

    override fun onTypingStarted(conversation: Conversation, participant: Participant) = onTypingStarted(conversation, participant)

    override fun onTypingEnded(conversation: Conversation, participant: Participant) = onTypingEnded(conversation, participant)

    override fun onSynchronizationChanged(conversation: Conversation) = onSynchronizationChanged(conversation)
}

inline fun <T> CallbackListener(
        crossinline onSuccess: (result: T) -> Unit = {},
        crossinline onError: (errorInfo: ErrorInfo) -> Unit = {}
): CallbackListener<T> = object : CallbackListener<T> {

    override fun onSuccess(result: T) = onSuccess(result)

    override fun onError(errorInfo: ErrorInfo) = onError(errorInfo)
}

inline fun StatusListener(
        crossinline onSuccess: () -> Unit = {},
        crossinline onError: (errorInfo: ErrorInfo) -> Unit = {}
): StatusListener = object : StatusListener {

    override fun onSuccess() = onSuccess()

    override fun onError(errorInfo: ErrorInfo) = onError(errorInfo)
}
