package com.twilio.video;

import android.content.Context;
import tvi.webrtc.CapturerObserver;
import tvi.webrtc.SurfaceTextureHelper;

/**
 * Generic video capturing interface that extends {@link tvi.webrtc.VideoCapturer}.
 *
 * <p>This interface provides an extension of the WebRTC video capturing interface. Implementations
 * of this interface or {@link tvi.webrtc.VideoCapturer} can be used when creating local video
 * tracks. As a result, callers can leverage capturers provided by the SDK, capturers provided by
 * WebRTC, or custom capturers.
 *
 * <p>The {@link LocalVideoTrack} API directly relates to the video capturer programming model. When
 * a caller creates a local video track, the following methods are called in the order listed.
 *
 * <p>
 *
 * <ol>
 *   <li>{@link tvi.webrtc.VideoCapturer#isScreencast()} - The return value is used to create a
 *       {@link tvi.webrtc.VideoSource}.
 *   <li>{@link tvi.webrtc.VideoCapturer#initialize(SurfaceTextureHelper, Context,
 *       CapturerObserver)} - This method is called so the video capturer can setup to capture
 *       frames. The video capturer should retain a reference to the capturer observer and prepare
 *       to forward frames. If the video capturer wants to deliver texture frames, it should do this
 *       by rendering on the SurfaceTexture provided by {@code surfaceTextureHelper}, register
 *       itself as a listener, and forward the frames to the capturer observer. This method will be
 *       called exactly once before {@link tvi.webrtc.VideoCapturer#startCapture(int, int, int)}.
 *       The local video track retains ownership of the {@code surfaceTextureHelper}.
 *   <li>{@link tvi.webrtc.VideoCapturer#startCapture(int, int, int)} - The video capturer should
 *       start capturing in a format as close as possible to {@code width x height} at {@code
 *       framerate}. The video capturer should capture and deliver frames to the capturer observer
 *       provided in initialize on a dedicated thread.
 * </ol>
 *
 * <p>When a caller releases a local video track, the following methods are called in the order
 * listed.
 *
 * <ol>
 *   <li>{@link tvi.webrtc.VideoCapturer#stopCapture()} - The video capturer should stop capturing
 *       frames. The SDK expects this method to block until frames have stopped being captured.
 *   <li>{@link tvi.webrtc.VideoCapturer#dispose()} - Perform final cleanup.
 * </ol>
 *
 * <p><strong>Threading Recommendations</strong>
 *
 * <p>Each of these methods are called on the same thread that creates and releases a local video
 * track which for most cases will be the main thread. As a result, video capturers should minimize
 * the amount of work done in each callback and delegate operations on a dedicated thread. A
 * recommended approach would be to create a capturing thread when {@link
 * tvi.webrtc.VideoCapturer#initialize(SurfaceTextureHelper, Context, CapturerObserver)} is called
 * and then begin capturing and delivering frames to the capturer observer on the capturing thread
 * when {@link tvi.webrtc.VideoCapturer#startCapture(int, int, int)} is called.
 *
 * <p>Capturers should then stop capturing and join the capturer thread when {@link
 * tvi.webrtc.VideoCapturer#stopCapture()}} is called.
 *
 * @see LocalVideoTrack
 */
public interface VideoCapturer extends tvi.webrtc.VideoCapturer {
    default VideoFormat getCaptureFormat() {
        return new VideoFormat(VideoDimensions.VGA_VIDEO_DIMENSIONS, 30);
    }

    @Override
    default void changeCaptureFormat(int width, int height, int framerate) {}

    /** This method provides an optional step to perform a final cleanup. */
    @Override
    default void dispose() {}
}
