package com.twilio.voice;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents options when accepting a {@link CallInvite}.
 */
public class AcceptOptions extends CallOptions {
    private final Map<String, String> callInviteMessage;
    private final boolean reject;

    /**
     * AcceptOptions can only be constructed via {@link AcceptOptions.Builder}
     */
    @SuppressWarnings("unused")
    private AcceptOptions() {
        callInviteMessage = null;
        reject = false;
    }

    private AcceptOptions(Builder builder) {
        this.audioTracks = (builder.audioTracks != null) ?
                builder.audioTracks : new ArrayList<LocalAudioTrack>();
        this.iceOptions = builder.iceOptions;
        this.enableInsights = builder.enableInsights;
        this.preferredAudioCodecs = builder.preferredAudioCodecs;
        this.callInviteMessage = (builder.callInvite == null) ?
                new HashMap<String, String>() : builder.callInvite.callInviteMessage;
        this.platformInfo = new PlatformInfo();
        this.region = builder.region;
        this.reject = builder.reject;
        this.eventListener = builder.eventListener;
    }

    /*
     * Invoked by JNI CallDelegate to get pointer to twilio::voice::ConnectOptions::Builder
     */
    @SuppressWarnings("unused")
    private long createNativeAcceptOptionsBuilder() {
        checkAudioTracksReleased(audioTracks);

        final String accessToken = null;

        String[] keys = new String[0];
        String[] values = new String[0];

        String[] callInviteMessageKeys = new String[callInviteMessage.size()];
        String[] callInviteMessageValues = new String[callInviteMessage.size()];
        int index = 0;
        for (Map.Entry<String, String> mapEntry : callInviteMessage.entrySet()) {
            callInviteMessageKeys[index] = mapEntry.getKey();
            callInviteMessageValues[index] = mapEntry.getValue();
            index++;
        }

        return nativeCreate(accessToken,
                keys,
                values,
                getLocalAudioTracksArray(),
                iceOptions,
                enableInsights,
                getAudioCodecsArray(),
                callInviteMessageKeys,
                callInviteMessageValues,
                reject,
                platformInfo,
                region);
    }

    private native long nativeCreate(String accessToken,
                                     String[] paramKeys,
                                     String[] paramValues,
                                     LocalAudioTrack[] audioTracks,
                                     IceOptions iceOptions,
                                     boolean enableInsights,
                                     AudioCodec[] preferredAudioCodecs,
                                     String[] incomingCallPushPayloadKeys,
                                     String[] incomingCallPushPayloadValues,
                                     boolean reject,
                                     PlatformInfo platformInfo,
                                     String region);

    /**
     * Build new {@link AcceptOptions}.
     * <p>
     * <p>All methods are optional.</p>
     */
    public static class Builder extends CallOptions.Builder {
        private final CallInvite callInvite;
        private final boolean reject;

        /**
         * Use this Builder when accepting a {@link CallInvite}
         */
        public Builder() {
            this.callInvite = null;
            this.reject = false;
        }

        /**
         * This builder is used internally when accepting or rejecting a {@link CallInvite}
         */
        Builder(@NonNull CallInvite callInvite, boolean reject) {
            Preconditions.checkNotNull(callInvite);
            this.callInvite = callInvite;
            this.reject = reject;
        }

        /**
         * Audio tracks that will be published upon connection.
         */
        @NonNull Builder audioTracks(@NonNull List<LocalAudioTrack> audioTracks) {
            Preconditions.checkNotNull(audioTracks, "audioTracks must not be null");
            super.audioTracks(audioTracks);
            return this;
        }

        /**
         * Custom ICE configuration used to connect to a Call.
         */
        @NonNull public Builder iceOptions(@NonNull IceOptions iceOptions) {
            Preconditions.checkNotNull(iceOptions, "iceOptions must not be null");
            super.iceOptions(iceOptions);
            return this;
        }

        /**
         * Specify reporting statistics to Insights. Sending stats data to Insights is enabled
         * by default.
         */
        @NonNull public Builder enableInsights(@NonNull boolean enable) {
            super.enableInsights(enable);
            return this;
        }

        /**
         * Set preferred audio codecs. The list specifies which audio codecs would be preferred when
         * negotiating audio with the backend service. The preferences are applied in the order found in
         * the list starting with the most preferred audio codec to the least preferred audio codec.
         * Audio codec preferences are not guaranteed to be satisfied because the backend service
         * is not guaranteed to support all audio codecs.
         *
         * <p>The following snippet demonstrates how to prefer a single audio codec.
         *
         * <pre><code>
         *     AcceptOptions acceptOptions = new AcceptOptions.Builder(token)
         *          .preferAudioCodecs(Collections.<AudioCodec>singletonList(new PcmuCodec()))
         *          .build();
         * </code></pre>
         *
         * <p>The following snippet demonstrates how to specify the exact order of codec
         * preferences.
         *
         * <pre><code>
         *     AcceptOptions acceptOptions = new AcceptOptions.Builder(token)
         *          .preferAudioCodecs(Arrays.asList(new PcmuCodec(), new OpusCodec()))
         *          .build();
         * </code></pre>
         */
        @NonNull public Builder preferAudioCodecs(@NonNull List<AudioCodec> preferredAudioCodecs) {
            Preconditions.checkNotNull(preferredAudioCodecs,"preferredAudioCodecs must not be null");
            checkAudioCodecs(preferredAudioCodecs);
            super.preferAudioCodecs(preferredAudioCodecs);
            return this;
        }

        /**
         * Sets the region (Twilio data center) for the SDK.
         * The default region uses Global Low Latency routing, which establishes a connection with
         * the closest region to the user.
         * <p>
         * NOTE: Setting the region during a call will not apply until all ongoing calls have ended
         * and a subsequent call is placed.
         *
         * @param region The region.
         */
        @NonNull public Builder region(@NonNull String region) {
            Preconditions.checkNotNull(region, "region must not be null");
            super.region(region);
            return this;
        }

        Builder eventListener(Call.EventListener eventListener) {
            super.eventListener(eventListener);
            return this;
        }

        /**
         * Builds {@link AcceptOptions} object.
         */
        @NonNull public AcceptOptions build() {
            checkAudioTracksReleased(audioTracks);
            return new AcceptOptions(this);
        }
    }
}
