package com.twilio.voice;

import android.content.Context;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Pair;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Queue;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * The Call class represents a signaling and media session between the host device and Twilio
 * infrastructure. Calls can represent an outbound call initiated from the SDK via
 * {@code Voice.connect(...)} or an incoming call accepted via a {@link CallInvite}. Devices that
 * initiate an outbound call represent the <i>caller</i> and devices that receive a
 * {@link CallInvite} represent the <i>callee</i>.
 * <p>
 * The lifecycle of a call differs for the caller and the callee. Making or accepting a call
 * requires a {@link Call.Listener} which provides {@link Call} state changes defined in
 * {@link Call.State}. The {@link Call.State} can be obtained at any time by calling
 * {@link Call#getState()}.
 * <p>
 * The following table provides expected call sequences for common scenarios from the caller
 * perspective. The caller scenario sequences are affected by the {@code answerOnBridge} flag provided
 * in the {@code Dial} verb of your TwiML application associated with the client. If the
 * {@code answerOnBridge} flag is {@code false}, which is the default, the
 * {@link Listener#onConnected(Call)} callback will be emitted immediately after
 * {@link Listener#onRinging(Call)}. If the {@code answerOnBridge} flag is {@code true} this will
 * cause the call to emit the {@link Listener#onConnected(Call)} callback only until the call is
 * answered by the callee.
 * See the <a href="https://www.twilio.com/docs/voice/twiml/dial#answeronbridge">Answer on Bridge documentation</a> for more details on how to use
 * it with the {@code Dial} TwiML verb.
 *
 * <table border="1" summary="Caller Call Sequences">
 * <tr>
 *  <td>Caller Scenario</td>
 *  <td>{@link Call.Listener} callbacks</td>
 *  <td>{@link Call.State} transitions</td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated and the caller disconnects before reaching the {@link State#RINGING} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated, reaches the {@link State#RINGING} state, and the caller disconnects before being {@link State#CONNECTED}.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onRinging(Call)} </li>
 *         <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#RINGING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated and the {@link Call} fails to connect before reaching the {@link State#RINGING} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onConnectFailure(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated and the {@link Call} fails to connect after reaching the {@link State#RINGING} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onRinging(Call)} </li>
 *         <li>{@link Listener#onConnectFailure(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#RINGING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link Call} is initiated, becomes {@link State#CONNECTED}, and ends for one of the following reasons:
 *      <ul>
 *          <li>The caller disconnects the {@link Call}</li>
 *          <li>The callee disconnects the {@link Call}</li>
 *          <li>An on-device error occurs. (eg. network loss)</li>
 *          <li>An error between the caller and callee occurs. (eg. media connection lost or Twilio infrastructure failure)</li>
 *      </ul>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link Listener#onRinging(Call)}</li>
 *          <li>{@link Listener#onConnected(Call)}</li>
 *          <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *      </ol>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link State#CONNECTING}</li>
 *          <li>{@link State#RINGING}</li>
 *          <li>{@link State#CONNECTED}</li>
 *          <li>{@link State#DISCONNECTED}</li>
 *      </ol>
 *  </td>
 * </tr>
 * </table>
 * <p>
 * The following table provides expected call sequences for common scenarios from the callee
 * perspective:
 *
 * <table border="1" summary="Callee Call Sequences">
 * <tr>
 * <td>Callee Scenario</td><td>{@link Call.Listener} callbacks</td><td>{@link Call.State} transitions</td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is accepted and the callee disconnects the {@link Call} before being {@link State#CONNECTED}.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is accepted and the {@link Call} fails to reach the {@link State#CONNECTED} state.</td>
 *  <td>
 *     <ol>
 *         <li>{@link Listener#onConnectFailure(Call, CallException)}</li>
 *     </ol>
 *  </td>
 *  <td>
 *     <ol>
 *         <li>{@link State#CONNECTING}</li>
 *         <li>{@link State#DISCONNECTED}</li>
 *     </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is accepted, becomes {@link State#CONNECTED}, and ends for one of the following reasons:
 *      <ul>
 *          <li>The caller disconnects the {@link Call}</li>
 *          <li>The callee disconnects the {@link Call}</li>
 *          <li>An on-device error occurs. (eg. network loss)</li>
 *          <li>An error between the caller and callee occurs. (eg. media connection lost or Twilio infrastructure failure)</li>
 *      </ul>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link Listener#onConnected(Call)}</li>
 *          <li>{@link Listener#onDisconnected(Call, CallException)}</li>
 *      </ol>
 *  </td>
 *  <td>
 *      <ol>
 *          <li>{@link State#CONNECTING}</li>
 *          <li>{@link State#CONNECTED}</li>
 *          <li>{@link State#DISCONNECTED}</li>
 *      </ol>
 *  </td>
 * </tr>
 * <tr>
 *  <td>A {@link CallInvite} is rejected.</td>
 *  <td>Reject will not result in any callbacks.</td>
 *  <td>The call invite was not accepted, therefore the {@link Call.State} is not available.</td>
 * </tr>
 * </table>
 * <p>

 * <p>
 * It is important to call the {@link #disconnect()} method to terminate the call. Not calling
 * {@link #disconnect()} results in leaked native resources and may lead to an out-of-memory crash.
 * If the call is disconnected by the caller, callee, or Twilio, the SDK will automatically free
 * native resources. However, {@link #disconnect()} is an idempotent operation so it is best to
 * call this method when you are certain your application no longer needs the object.
 * <p>
 * It is strongly recommended that Call instances are created and accessed from a single application
 * thread. Accessing an instance from multiple threads may cause synchronization problems.
 * Listeners are called on the thread that created the Call instance, unless the thread that created
 * the Call instance does not have a Looper. In that case, the listener will be called on the application's
 * main thread.
 *
 * <p>Insights :</p>
 *
 * <p>The following connection events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights connection events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>muted</td><td>Audio input of the {@link Call} is muted by calling {@link Call#mute(boolean)} with {@code true}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>unmuted</td><td>Audio input of the {@link Call} is unmuted by calling {@link Call#mute(boolean)} with {@code false}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>accepted-by-remote</td><td>Server returns an answer to the Client’s offer over the signaling channel</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>ringing</td><td>Call state transitions to {@link Call.State#RINGING}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>connected</td><td>Call state transitions to {@link Call.State#CONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>disconnect-called</td><td>{@link Call#disconnect()} is called</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>disconnected-by-local</td><td>Call disconnected as a result of calling {@link Call#disconnect()}. {@link Call.State} transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>disconnected-by-remote</td><td>Call is disconnected by the remote. {@link Call.State} transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>hold</td><td>{@link Call} is on hold by calling {@link Call#hold(boolean)}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>unhold</td><td>{@link Call} is unhold by calling {@link Call#hold(boolean)}</td><td>3.0.0-beta2</td>
 * </tr>
 * <tr>
 * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
 * </tr>
 * </table>
 *
 <p>The following ICE connection state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights ICE connection events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>new</td><td>The PeerConnection ice connection state changed to "new"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>checking</td><td>The PeerConnection ice connection state changed to "checking"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>connected</td><td>The PeerConnection ice connection state changed to "connected"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>completed</td><td>The PeerConnection ice connection state changed to "completed"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>closed</td><td>The PeerConnection ice connection state changed to “closed”</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>disconnected</td><td>The PeerConnection ice connection state changed to “disconnected”</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-connection-state</td><td>failed</td><td>The PeerConnection ice connection state changed to “failed”</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following ICE gathering state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights ICE gathering events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>ice-gathering-state</td><td>new</td><td>The PeerConnection ice gathering state changed to "new"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-gathering-state</td><td>gathering</td><td>The PeerConnection ice gathering state changed to "checking"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>ice-gathering-state</td><td>complete</td><td>The PeerConnection ice gathering state changed to "connected"</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following signaling state events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights peer connection signaling state events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>stable</td><td>The PeerConnection signaling state changed to "stable"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-local-offer</td><td>The PeerConnection signaling state changed to "have-local-offer"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-remote-offers</td><td>The PeerConnection signaling state changed to "have-remote-offers"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-local-pranswer</td><td>The PeerConnection signaling state changed to "have-local-pranswer"</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>have-remote-pranswer</td><td>The PeerConnection signaling state changed to “have-remote-pranswer”</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>signaling-state</td><td>closed</td><td>The PeerConnection signaling state changed to “closed”</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following network quality warning and network quality warning cleared events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights network quality events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>high-jitter</td><td>Three out of last five jitter samples exceed 30 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>high-jitter</td><td>high-jitter warning cleared if last five jitter samples are less than 30 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>low-mos</td><td>Three out of last five mos samples are lower than 3</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>low-mos</td><td>low-mos cleared if last five mos samples are higher than 3</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>high-packet-loss</td><td>Three out of last five packet loss samples show loss greater than 1%</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>high-packet-loss</td><td>high-packet-loss cleared if last five packet loss samples are lower than 1%</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-raised</td><td>high-rtt</td><td>Three out of last five RTT samples show greater than 400 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>network-quality-warning-cleared</td><td>high-rtt</td><td>high-rtt warning cleared if last five RTT samples are lower than 400 ms</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following audio level warning and audio level warning cleared events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights audio level events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-raised</td><td>constant-audio-input-level</td><td>Last ten audio input samples have the same audio level</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-cleared</td><td>constant-audio-input-level</td><td>constant-audio-input-level warning cleared if the current audio input level sample differs from the previous audio input level sample</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-raised</td><td>constant-audio-output-level</td><td>Last ten audio output samples have the same audio level</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>audio-level-warning-cleared</td><td>constant-audio-output-level</td><td>constant-audio-output-level warning cleared if the current audio output level sample differs from the previous audio output level sample</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 * <p>The following feedback events are reported to Insights during a {@link Call}</p>
 *
 * <table border="1" summary="Insights feedback events">
 * <tr>
 * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
 * </tr>
 * <tr>
 * <td>feedback</td><td>received</td><td>{@link Call#postFeedback(Score, Issue)} is called with a score other than {@link Call.Score#NOT_REPORTED} or an issue other than {@link Call.Issue#NOT_REPORTED} i</td><td>3.0.0-beta3</td>
 * </tr>
 * <tr>
 * <td>feedback</td><td>received-none</td><td>{@link Call#postFeedback(Score, Issue)} is called with {@link Call.Score#NOT_REPORTED} and {@link Call.Issue#NOT_REPORTED}</td><td>3.0.0-beta3</td>
 * </tr>
 * </table>
 *
 */
public class Call extends InternalCall {
    private final ThreadUtils.ThreadChecker threadChecker;
    private Listener listener;
    private EventListener eventListener;
    private static final Logger logger = Logger.getLogger(Call.class);
    private List<LocalAudioTrack> localAudioTracks;
    private MediaFactory mediaFactory;
    private long nativeCallDelegate;
    private Queue<Pair<Handler, StatsListener>> statsListenersQueue;
    private ConnectivityReceiver connectivityReceiver = null;

    /**
     * An enum representing call quality score.
     */
    public enum Score {
        /**
         * No score reported.
         */
        NOT_REPORTED(0),
        /**
         * Terrible call quality, call dropped, or caused great difficulty in communicating.
         */
        ONE(1),
        /**
         * Bad call quality, like choppy audio, periodic one-way-audio.
         */
        TWO(2),
        /**
         * Average call quality, manageable with some noise/minor packet loss.
         */
        THREE(3),
        /**
         * Good call quality, minor issues.
         */
        FOUR(4),
        /**
         * Great call quality. No issues.
         */
        FIVE(5);

        private final int score;

        private Score(int score) {
            this.score = score;
        }

        public int getValue() {
            return this.score;
        }
    }

    /**
     * An enum representing issue type associated with a call.
     */
    public enum Issue {
        /**
         * No issue reported.
         */
        NOT_REPORTED("not-reported"),
        /**
         * Call initially connected but was dropped.
         */
        DROPPED_CALL("dropped-call"),
        /**
         * Participants can hear each other but with significant delay.
         */
        AUDIO_LATENCY("audio-latency"),
        /**
         * One participant couldn’t hear the other.
         */
        ONE_WAY_AUDIO("one-way-audio"),
        /**
         * Periodically, participants couldn’t hear each other. Some words were lost.
         */
        CHOPPY_AUDIO("choppy-audio"),
        /**
         * There was disturbance, background noise, low clarity.
         */
        NOISY_CALL("noisy-call"),
        /**
         * There was echo during call.
         */
        ECHO("echo");

        private final String issueName;

        private Issue(String issueName) {
            this.issueName = issueName;
        }

        public String toString() {
            return this.issueName;
        }
    }

    /*
     * The contract for Call JNI callbacks is as follows:
     *
     * 1. All event callbacks are done on the same thread the developer used to connect to a call.
     * 2. Create and release all native memory on the same thread. In the case of a Call, the
     * CallDelegate is created and released on the developer thread and the native call and call
     * observer are created and released on notifier thread.
     * 3. All Call fields must be mutated on the developer's thread.
     *
     * Not abiding by this contract, may result in difficult to debug JNI crashes,
     * incorrect return values in the synchronous API methods, or missed callbacks.
     */
    private final Call.Listener callListenerProxy = new Call.Listener() {
        @Override
        public void onRinging(@NonNull final Call call) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onRinging()");

                    // Update call state
                    Call.this.state = State.RINGING;
                    call.sid = nativeGetSid(nativeCallDelegate);
                    Call.this.listener.onRinging(call);
                }
            });
        }

        @Override
        public void onConnected(@NonNull final Call call) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onConnected()");

                    // Update call state
                    Call.this.state = State.CONNECTED;
                    call.sid = nativeGetSid(nativeCallDelegate);
                    Call.this.listener.onConnected(call);
                }
            });
        }

        @Override
        public void onConnectFailure(@NonNull final Call call, @NonNull final CallException callException) {

            handler.post(new Runnable() {
                @Override
                public void run() {
                    // Release native call
                    releaseCall();

                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onConnectFailure()");

                    unregisterConnectivityBroadcastReceiver(context);
                    Voice.calls.remove(Call.this);
                    Voice.rejects.remove(Call.this);

                    // Update call state
                    Call.this.state = State.DISCONNECTED;

                    // Release native call delegate
                    release();

                    // Notify developer
                    Call.this.listener.onConnectFailure(call, callException);
                }
            });
        }

        @Override
        public void onDisconnected(@NonNull final Call call, final CallException callException) {

            handler.post(new Runnable() {
                @Override
                public void run() {
                    // Release native call
                    releaseCall();

                    Call.this.threadChecker.checkIsOnValidThread();
                    logger.d("onDisconnected()");

                    unregisterConnectivityBroadcastReceiver(context);
                    Voice.calls.remove(Call.this);
                    Voice.rejects.remove(Call.this);

                    // Update call state
                    Call.this.state = State.DISCONNECTED;

                    // Release native call delegate
                    release();

                    // Notify developer
                    Call.this.listener.onDisconnected(call, callException);
                }
            });
        }
    };

    Call.EventListener eventListenerProxy = new Call.EventListener() {

        @Override
        public void onEvent(HashMap<String, String> data) {
            handler.post((new Runnable() {
                @Override
                public void run() {
                    if (Call.this.eventListener != null) {
                        Call.this.eventListener.onEvent(data);
                    }
                    if (data.get(EventKeys.EVENT_GROUP).equals(EventGroupType.ICE_GATHERING_STATE_GROUP)) {
                        publishIceGatheringEvent(data.get(EventKeys.EVENT_NAME));
                    } else if (data.get(EventKeys.EVENT_GROUP).equals(EventGroupType.ICE_CONNECTION_STATE_GROUP)) {
                        publishIceConnectionEvent(data.get(EventKeys.EVENT_NAME));
                    } else if (data.get(EventKeys.EVENT_GROUP).equals(EventGroupType.CONNECTION_EVENT_GROUP)) {
                        if (data.get(EventKeys.EVENT_NAME).equals(EventType.CONNECTION_ERROR)) {
                            String errorMessage = data.get(EventKeys.ERROR_MESSAGE_KEY) + " : " + data.get(EventKeys.ERROR_EXPLANATION_KEY);
                            if (data.get(EventKeys.ERROR_CODE_KEY) != null) {
                                publishConnectionErrorEvent(data.get(EventKeys.EVENT_NAME),
                                        Integer.parseInt(data.get(EventKeys.ERROR_CODE_KEY)),
                                        errorMessage);
                            }
                        } else {
                            publishConnectionEvent(data.get(EventKeys.EVENT_NAME));
                        }
                    } else if (data.get(EventKeys.EVENT_GROUP).equals(EventGroupType.SIGNALING_STATE_GROUP)) {
                        publishSignalingStateEvent(data.get(EventKeys.EVENT_NAME));
                    } else if (data.get(EventKeys.EVENT_GROUP).equals(EventGroupType.EDGE_GROUP)) {
                        gateway = data.get(EventKeys.EDGE_HOST_NAME);
                        region = data.get(EventKeys.EDGE_HOST_REGION);
                    }
                }
            }));
        }

        @Override
        public void onMetric(HashMap<String, String> data) {
            if (Call.this.eventListener != null) {
                Call.this.eventListener.onMetric(data);
            }

            if (data.get(EventKeys.EVENT_GROUP).equals(EventGroupType.CALL_QUALITY_STATS_GROUP)) {
                RTCStatsSample rtcStatsSample = new RTCStatsSample();
                rtcStatsSample.packetsReceived = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.PACKETS_RECEIVED));
                rtcStatsSample.totalPacketsLost = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.TOTAL_PACKETS_LOST));
                rtcStatsSample.fractionLost = (int) Double.parseDouble(data.get(MetricEventConstants.MetricEventKeys.PACKETS_LOST_FRACTION));
                rtcStatsSample.packetsLost = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.PACKETS_LOST));
                rtcStatsSample.totalPacketsSent = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.TOTAL_PACKETS_SENT));
                rtcStatsSample.totalPacketsReceived = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.TOTAL_PACKETS_RECEIVED));
                rtcStatsSample.totalBytesReceived = Long.parseLong(data.get(MetricEventConstants.MetricEventKeys.TOTAL_BYTES_RECEIVED));
                rtcStatsSample.totalBytesSent = Long.parseLong(data.get(MetricEventConstants.MetricEventKeys.TOTAL_BYTES_SENT));
                rtcStatsSample.jitter = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.JITTER));
                rtcStatsSample.rtt = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.RTT));
                rtcStatsSample.audioInputLevel = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.AUDIO_INPUT_LEVEL));
                rtcStatsSample.audioOutputLevel = Integer.parseInt(data.get(MetricEventConstants.MetricEventKeys.AUDIO_OUTPUT_LEVEL));
                rtcStatsSample.mos = Double.parseDouble(data.get(MetricEventConstants.MetricEventKeys.MOS));
                rtcStatsSample.codec = data.get(MetricEventConstants.MetricEventKeys.AUDIO_CODEC);
                rtcStatsSample.timestampMS = (long) Double.parseDouble(data.get(MetricEventConstants.MetricEventKeys.TIMESTAMP_MS));

                // populate sample TimeStamp
                TimeZone tz = TimeZone.getTimeZone("UTC");
                DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
                df.setTimeZone(tz);
                Date now = new Date(rtcStatsSample.timestampMS);
                rtcStatsSample.timeStamp = df.format(now);

                onSample(rtcStatsSample);
            }
        }
    };

    /**
     * An enum describing the possible states of a Call.
     */
    public enum State {
        /**
         * The {@link Call} was created or was accepted and is in the process of connecting.
         */
        CONNECTING,
        /**
         * The {@link Call} is ringing.
         */
        RINGING,
        /**
         * The {@link Call} is connected.
         */
        CONNECTED,
        /**
         * The {@link Call} was disconnected, either due to a disconnect or an error.
         */
        DISCONNECTED
    }

    /**
     * Call.Listener interface defines a set of callbacks for events related to
     * call.
     *
     * @see Call Reference the call overview for a reference of callback sequences for making and
     * receiving calls.
     */
    public interface Listener {
        /**
         * The call failed to connect.
         * <p>
         * Calls that fail to connect will result in {@link Call.Listener#onConnectFailure(Call, CallException)}
         * and always return a {@link CallException} providing more information about what failure occurred.
         * </p>
         *
         * @param call          An object model representing a call that failed to connect.
         * @param callException CallException that describes why the connect failed.
         */
        void onConnectFailure(@NonNull Call call, @NonNull CallException callException);

        /**
         * Emitted once before the {@link Call.Listener#onConnected(Call)} callback. If
         * {@code answerOnBridge} is true, this represents the callee being alerted of a call.
         *
         * The {@link Call#getSid()} is now available.
         *
         * @param call  An object model representing a call.
         */
        void onRinging(@NonNull Call call);

        /**
         * The call has connected.
         *
         * @param call An object model representing a call.
         */
        void onConnected(@NonNull Call call);

        /**
         * The call was disconnected.
         * <p>
         * A call can be disconnected for the following reasons:
         * <ul>
         * <li>A user calls `disconnect()` on the `Call` object.</li>
         * <li>The other party disconnects or terminates the call.</li>
         * <li>An error occurs on the client or the server that terminates the call.</li>
         * </ul>
         * <p>
         * If the call ends due to an error the `CallException` is non-null. If the call ends normally `CallException` is null.
         * </p>
         *
         * @param call          An object model representing a call.
         * @param callException CallException that caused the call to disconnect.
         */
        void onDisconnected(@NonNull Call call, @Nullable CallException callException);
    }

    interface EventListener {
        void onEvent(HashMap<String, String> data);
        void onMetric(HashMap<String, String> data);
    }

    private final StatsListener statsListenerProxy = new StatsListener() {
        @Override
        public void onStats(final List<StatsReport> statsReports) {
            final Pair<Handler, StatsListener> statsPair = Call.this.statsListenersQueue.poll();
            if (statsPair != null) {
                statsPair.first.post(new Runnable() {
                    @Override
                    public void run() {
                        statsPair.second.onStats(statsReports);
                    }
                });
            }
        }
    };

    Call(final Context context, final CallInvite callInvite, final Listener listener) {
        Preconditions.checkApplicationContext(context, "must create Call with application context");
        this.context = context;
        this.listener = listener;
        this.from = callInvite.getFrom();
        this.to = callInvite.getTo();
        this.sid = callInvite.getCallSid();
        this.bridgeToken = callInvite.getBridgeToken();
        this.disconnectCalled = false;
        this.direction = Constants.Direction.INCOMING;
        this.handler = Utils.createHandler();
        this.threadChecker = new ThreadUtils.ThreadChecker(handler.getLooper().getThread());
        this.state = State.CONNECTING;
        this.publisher = new EventPublisher(Constants.CLIENT_SDK_PRODUCT_NAME,
                bridgeToken);
        this.publisher.addListener(this);
        this.rtcMonitor = new RTCMonitor();
        this.statsListenersQueue = new ConcurrentLinkedQueue<>();
    }

    Call(final Context context, final String accessToken, final Listener listener) {
        Preconditions.checkApplicationContext(context, "must create Call with application context");
        this.context = context;
        this.listener = listener;
        this.state = State.CONNECTING;
        this.direction = Constants.Direction.OUTGOING;
        this.handler = Utils.createHandler();
        this.threadChecker = new ThreadUtils.ThreadChecker(handler.getLooper().getThread());
        this.publisher = new EventPublisher(Constants.CLIENT_SDK_PRODUCT_NAME, accessToken);
        this.publisher.addListener(this);
        this.rtcMonitor = new RTCMonitor();
        this.statsListenersQueue = new ConcurrentLinkedQueue<>();
    }

    /**
     * Returns the caller information when available. The from field is {@code null} for an outgoing call
     * and may be {@code null} if it was not provided in the {@link CallInvite} for an incoming call.
     */
    @Nullable public String getFrom() {
        return from;
    }

    /**
     * Returns the callee information when available. Returns null for an outgoing call.
     */
    @Nullable public String getTo() {
        return to;
    }

    /**
     * Returns the call sid. The call sid is null until the call is in {@link State#RINGING} state.
     */
    @Nullable public String getSid() {
        return sid;
    }

    /**
     * Returns the current state of the call.
     *
     * <p>Call is in {@link State#CONNECTING} state when it is made or accepted.</p>
     * <p>Call is in {@link State#RINGING} state when it is ringing.</p>
     * <p>Call transitions to {@link State#CONNECTED} state when connected to Twilio.</p>
     * <p>Call transitions to {@link State#DISCONNECTED} state when disconnected.</p>
     *
     * @return Provides the state of this call.
     */
    @Override
    @NonNull public State getState() {
        return state;
    }

    /**
     * Retrieve stats for all media tracks and notify {@link StatsListener} via calling thread.
     * In case where call is in {@link State#DISCONNECTED} state, reports won't be delivered.
     *
     * @param statsListener listener that receives stats reports for all media tracks.
     */
    public synchronized void getStats(@NonNull StatsListener statsListener) {
        threadChecker.checkIsOnValidThread();

        Preconditions.checkNotNull(statsListener, "statsListener must not be null");
        if (state == State.DISCONNECTED) {
            return;
        }
        statsListenersQueue.offer(new Pair<>(Utils.createHandler(), statsListener));
        nativeGetStats(nativeCallDelegate);
    }

    /**
     * Posts the feedback collected for this call to Twilio. If `0` `Score` and `not-reported`
     * `Issue` are passed, Twilio will report feedback was not available for this call.
     *
     * @param score - the call quality score.
     * @param issue - the issue type associated with the call.
     */
    public void postFeedback(@NonNull Score score, @NonNull Issue issue) {
        Preconditions.checkNotNull(score, "score must not be null");
        Preconditions.checkNotNull(issue, "issue must not be null");
        publishFeedbackEvent(score, issue);
    }

    void connect(final ConnectOptions connectOptions) {
        threadChecker.checkIsOnValidThread();

        registerConnectivityBroadcastReceiver(context);
        Voice.calls.add(this);

        // Check if audio or video tracks have been released
        ConnectOptions.checkAudioTracksReleased(connectOptions.getAudioTracks());

        localAudioTracks = connectOptions.getAudioTracks();
        selectedRegion = connectOptions.getRegion();

        synchronized (callListenerProxy) {
            Voice.loadLibrary(context);
            mediaFactory = MediaFactory.instance(this, context);
            if (connectOptions.getEventListener() != null) {
                this.eventListener = connectOptions.getEventListener();
            }
            nativeCallDelegate = nativeConnect(connectOptions,
                    callListenerProxy,
                    statsListenerProxy,
                    eventListenerProxy,
                    mediaFactory.getNativeMediaFactoryHandle(),
                    handler);
        }
    }

    void accept(final AcceptOptions acceptOptions) {
        threadChecker.checkIsOnValidThread();

        registerConnectivityBroadcastReceiver(context);
        Voice.calls.add(this);

        // Check if audio or video tracks have been released
        AcceptOptions.checkAudioTracksReleased(acceptOptions.getAudioTracks());

        localAudioTracks = acceptOptions.getAudioTracks();
        selectedRegion = acceptOptions.getRegion();

        synchronized (callListenerProxy) {
            Voice.loadLibrary(context);
            mediaFactory = MediaFactory.instance(this, context);
            if (acceptOptions.getEventListener() != null) {
                this.eventListener = acceptOptions.getEventListener();
            }
            nativeCallDelegate = nativeAccept(acceptOptions,
                    callListenerProxy,
                    statsListenerProxy,
                    eventListenerProxy,
                    mediaFactory.getNativeMediaFactoryHandle(),
                    handler);
        }
    }

    /*
     * Synchronize accesses to call listener during initialization and make
     * sure that onConnect() callback won't get called before connect() exits and Call
     * creation is fully completed.
     */
    void reject(final AcceptOptions acceptOptions) {
        threadChecker.checkIsOnValidThread();

        registerConnectivityBroadcastReceiver(context);
        Voice.rejects.add(this);

        localAudioTracks = acceptOptions.getAudioTracks();
        selectedRegion = acceptOptions.getRegion();

        synchronized (callListenerProxy) {
            Voice.loadLibrary(context);
            if (acceptOptions.getEventListener() != null) {
                this.eventListener = acceptOptions.getEventListener();
            }
            nativeCallDelegate = nativeReject(acceptOptions,
                    callListenerProxy,
                    eventListenerProxy,
                    handler);
        }
    }

    /**
     * Mutes or unmutes the audio input.
     */
    public synchronized void mute(final boolean mute) {
        threadChecker.checkIsOnValidThread();
        if (isValidState()) {
            isMuted = mute;
            nativeMute(nativeCallDelegate, mute);
        }
    }

    /**
     * Sends a string of DTMF digits.
     *
     * @param digits A string of digits to be sent. Valid values are "0" - "9", "*", "#", and "w". Each "w" will cause a 500 ms pause between digits sent.
     */
    public synchronized void sendDigits(@NonNull String digits) {
        threadChecker.checkIsOnValidThread();
        Preconditions.checkNotNull(digits, "digits must not be null");
        if (!digits.matches("^[0-9\\*\\#w]+$")) {
            throw new IllegalArgumentException("digits string must not be null and should only contains 0-9, *, #, or w characters");
        }
        if (isValidState()) {
            nativeSendDigits(nativeCallDelegate, digits);
        }
    }

    /**
     * Holds or un-holds the audio.
     */
    public synchronized void hold(final boolean hold) {
        threadChecker.checkIsOnValidThread();
        if (isValidState()) {
            isOnHold = hold;
            nativeHold(nativeCallDelegate, hold);
        }
    }

    /**
     * Reports whether the audio input is muted.
     */
    public boolean isMuted() {
        return isMuted;
    }

    /**
     * Reports whether the call is on hold.
     */
    public boolean isOnHold() {
        return isOnHold;
    }

    /**
     * Disconnects the Call.
     */
    @Override
    public synchronized void disconnect() {
        threadChecker.checkIsOnValidThread();
        if (!disconnectCalled && isValidState()) {
            disconnectCalled = true;
            logger.d("Calling disconnect " + state);
            nativeDisconnect(nativeCallDelegate);
        }
    }

    void networkChange(Voice.NetworkChangeEvent networkChangeEvent) {
        threadChecker.checkIsOnValidThread();
        if (isValidState() && isPermittedNetworkChangeEvent(networkChangeEvent)) {
            nativeNetworkChange(nativeCallDelegate, networkChangeEvent);
        } else {
            logger.d("Ignoring networkChangeEvent: " + networkChangeEvent.name() +
                    " in Call.State: " + state);
        }
    }

    /*
     * Release the native CallDelegate from developer thread once the native Call memory
     * has been released.
     */
    synchronized void release() {
        this.threadChecker.checkIsOnValidThread();

        for (LocalAudioTrack localAudioTrack : localAudioTracks) {
            localAudioTrack.release();
        }

        if (nativeCallDelegate != 0) {
            nativeRelease(nativeCallDelegate);
            nativeCallDelegate = 0;
            if (mediaFactory != null) {
                mediaFactory.release(this);
            }
        }
    }

    private void registerConnectivityBroadcastReceiver(Context context) {
        connectivityReceiver = new ConnectivityReceiver();
        context.registerReceiver(connectivityReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
    }

    /*
     * A customer observed (CLIENT-5899) CONNECTION_CHANGED events handled before reaching
     * State.CONNECTED resulted in the Call being disconnected. The SDK has intentionally been
     * updated to ignore CONNECTION_CHANGED events while a Call is in CONNECTING or RINGING state to
     * prevent these premature Call disconnects in cases where the Call can connect despite the
     * network change occurring. This behavior is different from the iOS SDK and can result in the
     * following outcomes:
     *
     * 1. Ignoring the CONNECTION_CHANGED event results in the Call successfully being connected
     * 2. Ignoring the CONNECTION_CHANGED event results in the Call being disconnected. With this
     * behavior, the SDK could take more time to disconnect the Call than in previous releases.
     *
     * This change is considered a temporary workaround until the completion of the Mobile
     * Connection Robustness epic CLIENT-5756.
     */
    private boolean isPermittedNetworkChangeEvent(Voice.NetworkChangeEvent networkChangeEvent) {
        return !(networkChangeEvent == Voice.NetworkChangeEvent.CONNECTION_CHANGED &&
                (state == Call.State.CONNECTING || state == Call.State.RINGING));
    }

    private void unregisterConnectivityBroadcastReceiver(Context context) {
        context.unregisterReceiver(connectivityReceiver);
        connectivityReceiver = null;
    }

    private synchronized void releaseCall() {
        if (nativeCallDelegate != 0) {
            nativeReleaseCall(nativeCallDelegate);
        }
    }

    private native long nativeConnect(ConnectOptions ConnectOptions,
                                      Listener listenerProxy,
                                      StatsListener statsListenerProxy,
                                      EventListener eventListenerProxy,
                                      long nativeMediaFactoryHandle,
                                      Handler handler);

    private native long nativeAccept(AcceptOptions acceptOptions,
                                     Listener listenerProxy,
                                     StatsListener statsListenerProxy,
                                     EventListener eventListenerProxy,
                                     long nativeMediaFactoryHandle,
                                     Handler handler);

    private native long nativeReject(AcceptOptions acceptOptions,
                                     Listener listenerProxy,
                                     EventListener eventListenerProxy,
                                     Handler handler);

    private native String nativeGetSid(long nativeCallDelegate);

    private native void nativeGetStats(long nativeCallDelegate);

    private native void nativeMute(long nativeCallDelegate, boolean mute);

    private native void nativeSendDigits(long nativeCallDelegate, String digits);

    private native void nativeHold(long nativeCallDelegate, boolean hold);

    private native void nativeDisconnect(long nativeCallDelegate);

    private native void nativeNetworkChange(long nativeCallDelegate,
                                            Voice.NetworkChangeEvent networkChangeEvent);

    private native void nativeReleaseCall(long nativeCallDelegate);

    private native void nativeRelease(long nativeCallDelegate);

}
