package com.twilio.voice;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import static com.twilio.voice.Utils.isAudioPermissionGranted;

/**
 * Represents an incoming call message from Twilio. This object is used to respond
 * to an incoming call by calling {@link CallInvite#accept(Context, Call.Listener)} or
 * {@link CallInvite#reject(Context)}
 */
public class CallInvite implements Parcelable {
    private static final Logger logger = Logger.getLogger(InternalCall.class);
    private final String bridgeToken;
    private final String callSid;
    private final String from;
    private final String to;
    private final EventPublisher publisher;
    final Map<String, String> callInviteMessage;
    final Map<String, String> customParameters;

    /*
    * A callListenerProxy that receives callbacks for rejected CallInvite and clean up native resources.
    */
    private final Call.Listener callListenerProxy = new Call.Listener() {

        @Override
        public void onRinging(@NonNull Call call) {

        }

        @Override
        public void onConnected(@NonNull Call call) {

        }

        @Override
        public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
            call.release();
        }

        @Override
        public void onDisconnected(@NonNull Call call, CallException callException) {
            call.release();
        }
    };

    Call.EventListener eventListenerProxy = new Call.EventListener() {

        @Override
        public void onEvent(HashMap<String, String> data) {
            // no op
        }

        @Override
        public void onMetric(HashMap<String, String> data) {
            // no op
        }
    };

    /**
     * Creates and returns a CallInvite from a valid message.
     *
     * @param data message data as String key/value pairs.
     */
    static CallInvite create(Map<String, String> data) {
        return new CallInvite(data);
    }

    private CallInvite(Map<String, String> data) {
        from = data.get(VoiceConstants.FROM);
        to = data.get(VoiceConstants.TO);
        callSid = data.get(VoiceConstants.CALL_SID);
        bridgeToken = data.get(VoiceConstants.BRIDGE_TOKEN);
        customParameters = new HashMap<>();
        String query_pairs = data.get(VoiceConstants.CUSTOM_PARAMS);
        if (query_pairs != null) {
            final String[] pairs = query_pairs.split("&");
            for (String pair : pairs) {
                final int idx = pair.indexOf("=");
                final String key;
                try {
                    key = idx > 0 ? pair.substring(0, idx) : pair;
                    final String value = idx > 0 && pair.length() > idx + 1 ? URLDecoder.decode(pair.substring(idx + 1).replaceAll("\\+", "%20"), "UTF-8") : null;
                    customParameters.put(key, value);
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
            }
        }
        callInviteMessage = data;
        publisher = new EventPublisher(Constants.CLIENT_SDK_PRODUCT_NAME, bridgeToken);
        publisher.addListener(new EventPublisher.EventPublisherListener() {
            @Override
            public void onError(VoiceException voiceException) {
                logger.e("Error publishing data : " + voiceException.getMessage() + ":" + voiceException.getErrorCode());
            }
        });
        publishConnectionEvent(EventType.INCOMING);
    }

    private CallInvite(Parcel in) {
        String[] data = new String[4];
        in.readStringArray(data);
        from = data[0];
        to = data[1];
        callSid = data[2];
        bridgeToken = data[3];
        int size = in.readInt();
        callInviteMessage = new HashMap<>(size);
        for(int i = 0; i < size; i++){
            String key = in.readString();
            String value = in.readString();
            callInviteMessage.put(key,value);
        }
        int sizeOfCustomParams = in.readInt();
        customParameters = new HashMap<>(sizeOfCustomParams);
        for(int i = 0; i < sizeOfCustomParams; i++){
            String key = in.readString();
            String value = in.readString();
            customParameters.put(key,value);
        }
        publisher = null;
    }

    /**
     * Returns the caller information when available.
     */
    @Nullable public String getFrom() {
        return from;
    }

    /**
     * Returns the callee information.
     */
    @NonNull public String getTo() {
        return to;
    }

    /**
     * Returns the CallSid.
     */
    @NonNull public String getCallSid() {
        return callSid;
    }

    /**
     * Returns the custom parameters.
     *<p> <pre>
     * {@code
     * // Pass custom parameters in TwiML
     * <?xml version="1.0" encoding="UTF-8"?>
     * <Response>
     *    <Dial answerOnBridge="false" callerId="client:alice">
     *        <Client>
     *            <Identity>bob</Identity>
     *            <Parameter name="caller_first_name" value="alice"  />
     *            <Parameter name="caller_last_name" value="smith"  />
     *        </Client>
     *     </Dial>
     * </Response>
     * }
     * `callInvite.getCustomParameters()` returns a map of key-value pair passed in the TwiML.
     * {@code
     * "caller_first_name" -> "alice"
     * "caller_last_name" -> "smith"
     * }
     * </pre>
     * </p>
     *
     * NOTE: While the value field passed into <Parameter> gets URI encoded by the Twilio infrastructure
     * and URI decoded when parsed during the creation of a CallInvite, the name does not get URI encoded
     * or decoded. As a result, it is recommended that the name field only use ASCII characters.
     *
     */
    @NonNull public Map<String, String> getCustomParameters() {
        return customParameters;
    }

    String getBridgeToken() {
        return bridgeToken;
    }

    /**
     * Accepts the {@link CallInvite} with the provided {@link AcceptOptions} and returns a new
     * {@link Call}. A {@link SecurityException} will be thrown if RECORD_AUDIO is not granted.
     *
     * <p>{@link Call.Listener} receives the state of the {@link Call}.</p>
     *
     * <table border="1" summary="Call.Listener events.">
     * <tr>
     * <td>Callback Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnectFailure(Call, CallException)}</td><td>The call failed to connect. {@link CallException} provides details of the root cause.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onRinging(Call)}</td><td>This callback should not be invoked when calling {@link #accept(Context, AcceptOptions, Call.Listener)}</td><td>3.0.0-preview2</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnected(Call)}</td><td>The call has connected.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onDisconnected(Call, CallException)}</td><td>The call was disconnected. If the call ends due to an error the {@link CallException} is non-null. If the call ends normally {@link CallException} is null.</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * <p>If {@code accept} fails, {@link Call.Listener#onConnectFailure(Call, CallException)} callback is raised with {@link CallException}. {@link CallException#getMessage()} and {@link CallException#getExplanation()}
     * provide details of the failure.
     *
     * If {@link Call#disconnect()} is called while attempting to accept, the {@link Call.Listener#onDisconnected(Call, CallException)} callback will be raised with no error.
     * </p>
     *
     * <p>If {@link CallInvite#accept(Context, AcceptOptions, Call.Listener)} fails due to an authentication
     * error, the SDK receives the following error.</p>
     *
     * <table border="1" summary="Authentication Exceptions.">
     * <tr>
     * <td>Authentication Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTH_FAILURE}</td><td>20151</td><td>Twilio failed to authenticate the client</td>
     * </tr>
     * </table>
     *
     * <p>If {@link CallInvite#accept(Context, AcceptOptions, Call.Listener)} fails due to any other reason, the SDK receives one of the following errors.</p>
     *
     * <table border="1" summary="Call Exceptions.">
     * <tr>
     * <td>Call Exception</td><td>Error Code</td><td>Description</td>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CONNECTION_ERROR}</td><td>31005</td><td>Connection error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_CANCELLED}</td><td>31008</td><td>Call Cancelled</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TRANSPORT_ERROR}</td><td>31009</td><td>Transport error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MALFORMED_REQUEST}</td><td>31100</td><td>Malformed request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTHORIZATION_ERROR}</td><td>31201</td><td>Authorization error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TEMPORARILY_UNAVAILABLE_ERROR}</td><td>31480</td><td>Temporarily Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_OR_TRANSACTION_DOES_NOT_EXIST_ERROR}</td><td>31481</td><td>Call/Transaction Does Not Exist</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_HERE_ERROR}</td><td>31486</td><td>Busy Here</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TERMINATED_ERROR}</td><td>31487</td><td>Request Terminated</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DNS_RESOLUTION}</td><td>31530</td><td>DNS Resolution Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_EVERYWHERE_ERROR}</td><td>31600</td><td>Busy Everywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DECLINE_ERROR}</td><td>31603</td><td>Decline</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DOES_NOT_EXIST_ANYWHERE_ERROR}</td><td>31604</td><td>Does Not Exist Anywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED}</td><td>53001</td><td>Signaling connection disconnected</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_LOCAL_MEDIA_DESCRIPTION}</td><td>53400</td><td>Client is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_LOCAL_MEDIA_DESCRIPTION}</td><td>53401</td><td>Server is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_REMOTE_MEDIA_DESCRIPTION}</td><td>53402</td><td>Client is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_REMOTE_MEDIA_DESCRIPTION}</td><td>53403</td><td>Server is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NO_SUPPORTED_CODEC}</td><td>53404</td><td>No supported code</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MEDIA_CONNECTION_FAILED}</td><td>53405</td><td>Media connection failed</td>
     * </tr>
     * </table>
     *
     * <p>Insights :</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>accepted-by-local</td><td>Call state remains {@link Call.State#CONNECTING}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * <p>If accept fails, an error event is published to Insights.</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * @param context           An Android context.
     * @param acceptOptions     The options that configure the accepted call.
     * @param listener          A listener that receives call status.
     * @return A new {@link Call}
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * IceOptions iceOptions = new IceOptions.Builder()
     *         .iceTransportPolicy(IceTransportPolicy.RELAY)
     *         .build();
     * AcceptOptions acceptOptions = new AcceptOptions.Builder()
     *         .iceOptions(iceOptions)
     *         .build();
     * Call call = callInvite.accept(context, acceptOptions, new Call.Listener() {
     *     {@literal @}Override
     *      public void onRinging(@NonNull Call call) {
     *          // This callback will not be invoked when accepting a call invite
     *      }
     *
     *     {@literal @}Override
     *      public void onConnected(@NonNull final Call call) {
     *          Log.d(TAG, "Received onConnected " + call.getSid());
     *      }
     *
     *     {@literal @}Override
     *      public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
     *          Log.d(TAG, "Received onConnectFailure with CallException: " + callException.getErrorCode()+ ":" + callException.getMessage());
     *      }
     *
     *     {@literal @}Override
     *      public void onDisconnected(@NonNull Call call, CallException callException) {
     *          if (callException != null) {
     *              Log.d(TAG, "Received onDisconnected with CallException: " + callException.getMessage() + ": " + call.getSid());
     *          } else {
     *              Log.d(TAG, "Received onDisconnected");
     *          }
     *      }
     *  });
     *  }
     * </code></pre>
     */
    @NonNull public synchronized Call accept(@NonNull Context context,
                                             @NonNull AcceptOptions acceptOptions,
                                             @NonNull Call.Listener listener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(acceptOptions, "acceptOptions must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");
        if (!isAudioPermissionGranted(context)) {
            throw new SecurityException("Requires the RECORD_AUDIO permission");
        }

        AcceptOptions.Builder acceptOptionsBuilder = new AcceptOptions.Builder(this, false);

        // Pass the public options into the internal connect options
        if (acceptOptions.getIceOptions() != null) {
            acceptOptionsBuilder.iceOptions(acceptOptions.getIceOptions());
        }
        if (acceptOptions.getRegion() != null) {
            acceptOptionsBuilder.region(acceptOptions.getRegion());
        }
        if (acceptOptions.getPreferredAudioCodecs() != null) {
            acceptOptionsBuilder.preferAudioCodecs(acceptOptions.getPreferredAudioCodecs());
        }
        acceptOptionsBuilder.enableInsights(acceptOptions.enableInsights);

        LocalAudioTrack localAudioTrack = LocalAudioTrack.create(context, true);
        acceptOptionsBuilder.audioTracks(Collections.singletonList(localAudioTrack));
        acceptOptionsBuilder.eventListener(acceptOptions.getEventListener());
        AcceptOptions internalAcceptOptions = acceptOptionsBuilder.build();

        Call call = new Call(context.getApplicationContext(), this, listener);
        call.accept(internalAcceptOptions);
        return call;
    }

    /**
     * Accepts the {@link CallInvite} with default {@link AcceptOptions} and returns a new
     * {@link Call}. A {@link SecurityException} will be thrown if RECORD_AUDIO is not granted.
     *
     * <p>{@link Call.Listener} receives the state of the {@link Call}.</p>
     *
     * <table border="1" summary="Call.Listener events.">
     * <tr>
     * <td>Callback Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnectFailure(Call, CallException)}</td><td>The call failed to connect. {@link CallException} provides details of the root cause.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onRinging(Call)}</td><td>This callback should not be invoked when calling {@link #accept(Context, Call.Listener)}</td><td>3.0.0-preview2</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnected(Call)}</td><td>The call has connected.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onDisconnected(Call, CallException)}</td><td>The call was disconnected. If the call ends due to an error the {@link CallException} is non-null. If the call ends normally {@link CallException} is null.</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * <p>If {@code accept} fails, {@link Call.Listener#onConnectFailure(Call, CallException)} callback is raised with {@link CallException}. {@link CallException#getMessage()} and {@link CallException#getExplanation()}
     * provide details of the failure.
     *
     * If {@link Call#disconnect()} is called while attempting to accept, the {@link Call.Listener#onDisconnected(Call, CallException)} callback will be raised with no error.
     * </p>
     *
     * <p>If {@link CallInvite#accept(Context, Call.Listener)} fails due to an authentication
     * error, the SDK receives the following error.</p>
     *
     * <table border="1" summary="Authentication Exceptions.">
     * <tr>
     * <td>Authentication Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTH_FAILURE}</td><td>20151</td><td>Twilio failed to authenticate the client</td>
     * </tr>
     * </table>
     *
     * <p>If {@link CallInvite#accept(Context, Call.Listener)} fails due to any other reason, the SDK receives one of the following errors.</p>
     *
     * <table border="1" summary="Call Exceptions.">
     * <tr>
     * <td>Call Exception</td><td>Error Code</td><td>Description</td>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CONNECTION_ERROR}</td><td>31005</td><td>Connection error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_CANCELLED}</td><td>31008</td><td>Call Cancelled</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TRANSPORT_ERROR}</td><td>31009</td><td>Transport error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MALFORMED_REQUEST}</td><td>31100</td><td>Malformed request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTHORIZATION_ERROR}</td><td>31201</td><td>Authorization error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TEMPORARILY_UNAVAILABLE_ERROR}</td><td>31480</td><td>Temporarily Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_OR_TRANSACTION_DOES_NOT_EXIST_ERROR}</td><td>31481</td><td>Call/Transaction Does Not Exist</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_HERE_ERROR}</td><td>31486</td><td>Busy Here</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TERMINATED_ERROR}</td><td>31487</td><td>Request Terminated</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DNS_RESOLUTION}</td><td>31530</td><td>DNS Resolution Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_EVERYWHERE_ERROR}</td><td>31600</td><td>Busy Everywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DECLINE_ERROR}</td><td>31603</td><td>Decline</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DOES_NOT_EXIST_ANYWHERE_ERROR}</td><td>31604</td><td>Does Not Exist Anywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED}</td><td>53001</td><td>Signaling connection disconnected</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_LOCAL_MEDIA_DESCRIPTION}</td><td>53400</td><td>Client is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_LOCAL_MEDIA_DESCRIPTION}</td><td>53401</td><td>Server is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_REMOTE_MEDIA_DESCRIPTION}</td><td>53402</td><td>Client is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_REMOTE_MEDIA_DESCRIPTION}</td><td>53403</td><td>Server is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NO_SUPPORTED_CODEC}</td><td>53404</td><td>No supported code</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MEDIA_CONNECTION_FAILED}</td><td>53405</td><td>Media connection failed</td>
     * </tr>
     * </table>
     *
     * <p>Insights :</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>accepted-by-local</td><td>Call state remains {@link Call.State#CONNECTING}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * <p>If accept fails, an error event is published to Insights.</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * @param context     An Android context.
     * @param listener    A listener that receives call status.
     * @return A new {@link Call}
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * Call call = callInvite.accept(context, new Call.Listener() {
     *     {@literal @}Override
     *      public void onRinging(@NonNull Call call) {
     *          // This callback will not be invoked when accepting a call invite
     *      }
     *
     *     {@literal @}Override
     *      public void onConnected(@NonNull final Call call) {
     *          Log.d(TAG, "Received onConnected " + call.getSid());
     *      }
     *
     *     {@literal @}Override
     *      public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
     *          Log.d(TAG, "Received onConnectFailure with CallException: " + callException.getErrorCode()+ ":" + callException.getMessage());
     *      }
     *
     *     {@literal @}Override
     *      public void onDisconnected(@NonNull Call call, CallException callException) {
     *          if (callException != null) {
     *              Log.d(TAG, "Received onDisconnected with CallException: " + callException.getMessage() + ": " + call.getSid());
     *          } else {
     *              Log.d(TAG, "Received onDisconnected");
     *          }
     *      }
     *  });
     *  }
     * </code></pre>
     */
    @NonNull public synchronized Call accept(@NonNull Context context,
                                             @NonNull Call.Listener listener) {
        return accept(context, new AcceptOptions.Builder().build(), listener);
    }

    /**
     * Rejects the Call.
     */
    public synchronized void reject(@NonNull Context context) {
        reject(context, callListenerProxy, eventListenerProxy);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeStringArray(new String[]{from, to, callSid, bridgeToken});
        dest.writeInt(callInviteMessage.size());
        for(Map.Entry<String,String> entry : callInviteMessage.entrySet()){
            dest.writeString(entry.getKey());
            dest.writeString(entry.getValue());
        }
        if (customParameters != null) {
            dest.writeInt(customParameters.size());
            for (Map.Entry<String, String> entry : customParameters.entrySet()) {
                dest.writeString(entry.getKey());
                dest.writeString(entry.getValue());
            }
        }
    }

    public static final Parcelable.Creator CREATOR = new Parcelable.Creator() {
        public CallInvite createFromParcel(Parcel in) {
            return new CallInvite(in);
        }

        public CallInvite[] newArray(int size) {
            return new CallInvite[size];
        }
    };

    /*
     * Called by tests to validate internal callback behavior of reject
     */
    @VisibleForTesting
    void reject(@NonNull Context context, @NonNull Call.Listener listener, @NonNull Call.EventListener eventListener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(context, "listener must not be null");
        Preconditions.checkNotNull(context, "eventListener must not be null");

        AcceptOptions.Builder acceptOptionsBuilder = new AcceptOptions.Builder(this, true);
        AcceptOptions acceptOptions = acceptOptionsBuilder
                .eventListener(eventListener)
                .build();

        Call call = new Call(context.getApplicationContext(), this, listener);
        call.reject(acceptOptions);
    }

    static boolean isValid(Map<String, String> data) {
        final String messageType = data.get(VoiceConstants.VOICE_TWI_MESSAGE_TYPE);
        final String bridgeToken = data.get(VoiceConstants.BRIDGE_TOKEN);
        final String callSid = data.get(VoiceConstants.CALL_SID);
        final String to = data.get(VoiceConstants.TO);
        return messageType != null &&
                messageType.equals(VoiceConstants.MESSAGE_TYPE_CALL) &&
                bridgeToken != null &&
                callSid != null &&
                to != null;
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof CallInvite) {
            CallInvite callInvite = (CallInvite)o;
            // Validate equality based on the publicly visible state
            return getFrom().equals(callInvite.getFrom()) &&
                    getTo().equals(callInvite.getTo()) &&
                    getCallSid().equals(callInvite.getCallSid());
        } else {
            return false;
        }
    }

    void publishConnectionEvent(String eventName) {
        logger.d("Publishing event : " + eventName);
        EventPayload eventPayload = new EventPayload.Builder().callSid(callSid)
                .direction(Constants.Direction.INCOMING)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (this.publisher != null) {
                this.publisher.publish(com.twilio.voice.Constants.SeverityLevel.INFO,
                        EventGroupType.CONNECTION_EVENT_GROUP, eventName, connectionEventPayload);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
