package com.twilio.voice;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Represents options available when connecting to a {@link Call}.
 */
public abstract class CallOptions {
    private static final Set<Class> SUPPORTED_CODECS =
            new HashSet<Class>(
                    Arrays.asList(
                            // Audio codecs
                            OpusCodec.class,
                            PcmuCodec.class));

    List<LocalAudioTrack> audioTracks;
    IceOptions iceOptions;
    boolean enableInsights;
    List<AudioCodec> preferredAudioCodecs;
    PlatformInfo platformInfo;
    String region;
    Call.EventListener eventListener;

    static void checkAudioCodecs(@NonNull List<AudioCodec> audioCodecs) {
        for (AudioCodec audioCodec : audioCodecs) {
            Preconditions.checkNotNull(audioCodec);
            Preconditions.checkArgument(
                    SUPPORTED_CODECS.contains(audioCodec.getClass()),
                    String.format("Unsupported audio codec %s", audioCodec.getName()));
        }
    }

    @SuppressWarnings("unused")
    CallOptions() {
        // Package scoped constructor ensures the Javadoc does not show a default constructor
    }

    List<LocalAudioTrack> getAudioTracks() {
        return audioTracks;
    }

    /**
     * Returns the custom ICE configuration used to connect to a Call.
     */
    @Nullable public IceOptions getIceOptions() {
        return iceOptions;
    }

    /**
     * Returns the preferred audio codecs.
     */
    @Nullable public List<AudioCodec> getPreferredAudioCodecs() {
        return preferredAudioCodecs;
    }

    boolean isInsightsEnabled() {
        return enableInsights;
    }



    static void checkAudioTracksReleased(@Nullable List<LocalAudioTrack> audioTracks) {
        if (audioTracks != null) {
            for (LocalAudioTrack localAudioTrack : audioTracks) {
                Preconditions.checkState(
                        !localAudioTrack.isReleased(),
                        "LocalAudioTrack cannot be released");
            }
        }
    }

    LocalAudioTrack[] getLocalAudioTracksArray() {
        LocalAudioTrack[] audioTracksArray = new LocalAudioTrack[0];
        if (audioTracks.size() > 0) {
            audioTracksArray = new LocalAudioTrack[audioTracks.size()];
            audioTracksArray = audioTracks.toArray(audioTracksArray);
        }
        return audioTracksArray;
    }

    AudioCodec[] getAudioCodecsArray() {
        AudioCodec[] audioCodecsArray = new AudioCodec[0];
        if (preferredAudioCodecs != null && !preferredAudioCodecs.isEmpty()) {
            audioCodecsArray = new AudioCodec[preferredAudioCodecs.size()];
            audioCodecsArray = preferredAudioCodecs.toArray(audioCodecsArray);
        }
        return audioCodecsArray;
    }

    /**
     * Returns the region specified {@link CallOptions}.
     */
    @Nullable public String getRegion() {
        return region;
    }

    /**
     * Returns the eventListener specified.
     */
    Call.EventListener getEventListener() {
        return eventListener;
    }


    /**
     * Build new {@link CallOptions}.
     * <p>
     * <p>All methods are optional.</p>
     */
    public static abstract class Builder {
        IceOptions iceOptions;
        List<LocalAudioTrack> audioTracks;
        List<AudioCodec> preferredAudioCodecs;
        boolean enableInsights = true;
        String region = Constants.GLOBAL_LOW_LATENCY_REGION;
        Call.EventListener eventListener = null;

        @SuppressWarnings("unused")
        Builder() {
            // Package scoped constructor ensures the Javadoc does not show a default constructor
        }

        Builder audioTracks(@NonNull List<LocalAudioTrack> audioTracks) {
            Preconditions.checkNotNull(audioTracks, "audioTracks must not be null");
            this.audioTracks = new ArrayList<>(audioTracks);
            return this;
        }

        Builder iceOptions(@NonNull IceOptions iceOptions) {
            Preconditions.checkNotNull(iceOptions, "iceOptions must not be null");
            this.iceOptions = iceOptions;
            return this;
        }

        Builder enableInsights(boolean enable) {
            this.enableInsights = enable;
            return this;
        }

        Builder preferAudioCodecs(@NonNull List<AudioCodec> preferredAudioCodecs) {
            Preconditions.checkNotNull(preferredAudioCodecs, "preferredAudioCodecs must not be null");
            this.preferredAudioCodecs = new ArrayList<>(preferredAudioCodecs);
            return this;
        }

        Builder region(@NonNull String region) {
            Preconditions.checkNotNull(region, "region must not be null");
            this.region = region;
            return this;
        }

        Builder eventListener(Call.EventListener eventListener) {
            this.eventListener = eventListener;
            return this;
        }
    }
}
