package com.twilio.voice;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents options when connecting to a {@link Call}.
 */
public class ConnectOptions extends CallOptions {
    private final String accessToken;
    private final Map<String, String> params;

    /**
     * ConnectOptions can only be constructed via {@link ConnectOptions.Builder}
     */
    @SuppressWarnings("unused")
    private ConnectOptions() {
        // Make non-null to avoid warning
        accessToken = "";
        params = new HashMap<>();
    }

    private ConnectOptions(Builder builder) {
        this.accessToken = builder.accessToken;
        this.audioTracks = (builder.audioTracks != null) ?
                builder.audioTracks : new ArrayList<LocalAudioTrack>();
        this.iceOptions = builder.iceOptions;
        this.enableInsights = builder.enableInsights;
        this.preferredAudioCodecs = builder.preferredAudioCodecs;
        this.params = (builder.params == null) ? new HashMap<String, String>() : builder.params;
        this.platformInfo = new PlatformInfo();
        this.region = builder.region;
        this.eventListener = builder.eventListener;
        this.enableInsights = builder.enableInsights;
    }

    /**
     * The accessToken that provides the identity and grants of the caller.
     */
    @NonNull public String getAccessToken() {
        return accessToken;
    }

    /**
     * The parameters that are passed to the TwiML application specified by the access token.
     */
    @NonNull public Map<String, String> getParams() {
        return params;
    }

    /*
     * Invoked by JNI CallDelegate to get pointer to twilio::video::ConnectOptions::Builder
     */
    @SuppressWarnings("unused")
    private long createNativeConnectOptionsBuilder() {
        checkAudioTracksReleased(audioTracks);

        final boolean reject = false;

        String[] keys = new String[params.size()];
        String[] values = new String[params.size()];
        int index = 0;
        for (Map.Entry<String, String> mapEntry : params.entrySet()) {
            keys[index] = mapEntry.getKey();
            values[index] = mapEntry.getValue();
            index++;
        }

        String[] callInviteMessageKeys = new String[0];
        String[] callInviteMessageValues = new String[0];

        return nativeCreate(accessToken,
                keys,
                values,
                getLocalAudioTracksArray(),
                iceOptions,
                enableInsights,
                getAudioCodecsArray(),
                callInviteMessageKeys,
                callInviteMessageValues,
                reject,
                platformInfo,
                region);
    }

    private native long nativeCreate(String accessToken,
                                     String[] paramKeys,
                                     String[] paramValues,
                                     LocalAudioTrack[] audioTracks,
                                     IceOptions iceOptions,
                                     boolean enableInsights,
                                     AudioCodec[] preferredAudioCodecs,
                                     String[] incomingCallPushPayloadKeys,
                                     String[] incomingCallPushPayloadValues,
                                     boolean reject,
                                     PlatformInfo platformInfo,
                                     String region);

    /**
     * Build new {@link ConnectOptions}.
     *
     * <p>All methods are optional.</p>
     */
    public static class Builder extends CallOptions.Builder {
        private final String accessToken;
        private Map<String, String> params;
        /**
         * Use this Builder when making a {@link Call}
         *
         * <p>
         * The identity provided in the Access Token may only contain alpha-numeric and underscore characters. Other
         * characters, including spaces, will result in undefined behavior.
         * </p>
         *
         * @param accessToken The accessToken that provides the identity and grants of the caller in a JSON Web Token(JWT) format.
         */
        public Builder(@NonNull String accessToken) {
            Preconditions.checkNotNull(accessToken, "accessToken must not be null");
            this.accessToken = accessToken;
        }

        /**
         * Set the parameters that are passed to the TwiML application specified by the access token.
         *
         * <p>
         * Parameters specified in this map will be provided as HTTP GET query params or as part of the HTTP POST body
         * to the request url specified in your TwiML Application (https://www.twilio.com/console/voice/twiml/apps).
         * These parameters can be used to configure the behavior of your call within the context of your TwiML Application.
         *
         * NOTE: The Voice SDK URI encodes the parameters before passing them to your TwiML Application.
         * </p>
         */
        @NonNull public Builder params(@NonNull Map<String, String> params) {
            Preconditions.checkNotNull(params, "params must not be null");
            this.params = params;
            return this;
        }

        /**
         * Audio tracks that will be published upon connection.
         */
        @NonNull Builder audioTracks(@NonNull List<LocalAudioTrack> audioTracks) {
            Preconditions.checkNotNull(audioTracks, "audioTracks must not be null");
            super.audioTracks(audioTracks);
            return this;
        }

        /**
         * Custom ICE configuration used to connect to a Call.
         */
        @NonNull public Builder iceOptions(@NonNull IceOptions iceOptions) {
            Preconditions.checkNotNull(iceOptions, "iceOptions must not be null");
            super.iceOptions(iceOptions);
            return this;
        }

        /**
         * Specify reporting statistics to Insights. Sending stats data to Insights is enabled
         * by default.
         */
        @NonNull public Builder enableInsights(@NonNull boolean enable) {
            super.enableInsights(enable);
            return this;
        }

        /**
         * Set preferred audio codecs. The list specifies which audio codecs would be preferred when
         * negotiating audio with the backend service. The preferences are applied in the order found in
         * the list starting with the most preferred audio codec to the least preferred audio codec.
         * Audio codec preferences are not guaranteed to be satisfied because the backend service
         * is not guaranteed to support all audio codecs.
         *
         * <p>The following snippet demonstrates how to prefer a single audio codec.
         *
         * <pre><code>
         *     ConnectOptions connectOptions = new ConnectOptions.Builder(token)
         *          .preferAudioCodecs(Collections.<AudioCodec>singletonList(new PcmuCodec()))
         *          .build();
         * </code></pre>
         *
         * <p>The following snippet demonstrates how to specify the exact order of codec
         * preferences.
         *
         * <pre><code>
         *     ConnectOptions connectOptions = new ConnectOptions.Builder(token)
         *          .preferAudioCodecs(Arrays.asList(new PcmuCodec(), new OpusCodec()))
         *          .build();
         * </code></pre>
         */
        @NonNull public Builder preferAudioCodecs(@NonNull List<AudioCodec> preferredAudioCodecs) {
            Preconditions.checkNotNull(preferredAudioCodecs, "preferredAudioCodecs must not be null");
            checkAudioCodecs(preferredAudioCodecs);
            super.preferAudioCodecs(preferredAudioCodecs);
            return this;
        }

        /**
         * Sets the region (Twilio data center) for the SDK.
         * The default region uses Global Low Latency routing, which establishes a connection with
         * the closest region to the user.
         *
         * NOTE: Setting the region during a call will not apply until all ongoing calls have ended
         * and a subsequent call is placed.
         *
         * @param region The region.
         */
        @NonNull public Builder region(@NonNull String region) {
            Preconditions.checkNotNull(region, "region must not be null");
            super.region(region);
            return this;
        }

        Builder eventListener(Call.EventListener eventListener) {
            super.eventListener(eventListener);
            return this;
        }

        /**
         * Builds {@link ConnectOptions} object.
         */
        @NonNull public ConnectOptions build() {
            checkAudioTracksReleased(audioTracks);
            if (params != null) {
                for (Map.Entry<String, String> entry : params.entrySet()) {
                    Preconditions.checkState(entry.getKey() != null,"params entry key should not be null");
                    Preconditions.checkState(entry.getValue() != null, "params entry value should not be null");
                }
            }
            return new ConnectOptions(this);
        }
    }
}
