package com.twilio.voice;

import android.annotation.SuppressLint;

import com.twilio.voice.Constants.Direction;

import org.json.JSONArray;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

class EventPayload {
    boolean isPrivate = false;
    private String timeStamp;
    private String productName;
    private String callSid;
    private String payloadType;
    private long timestampMS;
    private JSONObject payload;
    private String clientName;
    private Direction direction;
    private List<RTCStatsSample> sampleList;
    private String qualityParam;
    private int qualityThreshold;
    private int sampleValue;
    private String tempCallSid;
    private int qualityScore;
    private String issueName;
    private int errorCode;
    private String errorMessage;
    private String requestId;
    private String selectedRegion;
    private String gateway;
    private String region;

    /**
     * An enum representing warning names that can occur during a call.
     */
    public enum WarningName {
        WARN_HIGH_JITTER("high-jitter"),
        WARN_HIGH_PACKET_LOSS("high-packet-loss"),
        WARN_HIGH_RTT("high-rtt"),
        WARN_LOW_MOS("low-mos"),
        WARN_CONSTANT_AUDIO_IN_LEVEL("constant-audio-input-level"),
        WARN_CONSTANT_AUDIO_OUT_LEVEL("constant-audio-output-level");

        private final String warningName;

        WarningName(String warningName) {
            this.warningName = warningName;
        }

        public String toString() {
            return this.warningName;
        }
    }

    private EventPayload(Builder builder) {
        this.productName = builder.productName;
        this.payload = builder.payload;
        this.payloadType = builder.payloadType;
        this.timeStamp = builder.timeStamp;
        this.timestampMS = builder.timestampMS;
        this.clientName = builder.clientName;
        this.direction = builder.direction;
        this.qualityParam = builder.qualityParam;
        this.qualityThreshold = builder.qualityThreshold;
        this.sampleValue = builder.sampleValue;
        this.sampleList = builder.sampleList;
        this.callSid = builder.callSid;
        this.tempCallSid = builder.tempCallSid;
        this.qualityScore = builder.qualityScore;
        this.issueName = builder.issue;
        this.errorCode = builder.errorCode;
        this.errorMessage = builder.errorMessage;
        this.requestId = builder.requestId;
        this.selectedRegion = builder.selectedRegion;
        this.gateway = builder.gateway;
        this.region = builder.region;
    }

    String getTimeStamp() {
        return this.timeStamp;
    }

    Object getPayloadType() {
        return this.payloadType;
    }

    String getProductName() {
        return this.productName;
    }

    boolean isPrivate() {
        return this.isPrivate;
    }

    String getCallSid() {
        return this.callSid;
    }

    long getTimestampMillis() {
        return this.timestampMS;
    }

    JSONObject getPayload() throws Exception {
        jsonPayloadPreparation();
        return this.payload;
    }

    String getClientName() {
        return this.clientName;
    }

    Direction getDirection() {
        return this.direction;
    }

    List<RTCStatsSample> getSampleList() {
        return this.sampleList;
    }

    String getQualityParam() {
        return this.qualityParam;
    }

    int getQualityThreshold() {
        return this.qualityThreshold;
    }

    int getSampleValue() {
        return this.sampleValue;
    }

    String getTempCallSid() {
        return this.tempCallSid;
    }

    int getQualityScore() {
        return this.qualityScore;
    }

    String getIssueName() {
        return this.issueName;
    }


    JSONObject jsonPayloadPreparation()
            throws Exception {
        JSONObject payload = new JSONObject();
        payload.put(EventKeys.TIMESTAMP_MS, this.timestampMS);
        payload.put(EventKeys.CLIENT_NAME, this.clientName);
        payload.put(EventKeys.TEMP_CALL_SID, this.tempCallSid);
        payload.put(EventKeys.CALL_SID, this.callSid);
        payload.put(EventKeys.SDK_VERSION_KEY, BuildConfig.VERSION_NAME);
        payload.put(EventKeys.PLATFORM, Constants.PLATFORM_ANDROID);
        payload.put(EventKeys.DIRECTION_KEY, this.direction);
        payload.put(EventKeys.SELECTED_REGION_KEY, this.selectedRegion);
        payload.put(EventKeys.GATEWAY, this.gateway);
        payload.put(EventKeys.REGION, this.region);
        if (this.errorCode > 0) {
            payload.put(EventKeys.ERROR_CODE, this.errorCode);
            payload.put(EventKeys.ERROR_MESSAGE, this.errorMessage);
        }
        if (this.requestId != null) {
            payload.put(EventKeys.REQUEST_ID_KEY, this.requestId);
        }

        JSONObject data = new JSONObject();
        if (this.sampleList != null) {
            data.put(EventKeys.THRESHOLD_KEY, this.qualityThreshold);
            JSONArray values = new JSONArray();
            for (RTCStatsSample sample : this.sampleList) {
                if (sample != null) {
                    if (this.qualityParam.compareTo(MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME) == 0) {
                        values.put(sample.getMosScore());
                    } else if (this.qualityParam.compareTo(MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME) == 0) {
                        values.put(sample.getJitter());
                    } else if (this.qualityParam.compareTo(MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME) == 0) {
                        values.put(sample.getRtt());
                    } else if (this.qualityParam
                            .compareTo(MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME) == 0) {
                        values.put(sample.getFractionLost());
                    }
                }
            }
            data.put(EventKeys.VALUES_KEY, values);
        } else if (this.sampleValue >= 0) {
            data.put(EventKeys.THRESHOLD_KEY, this.qualityThreshold);
            data.put(EventKeys.VALUE_KEY, this.sampleValue);
        } else {
            if (this.qualityScore > 0) {
                // Feedback event score
                payload.put(EventKeys.QUALITY_SCORE, this.qualityScore);
            }
            if (this.issueName != null && !this.issueName.equals(Call.Issue.NOT_REPORTED.toString())) {
                // Feedback event issue name
                payload.put(EventKeys.ISSUE_NAME, this.issueName);
            }
            // Connection Event and more that don't include sample value(s).
            data = null;
        }

        payload.put(EventKeys.DATA, data);
        this.payload = payload;
        return this.payload;
    }

    static class Builder {
        private String productName;
        private JSONObject payload;
        private String payloadType;
        private long timestampMS;
        private String timeStamp;
        private String callSid;
        private String clientName;
        private Direction direction;
        private List<RTCStatsSample> sampleList;
        private String qualityParam;
        private int qualityThreshold;
        private int sampleValue = -1;
        private String tempCallSid;
        private int qualityScore;
        private String issue;
        private int errorCode;
        private String errorMessage;
        private String requestId;
        private String selectedRegion;
        private String gateway;
        private String region;

        @SuppressLint("SimpleDateFormat")
        Builder() {
            TimeZone tz = TimeZone.getTimeZone("UTC");
            DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
            df.setTimeZone(tz);
            Date now = new Date();
            this.timeStamp = df.format(now);
            this.timestampMS = now.getTime();
        }

        Builder productName(String productName) {
            this.productName = productName;
            return this;
        }

        Builder payLoadType(String payloadType) {
            this.payloadType = payloadType;
            return this;
        }

        Builder callSid(String callSid) {
            this.callSid = callSid;
            return this;
        }

        Builder clientName(String clientName) {
            this.clientName = clientName;
            return this;
        }

        Builder direction(Direction direction) {
            this.direction = direction;
            return this;
        }

        Builder qualityParam(String qualityParam) {
            this.qualityParam = qualityParam;
            return this;
        }

        Builder qualityThreshold(int qualityThreshold) {
            this.qualityThreshold = qualityThreshold;
            return this;
        }

        Builder sampleList(List<RTCStatsSample> sampleList) {
            this.sampleList = sampleList;
            return this;
        }

        Builder sampleValue(int sampleValue) {
            this.sampleValue = sampleValue;
            return this;
        }

        Builder tempCallSid(String tempCallSid) {
            this.tempCallSid = tempCallSid;
            return this;
        }

        Builder score(Call.Score score) {
            if (score != null) {
                this.qualityScore = score.getValue();
            }
            return this;
        }

        Builder issue(Call.Issue issue) {
            if (issue != null) {
                this.issue = issue.toString();
            }
            return this;
        }

        Builder errorCode(int errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        Builder requestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        Builder selectedRegion(String selectedRegion) {
            this.selectedRegion = selectedRegion;
            return this;
        }

        Builder gateway(String gateway) {
            this.gateway = gateway;
            return this;
        }

        Builder region(String region) {
            this.region = region;
            return this;
        }

        EventPayload build() {

            if (this.productName == null) {
                throw new NullPointerException("productName must not be null");
            }

            if (this.payloadType == null) {
                throw new NullPointerException("payloadType must not be null");
            }

            return new EventPayload(this);
        }
    }
}
