package com.twilio.voice;

import android.content.Context;
import android.os.Handler;
import android.support.annotation.VisibleForTesting;

import com.twilio.voice.EventPayload.WarningName;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

abstract class InternalCall implements EventPublisher.EventPublisherListener {

    private static final Logger logger = Logger.getLogger(InternalCall.class);
    private static final String TEMP_CALL_SID_PREFIX = "TSID";
    String sid;
    String bridgeToken;
    String from;
    String to;
    String selectedRegion;
    String gateway;
    String region;
    Handler handler;
    Context context;
    boolean isMuted;
    boolean isOnHold;
    Call.State state;
    boolean disconnectCalled;
    EventPublisher publisher;
    RTCMonitor rtcMonitor;
    Constants.Direction direction;
    private JSONArray payload;
    private final UUID uuid = UUID.randomUUID();
    // Generate a temporary call SID for cases where one has not yet been assigned.
    private final String tempCallSid = TEMP_CALL_SID_PREFIX + uuid;

    @SuppressWarnings("unused")
    void setSid(String sid) {
        this.sid = sid;
    }

    boolean isValidState() {
        return state != Call.State.DISCONNECTED;
    }

    public abstract String getSid();

    public abstract void sendDigits(String digits);

    public abstract Call.State getState();

    public abstract boolean isMuted();

    public abstract void mute(boolean mute);

    public abstract void disconnect();

    @Override
    public void onError(VoiceException voiceException) {
        logger.e("Error publishing data : " + voiceException.getMessage() + ":" + voiceException.getErrorCode());
    }

    /**
     * This method gets called when a new stats becomes available.
     */
    public void onSample(RTCStatsSample currentSample) {
        currentSample.setCallSid(sid);
        currentSample.setDirection(direction);

        Map<WarningName, HashMap<String, Object>> warningsMap = rtcMonitor.monitor(currentSample, isMuted, isOnHold);
        processWarnings(warningsMap);

        if (this.payload == null) {
            this.payload = new JSONArray();
        }
        this.payload.put(currentSample.toJson());
        if (this.payload.length() == MetricEventConstants.BATCH_PAYLOAD_COUNT) {
            if (this.publisher != null) {
                try {
                    this.publisher.publishMetrics(EventGroupType.CALL_QUALITY_STATS_GROUP, EventType.CALL_METRIC_EVENT,
                            this.payload);

                } catch (Exception e) {
                    e.printStackTrace();
                }
                this.payload = new JSONArray();
            } else {
                // If the payload reaches the BATCH_PAYLOAD_COUNT but the
                // publisher is null, reset the payload to
                // eliminate buffer overflow.
                this.payload = new JSONArray();
            }
        }
    }

    public void onWarning(HashMap<String, Object> warningDetails) {
        String callSid = this.getSid();
        String groupType;

        WarningName warningName = (WarningName) warningDetails.get(WarningEventConstants.WarningEventKeys.WARNING_NAME);
        String warningParam = (String) warningDetails.get(WarningEventConstants.WarningEventKeys.WARNING_PARAM);
        int threshold = (int) warningDetails.get(WarningEventConstants.WarningEventKeys.THRESHOLD_KEY);
        int sampleValue = ((warningDetails.get(WarningEventConstants.WarningEventKeys.RECENT_SAMPLE_VALUE) == null) ? 0
                : (int) (warningDetails.get(WarningEventConstants.WarningEventKeys.RECENT_SAMPLE_VALUE)));
        List<RTCStatsSample> sampleList = (List<RTCStatsSample>) warningDetails
                .get(WarningEventConstants.WarningEventKeys.RECENT_SAMPLES);

        if (warningName.toString().compareTo(WarningName.WARN_CONSTANT_AUDIO_IN_LEVEL.toString()) == 0 ||
                warningName.toString().compareTo(WarningName.WARN_CONSTANT_AUDIO_OUT_LEVEL.toString()) == 0)
            groupType = EventGroupType.AUDIO_LEVEL_WARNING_RAISED;
        else {
            groupType = EventGroupType.NETWORK_QUALITY_WARNING_RAISED;
        }

        EventPayload warningEventPayload = new EventPayload.Builder()
                .callSid(callSid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to)).sampleList(sampleList).sampleValue(sampleValue)
                .qualityParam(warningParam).qualityThreshold(threshold)
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject eventPayload = warningEventPayload.getPayload();
            this.publisher.publish(com.twilio.voice.Constants.SeverityLevel.WARNING,
                    groupType, warningName.toString(), eventPayload);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void onWarningCleared(WarningName warningName) {
        // Clear the warningName calling ConnectionListener (new) callback and
        // publishing warning-cleared event to the backend.
        String callSid = this.getSid();
        String groupType;

        if (warningName.toString().compareTo(WarningName.WARN_CONSTANT_AUDIO_IN_LEVEL.toString()) == 0 ||
                warningName.toString().compareTo(WarningName.WARN_CONSTANT_AUDIO_OUT_LEVEL.toString()) == 0)
            groupType = EventGroupType.AUDIO_LEVEL_WARNING_CLEARED;
        else {
            groupType = EventGroupType.NETWORK_QUALITY_WARNING_CLEARED;
        }

        EventPayload warningClearedEventPayload = new EventPayload.Builder().callSid(callSid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();

        try {
            JSONObject eventPayload = warningClearedEventPayload.getPayload();
            this.publisher.publish(com.twilio.voice.Constants.SeverityLevel.INFO,
                    groupType, warningName.toString(), eventPayload);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishConnectionEvent(String eventName) {
        logger.d("Publishing event : " + eventName);
        EventPayload eventPayload = new EventPayload.Builder()
                .callSid(sid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (this.publisher != null) {
                this.publisher.publish(com.twilio.voice.Constants.SeverityLevel.INFO,
                        EventGroupType.CONNECTION_EVENT_GROUP, eventName, connectionEventPayload);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishConnectionErrorEvent(String eventName, int errorCode, String errorMessage) {
        logger.d("Publishing event : " + eventName);

        EventPayload eventPayload = new EventPayload.Builder().callSid(sid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .errorCode(errorCode)
                .errorMessage(errorMessage)
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (this.publisher != null) {
                this.publisher.publish(Constants.SeverityLevel.ERROR,
                        EventGroupType.CONNECTION_EVENT_GROUP, eventName, connectionEventPayload);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishIceGatheringEvent(String eventName) {
        logger.d("Publishing event : " + eventName);
        EventPayload eventPayload = new EventPayload.Builder()
                .callSid(sid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (this.publisher != null) {
                this.publisher.publish(Constants.SeverityLevel.DEBUG,
                        EventGroupType.ICE_GATHERING_STATE_GROUP, eventName, connectionEventPayload);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishIceConnectionEvent(String eventName) {
        logger.d("Publishing event : " + eventName);
        EventPayload eventPayload = new EventPayload.Builder()
                .callSid(sid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (this.publisher != null) {
                Constants.SeverityLevel severityLogLevel = (eventName.equals(EventType.ICE_CONNECTION_FAILED)) ?
                        Constants.SeverityLevel.ERROR : Constants.SeverityLevel.DEBUG;
                this.publisher.publish(severityLogLevel,
                        EventGroupType.ICE_CONNECTION_STATE_GROUP, eventName, connectionEventPayload);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishSignalingStateEvent(String eventName) {
        logger.d("Publishing event : " + eventName);
        EventPayload eventPayload = new EventPayload.Builder().callSid(sid)
                .tempCallSid(tempCallSid)
                .direction(direction)
                .selectedRegion(selectedRegion)
                .gateway(gateway)
                .region(region)
                .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                .clientName(Utils.parseClientIdentity(to))
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            if (this.publisher != null) {
                this.publisher.publish(com.twilio.voice.Constants.SeverityLevel.INFO,
                        EventGroupType.SIGNALING_STATE_GROUP, eventName, connectionEventPayload);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishFeedbackEvent(Call.Score score, Call.Issue issue) {
        logger.d("Publishing event feedback event");
        EventPayload eventPayload ;
        String eventName;
        if (score == Call.Score.NOT_REPORTED && issue == Call.Issue.NOT_REPORTED) {
            eventPayload = new EventPayload.Builder()
                    .callSid(sid)
                    .tempCallSid(tempCallSid)
                    .direction(direction)
                    .selectedRegion(selectedRegion)
                    .gateway(gateway)
                    .region(region)
                    .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                    .clientName(Utils.parseClientIdentity(to))
                    .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
            eventName = EventType.FEEDBACK_RECEIVED_NONE;
        } else {
            eventPayload = new EventPayload.Builder()
                    .callSid(sid)
                    .tempCallSid(tempCallSid)
                    .direction(direction)
                    .selectedRegion(selectedRegion)
                    .gateway(gateway)
                    .region(region)
                    .productName(com.twilio.voice.Constants.CLIENT_SDK_PRODUCT_NAME)
                    .score(score)
                    .issue(issue)
                    .clientName(Utils.parseClientIdentity(to))
                    .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOADTYPE).build();
            eventName = EventType.FEEDBACK_RECEIVED;
        }

        try {
            JSONObject connectionEventPayload = eventPayload.getPayload();
            this.publisher.publish(com.twilio.voice.Constants.SeverityLevel.INFO,
                    EventGroupType.FEEDBACK_EVENT_GROUP, eventName, connectionEventPayload);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void processWarnings(Map<WarningName, HashMap<String, Object>> warnings) {
        Iterator it = warnings.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry pair = (Map.Entry) it.next();
            HashMap<String, Object> warningDetails = (HashMap) pair.getValue();
            if (warningDetails.get(WarningEventConstants.WarningEventKeys.CLEAR_WARNING) != null) {
                onWarningCleared((WarningName) warningDetails.get(WarningEventConstants.WarningEventKeys.CLEAR_WARNING));
            } else {
                onWarning(warningDetails);
            }
        }
    }

    @VisibleForTesting
    EventPublisher getPublisher() {
        return this.publisher;
    }
}
