package com.twilio.voice;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;

import java.util.HashSet;
import java.util.Set;

class MediaFactory {
    private static final String RELEASE_MESSAGE_TEMPLATE = "MediaFactory released %s unavailable";
    private static final Logger logger = Logger.getLogger(MediaFactory.class);
    private volatile static MediaFactory instance;
    private volatile static Set<Object> mediaFactoryOwners = new HashSet<>();

    private long nativeMediaFactoryHandle;

    static MediaFactory instance(@NonNull Object owner, @NonNull Context context) {
        Preconditions.checkNotNull(owner, "owner must not be null");
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkApplicationContext(context);
        synchronized (MediaFactory.class) {
            if (instance == null) {
                Voice.loadLibrary(context);
                long nativeMediaFactoryHandle = nativeCreate(context);

                if (nativeMediaFactoryHandle == 0) {
                    logger.e("Failed to instance MediaFactory");
                } else {
                    instance = new MediaFactory(nativeMediaFactoryHandle);
                }
            }
            mediaFactoryOwners.add(owner);
        }

        return instance;
    }

    synchronized @Nullable
    LocalAudioTrack createAudioTrack(Context context,
                                     boolean enabled,
                                     @Nullable AudioOptions audioOptions,
                                     String name) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkApplicationContext(context,
                "must create local audio track with application context");
        Preconditions.checkState(nativeMediaFactoryHandle != 0,
                RELEASE_MESSAGE_TEMPLATE,
                "createAudioTrack");
        return nativeCreateAudioTrack(nativeMediaFactoryHandle,
                context,
                enabled,
                audioOptions,
                name);
    }

    void release(Object owner) {
        if (instance != null) {
            synchronized (MediaFactory.class) {
                mediaFactoryOwners.remove(owner);
                if (instance != null && mediaFactoryOwners.isEmpty()) {
                    // Release native media factory
                    nativeRelease(nativeMediaFactoryHandle);
                    nativeMediaFactoryHandle = 0;
                    instance = null;
                }
            }
        }
    }

    long getNativeMediaFactoryHandle() {
        return nativeMediaFactoryHandle;
    }

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    static boolean isReleased() {
        synchronized (MediaFactory.class) {
            return instance == null;
        }
    }

    /*
     * Manually release the media factory instance if a test did not properly release it.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    static void manualRelease() {
        synchronized (MediaFactory.class) {
            if (instance != null) {
                mediaFactoryOwners.clear();
                // Provide a dummy object to initiate release process
                Object o = new Object();
                instance.release(o);
            }
        }
    }

    private MediaFactory(long nativeMediaFactoryHandle) {
        this.nativeMediaFactoryHandle = nativeMediaFactoryHandle;
    }

    private static native long nativeCreate(Context context);

    private native LocalAudioTrack nativeCreateAudioTrack(long nativeMediaFactoryHandle,
                                                          Context context,
                                                          boolean enabled,
                                                          AudioOptions audioOptions,
                                                          String name);
    private native void nativeRelease(long mediaFactoryHandle);

    @VisibleForTesting(otherwise = VisibleForTesting.NONE)
    private native void nativeTestRelease(long mediaFactoryHandle);
}
