package com.twilio.voice;

import com.twilio.voice.EventPayload.WarningName;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The RTC Monitor class should monitor a MediaStream, and emit warnings when certain thresholds
 * have been crossed. Additionally, RTCMonitor notify listeners when a new sample is available. It
 * emits a stats sample to the customer that exposes all of the useful calculations,
 * such as MOS. Any object interested in receiving WARNING events needs to add itself as a lister.
 */
class RTCMonitor {
    private static final int WARNING_TIMEOUT_IN_MILLISECONDS = 5 * 1000;
    private ArrayList<RTCStatsSample> recentSamples = new ArrayList<>();
    private Map<WarningName, Date> activeWarnings = new HashMap<>();
    // Counters to keep track of sample quality
    private int constantAudioInputLevelSampleCounter;
    private int constantAudioOutputLevelSampleCounter;
    private int previousAudioInputLevel;
    private int previousAudioOutputLevel;

    static int SAMPLE_COUNT_METRICS = 5;
    static final int SAMPLE_COUNT_TO_RAISE_WARNING = 3;
    // DEFAULT THRESHOLD
    static MonitorThresholds thresholds = new MonitorThresholds.Builder()
            .jitterThreshold(MetricEventConstants.ThresholdsValue.MAX_JITTER_THRESHOLD)
            .mosScoreThreshhold(MetricEventConstants.ThresholdsValue.MIN_MOS_SCORE_THRESHOLD)
            .packetsLostFraction(MetricEventConstants.ThresholdsValue.MAX_PACKET_LOST_FRACTION)
            .rttThreshold(MetricEventConstants.ThresholdsValue.MAX_RTT_THRESHOLD)
            .audioConstantInputLevelSampleCounter(MetricEventConstants.ThresholdsValue.MAX_DURATION_CONSTANT_AUDIO_INPUT_LEVEL)
            .audioConstantOutputLevelSampleCounter(MetricEventConstants.ThresholdsValue.MAX_DURATION_CONSTANT_AUDIO_OUTPUT_LEVEL)
            .build();
    /**
     * EventListener
     */
    private List<Listener> listenerList = new ArrayList<>();

    /**
     * RTCMonitor Constructor that uses the default thresholds.
     *
     */
    RTCMonitor() {}

    /**
     * Method to add as a Listener.
     *
     * @param listener listener to receive Warning events and warning-clear events
     */
    void addListener(Listener listener) {
        listenerList.add(listener);
    }

    /**
     * Monitors stats quality.
     *
     * @param currentSample - latest stats sample collected
     */
    Map<WarningName, HashMap<String, Object>> monitor(final RTCStatsSample currentSample, boolean isMuted, boolean isOnHold) {
        Map<WarningName, HashMap<String, Object>> warnings = new HashMap();
        //Adjust the sample list
        this.recentSamples.add(currentSample);
        setCurrentSample(currentSample);
        if (this.recentSamples.size() > SAMPLE_COUNT_METRICS) {
            this.recentSamples = new ArrayList<>(this.recentSamples.subList(1, SAMPLE_COUNT_METRICS + 1));
        }

        if (!isOnHold) {
            if (!isMuted) {
                if (thresholds.getMaxDurationConstantAudioInputLevel() > -1) {
                    checkAudioLevel(warnings, currentSample.getAudioInputLevel(), this.previousAudioInputLevel,
                            thresholds.getMaxDurationConstantAudioInputLevel(),
                            WarningName.WARN_CONSTANT_AUDIO_IN_LEVEL);
                }
            }
            if (thresholds.getMaxDurationConstantAudioOutputLevel() > -1) {
                checkAudioLevel(warnings, currentSample.getAudioOutputLevel(), this.previousAudioOutputLevel,
                        thresholds.getMaxDurationConstantAudioOutputLevel(),
                        WarningName.WARN_CONSTANT_AUDIO_OUT_LEVEL);
            }
        }

        if (thresholds.getMinMosScoreThreshhold() > -1) {
            checkAudioQuality(warnings, MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME,
                    thresholds.getMinMosScoreThreshhold(), ComparisonType.MIN);
        }
        if (thresholds.getMaxJitterThreshold() > -1) {
            checkAudioQuality(warnings, MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME,
                    thresholds.getMaxJitterThreshold(), ComparisonType.MAX);
        }
        if (thresholds.getMaxRttThreshold() > -1) {
            checkAudioQuality(warnings, MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME,
                    thresholds.getMaxRttThreshold(), ComparisonType.MAX);
        }
        if (thresholds.getMaxPacketsLostFraction() > -1) {
            checkAudioQuality(warnings, MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME,
                    thresholds.getMaxPacketsLostFraction(),
                    ComparisonType.MAX);
        }

        return warnings;
    }

    /**
     * Method to detect constant Audio input/output level. If the audio input/output level is constant max duration threshold, raise warning.
     *
     * @param currentLevel  - Audio level of current sample
     * @param previousLevel - Audio level of previous sample
     * @param threshold     - the audio threshold
     * @param warningName   - name of the warning generated when threshold exceeds
     */
    private void checkAudioLevel(final Map<WarningName, HashMap<String, Object>> warnings, final int currentLevel, final int previousLevel, final int threshold, final WarningName warningName) {

        if (warningName == WarningName.WARN_CONSTANT_AUDIO_IN_LEVEL) {
            this.constantAudioInputLevelSampleCounter = (currentLevel == previousLevel) ? this.constantAudioInputLevelSampleCounter + 1 : 0;
            if (RTCMonitor.this.constantAudioInputLevelSampleCounter == threshold) {
                RTCMonitor.this.constantAudioInputLevelSampleCounter = 0;
                raiseWarningWithValue(warnings, WarningName.WARN_CONSTANT_AUDIO_IN_LEVEL, MetricEventConstants.Thresholds.AUDIO_INPUT_CONSTANT_MAX_DURATIOTN, thresholds.getMaxDurationConstantAudioInputLevel(), currentLevel);
            } else if (this.constantAudioInputLevelSampleCounter == 0) {
                clearWarning(warnings, WarningName.WARN_CONSTANT_AUDIO_IN_LEVEL);
            }
            this.previousAudioInputLevel = currentLevel;
        } else if (warningName == WarningName.WARN_CONSTANT_AUDIO_OUT_LEVEL) {
            this.constantAudioOutputLevelSampleCounter = (currentLevel == previousLevel) ? this.constantAudioOutputLevelSampleCounter + 1 : 0;
            if (this.constantAudioOutputLevelSampleCounter == threshold) {
                this.constantAudioOutputLevelSampleCounter = 0;
                raiseWarningWithValue(warnings, WarningName.WARN_CONSTANT_AUDIO_OUT_LEVEL, MetricEventConstants.Thresholds.AUDIO_OUTPUT_CONSTANT_MAX_DURATIOTN, thresholds.getMaxDurationConstantAudioOutputLevel(), currentLevel);
            } else if (this.constantAudioOutputLevelSampleCounter == 0) {
                clearWarning(warnings, WarningName.WARN_CONSTANT_AUDIO_OUT_LEVEL);
            }
            this.previousAudioOutputLevel = currentLevel;
        }
    }

    /**
     * Method to check Audio quality. Warning is raised when SAMPLE_COUNT_TO_RAISE_WARNING(3) of last 5 samples exceed/go below threshold.
     * Warning is cleared when 5 recent samples have good audio quality with above/below the threshold cut off.
     *
     * @param name      - name of the quality parameter we are comparing.
     * @param threshold - the audio threshold to compare with
     * @param type      - comparison type : min, max, maxConstantDuration
     */

    private void checkAudioQuality(final Map<WarningName, HashMap<String, Object>> warnings, final String name, final int threshold, ComparisonType type) {
        int counter = 0;
        if (type == ComparisonType.MAX) {
            counter = countHigh(name, threshold, this.recentSamples);
        } else if (type == ComparisonType.MIN) {
            counter = countLow(name, threshold, this.recentSamples);
        }

        if (counter == SAMPLE_COUNT_TO_RAISE_WARNING) {
            if (name.equals(MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME)) {
                raiseWarningWithSamples(warnings, WarningName.WARN_HIGH_JITTER, MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME, thresholds.getMaxJitterThreshold());
            } else if (name.equals(MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME)) {
                raiseWarningWithSamples(warnings, WarningName.WARN_HIGH_RTT, MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME, thresholds.getMaxRttThreshold());
            } else if (name.endsWith(MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME)) {
                raiseWarningWithSamples(warnings,WarningName.WARN_HIGH_PACKET_LOSS, MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME, thresholds.getMaxPacketsLostFraction());
            } else if (name.equals(MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME)) {
                raiseWarningWithSamples(warnings, WarningName.WARN_LOW_MOS, MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME, thresholds.getMinMosScoreThreshhold());
            }
        } else if (counter == 0) {
            //clear warnings, when countHigh or countLow returns zero.
            if (name.equals(MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME)) {
                clearWarning(warnings, WarningName.WARN_HIGH_JITTER);
            } else if (name.equals(MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME)) {
                clearWarning(warnings, WarningName.WARN_HIGH_RTT);
            } else if (name.equals(MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME)) {
                clearWarning(warnings, WarningName.WARN_HIGH_PACKET_LOSS);
            } else if (name.equals(MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME)) {
                clearWarning(warnings, WarningName.WARN_LOW_MOS);
            }
        }
    }

    /**
     * Method to set currentSample. Once this currentSample gets set all the
     * Observers get notified.
     *
     * @param sample
     */
    private void setCurrentSample(RTCStatsSample sample) {
        if (this.listenerList != null) {
            for (Listener listener : listenerList) {
                if (listener != null) {
                    listener.onSample(sample);
                }
            }
        }
    }

    /**
     * Method to raise warning with samples.
     *
     * @param warnings a map that contains warning details
     * @param nameOfTheWarning
     * @param warningParam
     * @param threshold
     */
    @SuppressWarnings("unchecked")
    private void raiseWarningWithSamples(Map<WarningName, HashMap<String, Object>> warnings, WarningName nameOfTheWarning, String warningParam, int threshold) {
        Date timeOfWarning = this.activeWarnings.get(nameOfTheWarning);
        // If there is an active warning, do not raise another one
        if (timeOfWarning != null) {
            return;
        }
        // Keep track of the warning and the time when it was raised.
        this.activeWarnings.put(nameOfTheWarning, new Date());
        HashMap<String, Object> warningDetails = new HashMap();
        warningDetails.put(WarningEventConstants.WarningEventKeys.WARNING_NAME, nameOfTheWarning);
        warningDetails.put(WarningEventConstants.WarningEventKeys.WARNING_PARAM, warningParam);
        warningDetails.put(WarningEventConstants.WarningEventKeys.THRESHOLD_KEY, threshold);
        warningDetails.put(WarningEventConstants.WarningEventKeys.RECENT_SAMPLES, this.recentSamples);

        warnings.put(nameOfTheWarning, warningDetails);
    }

    /**
     * Method to raise warning with value.
     *
     * @param nameOfTheWarning
     * @param warningParam
     * @param threshold
     * @param value
     */
    private void raiseWarningWithValue(Map<WarningName, HashMap<String, Object>> warnings, WarningName nameOfTheWarning, String warningParam, int threshold, int value) {
        Date timeOfWarning = this.activeWarnings.get(nameOfTheWarning);
        // If there is an active warning, do not raise another one
        if (timeOfWarning != null) {
            return;
        }
        this.activeWarnings.put(nameOfTheWarning, new Date());
        HashMap<String, Object> warningDetails = new HashMap();
        warningDetails.put(WarningEventConstants.WarningEventKeys.WARNING_NAME, nameOfTheWarning);
        warningDetails.put(WarningEventConstants.WarningEventKeys.WARNING_PARAM, warningParam);
        warningDetails.put(WarningEventConstants.WarningEventKeys.THRESHOLD_KEY, threshold);
        warningDetails.put(WarningEventConstants.WarningEventKeys.RECENT_SAMPLE_VALUE, value);

        warnings.put(nameOfTheWarning, warningDetails);
    }

    /**
     * Method to raise warning.
     *
     * @param nameOfTheWarning
     */
    private void clearWarning(Map<WarningName, HashMap<String, Object>> warnings, WarningName nameOfTheWarning) {
        // A warning can be cleared only after 5 seconds have passed since the warning was raised
        Date timeOfWarning = this.activeWarnings.get(nameOfTheWarning);
        if (timeOfWarning == null) {
            return;
        }
        int timeDiffInMillis = (int) (new Date().getTime() - timeOfWarning.getTime());
        if (timeDiffInMillis < RTCMonitor.WARNING_TIMEOUT_IN_MILLISECONDS) {
            return;
        }

        this.activeWarnings.remove(nameOfTheWarning);
        HashMap<String, Object> warningDetails = new HashMap<String, Object>();
        warningDetails.put(WarningEventConstants.WarningEventKeys.CLEAR_WARNING, nameOfTheWarning);

        warnings.put(nameOfTheWarning, warningDetails);
    }

    /**
     * Quality Calculation helper methods. This method compares 5 recent samples with specified threshold and increase the counter.
     * The counter is later compared with SAMPLE_COUNT_TO_RAISE_WARNING. If the counter is equal or more than SAMPLE_COUNT_TO_RAISE_WARNING
     * we raise a warning.
     *
     * @param name      - name of the quality parameter we are comparing.
     * @param threshold - the threshold he quality parameter need to compared against.
     * @param samples   - five recent samples.
     * @return count of low values
     */
    static int countLow(String name, final int threshold, ArrayList<RTCStatsSample> samples) {
        int lowCount = 0;
        int sampleValue = 0;
        for (RTCStatsSample sample : samples) {
            if (name.compareTo(MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME) == 0) {
                sampleValue = (int) sample.getMosScore();
            } else if (name.compareTo(MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME) == 0) {
                sampleValue = sample.getJitter();
            } else if (name.compareTo(MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME) == 0) {
                sampleValue = sample.getRtt();
            } else if (name.compareTo(MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME) == 0) {
                sampleValue = (int) sample.getFractionLost();
            }
            lowCount += (sampleValue < threshold) ? 1 : 0;
        }
        return lowCount;
    }

    /**
     * Quality Calculation helper methods. This method compares 5 recent samples with specified threshold and increase the counter.
     * The counter is later compared with SAMPLE_COUNT_TO_RAISE_WARNING. If the counter to equal to more than SAMPLE_COUNT_TO_RAISE_WARNING
     * we raise a warning.
     *
     * @param name      - name of the quality parameter we are comparing.
     * @param threshold - the threshold he quality parameter need to compared against.
     * @param samples   - five recent samples.
     * @return count of high values
     */
    static int countHigh(String name, final int threshold, ArrayList<RTCStatsSample> samples) {
        int highCount = 0;
        int sampleValue = 0;
        for (RTCStatsSample sample : samples) {
            if (name.compareTo(MetricEventConstants.Thresholds.MOS_THRESHOLD_NAME) == 0) {
                sampleValue = (int) sample.getMosScore();
            } else if (name.compareTo(MetricEventConstants.Thresholds.JITTER_THRESHOLD_NAME) == 0) {
                sampleValue = sample.getJitter();
            } else if (name.compareTo(MetricEventConstants.Thresholds.RTT_THRESHOLD_NAME) == 0) {
                sampleValue = sample.getRtt();
            } else if (name.compareTo(MetricEventConstants.Thresholds.PACKET_FRACTION_LOSS_MIN_THRESHOLD_NAME) == 0) {
                sampleValue = (int) sample.getFractionLost();
            }
            highCount += (sampleValue > threshold) ? 1 : 0;
        }
        return highCount;
    }

    enum ComparisonType {
        MIN, MAX, MAXCONSTANTDURATION
    }

    /**
     * Listener interface to receive Warning events and warning-clear events
     */
    interface Listener {
        /**
         * Called when stats is collected.
         *
         * @param sample - last sample collected in universal format.
         */
        void onSample(RTCStatsSample sample);

    }
}
