package com.twilio.voice;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.Nullable;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Collections;
import java.util.List;

import static android.content.pm.PackageManager.PERMISSION_GRANTED;

class Utils {
    /*
     * Return a handler on the thread looper, or the main thread looper if the calling thread
     * does not have a looper. If neither are available this handler will return null.
     */
    static Handler createHandler() {
        Handler handler = null;
        Looper looper;
        if ((looper = Looper.myLooper()) != null) {
            handler = new Handler(looper);
        } else if ((looper = Looper.getMainLooper()) != null) {
            handler = new Handler(looper);
        }
        if (handler == null) {
            throw new IllegalThreadStateException("This thread must be able to obtain a Looper");
        }

        return handler;
    }

    static boolean isAudioPermissionGranted(Context context) {
        int MICROPHONE_PERMISSION = context.checkCallingOrSelfPermission(Manifest.permission.RECORD_AUDIO);
        return MICROPHONE_PERMISSION == PERMISSION_GRANTED;
    }

    /**
     * Get IP address from first non-localhost interface
     *
     * @param useIPv4 true=return ipv4, false=return ipv6
     * @return address or empty string
     */
    public static String getIPAddress(boolean useIPv4) {
        try {
            List<NetworkInterface> interfaces = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface intf : interfaces) {
                List<InetAddress> addrs = Collections.list(intf.getInetAddresses());
                for (InetAddress addr : addrs) {
                    if (!addr.isLoopbackAddress()) {
                        String sAddr = addr.getHostAddress();
                        boolean isIPv4 = sAddr.indexOf(':') < 0;

                        if (useIPv4) {
                            if (isIPv4)
                                return sAddr;
                        } else {
                            if (!isIPv4) {
                                int delim = sAddr.indexOf('%'); // drop ip6 zone suffix
                                return delim < 0 ? sAddr.toUpperCase() : sAddr.substring(0, delim).toUpperCase();
                            }
                        }
                    }
                }
            }
        } catch (Exception ex) {
        } // for now eat exceptions
        return "";
    }

    static boolean permissionGranted(Context context, String permission) {
        int permissionCheck = context.checkCallingOrSelfPermission(permission);

        return permissionCheck == PackageManager.PERMISSION_GRANTED;
    }

    @Nullable static String parseClientIdentity(@Nullable String toField) {
        if (toField != null) {
            return toField.replaceFirst("^client:", "");
        }
        return null;
    }
}
