package com.twilio.voice;

import android.content.Context;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.util.Log;

import com.getkeepsafe.relinker.ReLinker;

import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import static com.twilio.voice.Utils.isAudioPermissionGranted;

public abstract class Voice {
    static AtomicInteger networkChangedCount = new AtomicInteger(0);
    static Map<LogModule, LogLevel> moduleLogLevel = new EnumMap(LogModule.class);
    private static final LogLevel defaultLogLevel = LogLevel.ERROR;
    static LogLevel level = defaultLogLevel;
    static boolean isLibraryLoaded = false;
    static final Set<Call> calls = new HashSet<>();
    static final Set<Call> rejects = new HashSet<>();

    public enum RegistrationChannel {
        FCM, GCM;

        @Override
        public String toString() {
            return name().toLowerCase();
        }
    }

    @SuppressWarnings("unused")
    Voice() {
        // Package scoped constructor ensures the Javadoc does not show a default constructor
    }

    /**
     * Register for incoming call messages. A successful registration will ensure that push
     * notifications will arrive via the GCM or FCM service for the lifetime of the registration
     * device token provided by the GCM or FCM service instance. When registration is successful,
     * {@code RegistrationListener.onRegistered(...)} callback is raised. Once successfully registered,
     * the registered binding has a time-to-live(TTL) of 1 year. If the registered binding is inactive
     * for 1 year, it is deleted and push notifications to the registered identity will not succeed.
     * However, whenever the registered binding is used to reach the registered identity, the TTL is
     * reset to 1 year.
     *
     * <p>
     * If registration fails, {@code RegistrationListener.onError(...)} callback is raised with
     * {@code RegistrationException}. {@code RegistrationException.getMessage(...)}
     * provides the root cause of the failure.
     * </p>
     *
     * <table border="1" summary="Registration Exceptions.">
     * <tr>
     * <td>Registration Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_ACCESS_TOKEN_REJECTED}</td><td>51007</td><td>Token authentication is rejected by authentication service</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REGISTRATION_ERROR}</td><td>31301</td><td>Registration failed. Look at RegistrationException.getMessage(...) for details</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request. The request could not be understood due to malformed syntax</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden. The server understood the request, but is refusing to fulfill it</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found. The server has not found anything matching the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout. A request timeout occurred</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_CONFLICT}</td><td>31409</td><td>Conflict. The request could not be processed because of a conflict in the current state of the resource. Another request may be in progress</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_UPGRADE_REQUIRED}</td><td>31426</td><td>Upgrade Required. The client should switch to a different protocol</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_TOO_MANY_REQUEST}</td><td>31429</td><td>Too Many Requests. Too many requests were sent in a given amount of time</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error. The server could not fulfill the request due to some unexpected condition</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway. The server is acting as a gateway or proxy, and received an invalid response from a downstream server while attempting to fulfill the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable. The server is currently unable to handle the request due to a temporary overloading or maintenance of the server</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout. The server, while acting as a gateway or proxy, did not receive a timely response from an upstream server</td>
     * </tr>
     *
     * </table>
     *
     * <p>
     * The identity provided in the {@code accessToken} may only contain alpha-numeric and
     * underscore characters. Other characters, including spaces, will result in undefined behavior.
     * </p>
     * Insights :
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>registration</td><td>Registration is successful</td><td>3.0.0-preview4</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>registration-error</td><td>Registration failed</td><td>3.0.0-preview4</td>
     * </tr>
     * </table>
     *
     * @param accessToken          An access token.
     * @param registrationChannel  An enumeration of registration channels.
     * @param registrationToken    A GCM or FCM registration token.
     * @param listener             A listener that receives registration request status.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre>
     * final String registrationToken = FirebaseInstanceId.getInstance().getToken();
     * if (registrationToken != null) {
     *     Voice.register(accessToken, Voice.RegistrationChannel.FCM, registrationToken, registrationListener);
     * }
     * </pre>
     */
    public static void register(@NonNull final String accessToken,
                                @NonNull final RegistrationChannel registrationChannel,
                                @NonNull final String registrationToken,
                                @NonNull final RegistrationListener listener) {
        Preconditions.checkNotNull(accessToken, "accessToken must not be null");
        Preconditions.checkNotNull(registrationChannel, "registrationChannel must not be null");
        Preconditions.checkNotNull(registrationToken, "registrationToken must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");
        Registrar registrar = new Registrar(accessToken, registrationChannel.toString(),
                registrationToken);
        registrar.register(listener);
    }

    /**
     * Unregister from receiving further incoming call messages. When unregistration is successful,
     * {@code UnregistrationListener.onUnregistered(...)} callback is raised.
     *
     * <p>
     * If unregistration fails, {@code UnregistrationListener.onError(...)} callback is raised with
     * {@code RegistrationException}. {@code RegistrationException.getMessage(...)}
     * provides the root cause of the failure.
     * </p>
     *
     * <table border="1" summary="Unregistration Exceptions.">
     * <tr>
     * <td>Registration Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_ACCESS_TOKEN_REJECTED}</td><td>51007</td><td>Token authentication is rejected by authentication service</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REGISTRATION_ERROR}</td><td>31301</td><td>Registration failed. Look at RegistrationException.getMessage(...) for details</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request. The request could not be understood due to malformed syntax</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden. The server understood the request, but is refusing to fulfill it</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found. The server has not found anything matching the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout. A request timeout occurred</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_CONFLICT}</td><td>31409</td><td>Conflict. The request could not be processed because of a conflict in the current state of the resource. Another request may be in progress</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_UPGRADE_REQUIRED}</td><td>31426</td><td>Upgrade Required. The client should switch to a different protocol</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_TOO_MANY_REQUEST}</td><td>31429</td><td>Too Many Requests. Too many requests were sent in a given amount of time</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error. The server could not fulfill the request due to some unexpected condition</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway. The server is acting as a gateway or proxy, and received an invalid response from a downstream server while attempting to fulfill the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable. The server is currently unable to handle the request due to a temporary overloading or maintenance of the server</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout. The server, while acting as a gateway or proxy, did not receive a timely response from an upstream server</td>
     * </tr>
     * </table>
     *
     * <p>
     * The identity provided in the {@code accessToken} may only contain alpha-numeric and
     * underscore characters. Other characters, including spaces, will result in undefined behavior.
     * </p>
     * Insights :
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>unregistration</td><td>Unregistration is successful</td><td>3.0.0-preview4</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>unregistration-error</td><td>Unregistration failed</td><td>3.0.0-preview4</td>
     * </tr>
     * </table>
     *
     * @param accessToken          An access token.
     * @param registrationChannel  An enumeration of registration channels.
     * @param registrationToken    An FCM or GCM device token.
     * @param listener             A listener that receives unregistration request status.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre>
     * Voice.unregister(accessToken, Voice.RegistrationChannel.FCM, registrationToken, unregistrationListener);
     * </pre>
     *
     */
    public static void unregister(@NonNull final String accessToken,
                                  @NonNull final RegistrationChannel registrationChannel,
                                  @NonNull final String registrationToken,
                                  @NonNull final UnregistrationListener listener) {
        Preconditions.checkNotNull(accessToken, "accessToken must not be null");
        Preconditions.checkNotNull(registrationChannel, "registrationChannel must not be null");
        Preconditions.checkNotNull(registrationToken, "registrationToken must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");
        Registrar registrar = new Registrar(accessToken, registrationChannel.toString(), registrationToken);
        registrar.unregister(listener);
    }

    /**
     * Creates and returns a new {@link Call}. A {@link SecurityException} will be thrown if RECORD_AUDIO
     * is not granted.
     *
     * <p>{@link Call.Listener} receives the state of the {@link Call}.</p>
     *
     * <table border="1" summary="Call.Listener events.">
     * <tr>
     * <td>Callback Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnectFailure(Call, CallException)}</td><td>The call failed to connect. {@link CallException} provides details of the root cause.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onRinging(Call)}</td><td>Emitted once before the {@link Call.Listener#onConnected(Call)} callback when the callee is being alerted of a {@link Call}.</td><td>3.0.0-preview2</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnected(Call)}</td><td>The call has connected.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onDisconnected(Call, CallException)}</td><td>The call was disconnected. If the call ends due to an error the `CallException` is non-null. If the call ends normally `CallException` is null.</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * <p>If {@code connect} fails, {@link Call.Listener#onConnectFailure(Call, CallException)} callback is raised with {@link CallException}. {@link CallException#getMessage()} and {@link CallException#getExplanation()}
     * provide details of the failure.
     *
     * If {@link Call#disconnect()} is called while attempting to connect, the {@link Call.Listener#onDisconnected(Call, CallException)} callback will be raised with no error.
     * </p>
     *
     * <p>If {@link Voice#connect(Context, ConnectOptions, Call.Listener)} fails due to an authentication error, the SDK receives one of the following errors.</p>
     *
     * <table border="1" summary="Authentication Exceptions.">
     * <tr>
     * <td>Authentication Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTH_FAILURE}</td><td>20151</td><td>Twilio failed to authenticate the client</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_APPLICATION_SID}</td><td>21218</td><td>Invalid ApplicationSid</td>
     * </tr>
     * </table>
     *
     * <p>If {@link Voice#connect(Context, ConnectOptions, Call.Listener)} fails due to any other reasons, the SDK receives one of the following errors.</p>
     *
     * <table border="1" summary="Call Exceptions.">
     * <tr>
     * <td>Call Exception</td><td>Error Code</td><td>Description</td>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CONNECTION_ERROR}</td><td>31005</td><td>Connection error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_CANCELLED}</td><td>31008</td><td>Call Cancelled</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TRANSPORT_ERROR}</td><td>31009</td><td>Transport error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MALFORMED_REQUEST}</td><td>31100</td><td>Malformed request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTHORIZATION_ERROR}</td><td>31201</td><td>Authorization error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TEMPORARILY_UNAVAILABLE_ERROR}</td><td>31480</td><td>Temporarily Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_OR_TRANSACTION_DOES_NOT_EXIST_ERROR}</td><td>31481</td><td>Call/Transaction Does Not Exist</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_HERE_ERROR}</td><td>31486</td><td>Busy Here</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TERMINATED_ERROR}</td><td>31487</td><td>Request Terminated</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DNS_RESOLUTION}</td><td>31530</td><td>DNS Resolution Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_EVERYWHERE_ERROR}</td><td>31600</td><td>Busy Everywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DECLINE_ERROR}</td><td>31603</td><td>Decline</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DOES_NOT_EXIST_ANYWHERE_ERROR}</td><td>31604</td><td>Does Not Exist Anywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED}</td><td>53001</td><td>Signaling connection disconnected</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_LOCAL_MEDIA_DESCRIPTION}</td><td>53400</td><td>Client is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_LOCAL_MEDIA_DESCRIPTION}</td><td>53401</td><td>Server is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_REMOTE_MEDIA_DESCRIPTION}</td><td>53402</td><td>Client is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_REMOTE_MEDIA_DESCRIPTION}</td><td>53403</td><td>Server is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NO_SUPPORTED_CODEC}</td><td>53404</td><td>No supported code</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MEDIA_CONNECTION_FAILED}</td><td>53405</td><td>Media connection failed</td>
     * </tr>
     * </table>
     *
     * <p>Insights :</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>outgoing</td><td>Outgoing call is made. Call state is in {@link Call.State#CONNECTING} state</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * <p>If connect fails, an error event is published to Insights.</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * @param context     An Android context.
     * @param connectOptions A set of options that allow you to configure your {@link Call}.
     * @param listener    A listener that receives call status.
     * @return A new {@link Call}
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * ConnectOptions connectOptions = new ConnectOptions.Builder(accessToken).build();
     * Call call = Voice.connect(context, connectOptions, new Call.Listener() {
     *     {@literal @}Override
     *      public void onRinging(@NonNull Call call) {
     *          Log.d(TAG, "Ringing");
     *      }
     *
     *     {@literal @}Override
     *      public void onConnected(@NonNull final Call call) {
     *          Log.d(TAG, "Received onConnected " + call.getSid());
     *      }
     *
     *     {@literal @}Override
     *      public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
     *          Log.d(TAG, "Received onConnectFailure with CallException: " + callException.getErrorCode()+ ":" + callException.getMessage());
     *      }
     *
     *     {@literal @}Override
     *      public void onDisconnected(@NonNull Call call, CallException callException) {
     *          if (callException != null) {
     *              Log.d(TAG, "Received onDisconnected with CallException: " + callException.getMessage() + ": " + call.getSid());
     *          } else {
     *              Log.d(TAG, "Received onDisconnected");
     *          }
     *      }
     *  });
     *  }
     * </code></pre>
     *
     */
    @NonNull public static Call connect(@NonNull final Context context,
                                        @NonNull final ConnectOptions connectOptions,
                                        @NonNull final Call.Listener listener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(connectOptions, "connectOptions must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");

        if (!isAudioPermissionGranted(context)) {
            throw new SecurityException("Requires the RECORD_AUDIO permission");
        }

        /*
         * Build a new ConnectOptions by transferring the public interface portions of the provided
         * ConnectOptions until the remainder of the options become public.
         */
        ConnectOptions.Builder builder = new ConnectOptions.Builder(connectOptions.getAccessToken());
        builder.params(connectOptions.getParams());
        if (connectOptions.getIceOptions() != null) {
            builder.iceOptions(connectOptions.getIceOptions());
        }
        if (connectOptions.getRegion() != null) {
            builder.region(connectOptions.getRegion());
        }
        if (connectOptions.getPreferredAudioCodecs() != null) {
            builder.preferAudioCodecs(connectOptions.getPreferredAudioCodecs());
        }
        builder.enableInsights(connectOptions.enableInsights);

        LocalAudioTrack localAudioTrack = LocalAudioTrack.create(context, true);
        builder.audioTracks(Collections.singletonList(localAudioTrack));
        builder.eventListener(connectOptions.getEventListener());
        ConnectOptions internalConnectOptions = builder.build();

        final Call call = new Call(context.getApplicationContext(), internalConnectOptions.getAccessToken(), listener);
        call.connect(internalConnectOptions);

        return call;
    }

    /**
     * Creates and returns a new {@link Call}. A {@link SecurityException} will be thrown if RECORD_AUDIO
     * is not granted.
     *
     * <p>{@link Call.Listener} receives the state of the {@link Call}.</p>
     *
     * <table border="1" summary="Call.Listener events.">
     * <tr>
     * <td>Callback Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnectFailure(Call, CallException)}</td><td>The call failed to connect. {@link CallException} provides details of the root cause.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onRinging(Call)}</td><td>Emitted once before the {@link Call.Listener#onConnected(Call)} callback when the callee is being alerted of a {@link Call}.</td><td>3.0.0-preview2</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnected(Call)}</td><td>The call has connected.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onDisconnected(Call, CallException)}</td><td>The call was disconnected. If the call ends due to an error the `CallException` is non-null. If the call ends normally `CallException` is null.</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * <p>If {@code connect} fails, {@link Call.Listener#onConnectFailure(Call, CallException)} callback is raised with {@link CallException}. {@link CallException#getMessage()} and {@link CallException#getExplanation()}
     * provide details of the failure.
     *
     * If {@link Call#disconnect()} is called while attempting to connect, the {@link Call.Listener#onDisconnected(Call, CallException)} callback will be raised with no error.
     * </p>
     *
     * <p>If {@link Voice#connect(Context, String, Call.Listener)} fails due to an authentication error, the SDK receives one of the following errors.</p>
     *
     * <table border="1" summary="Authentication Exceptions.">
     * <tr>
     * <td>Authentication Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTH_FAILURE}</td><td>20151</td><td>Twilio failed to authenticate the client</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_APPLICATION_SID}</td><td>21218</td><td>Invalid ApplicationSid</td>
     * </tr>
     * </table>
     *
     * <p>If {@link Voice#connect(Context, String, Call.Listener)} fails due to any other reason, the SDK receives one of the following errors.</p>
     *
     * <table border="1" summary="Call Exceptions.">
     * <tr>
     * <td>Call Exception</td><td>Error Code</td><td>Description</td>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CONNECTION_ERROR}</td><td>31005</td><td>Connection error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_CANCELLED}</td><td>31008</td><td>Call Cancelled</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TRANSPORT_ERROR}</td><td>31009</td><td>Transport error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MALFORMED_REQUEST}</td><td>31100</td><td>Malformed request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTHORIZATION_ERROR}</td><td>31201</td><td>Authorization error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TEMPORARILY_UNAVAILABLE_ERROR}</td><td>31480</td><td>Temporarily Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_OR_TRANSACTION_DOES_NOT_EXIST_ERROR}</td><td>31481</td><td>Call/Transaction Does Not Exist</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_HERE_ERROR}</td><td>31486</td><td>Busy Here</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TERMINATED_ERROR}</td><td>31487</td><td>Request Terminated</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DNS_RESOLUTION}</td><td>31530</td><td>DNS Resolution Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_EVERYWHERE_ERROR}</td><td>31600</td><td>Busy Everywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DECLINE_ERROR}</td><td>31603</td><td>Decline</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DOES_NOT_EXIST_ANYWHERE_ERROR}</td><td>31604</td><td>Does Not Exist Anywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED}</td><td>53001</td><td>Signaling connection disconnected</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_LOCAL_MEDIA_DESCRIPTION}</td><td>53400</td><td>Client is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_LOCAL_MEDIA_DESCRIPTION}</td><td>53401</td><td>Server is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_REMOTE_MEDIA_DESCRIPTION}</td><td>53402</td><td>Client is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_REMOTE_MEDIA_DESCRIPTION}</td><td>53403</td><td>Server is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NO_SUPPORTED_CODEC}</td><td>53404</td><td>No supported code</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MEDIA_CONNECTION_FAILED}</td><td>53405</td><td>Media connection failed</td>
     * </tr>
     * </table>
     *
     * <p>
     * The identity provided in the {@code accessToken} may only contain alpha-numeric and
     * underscore characters. Other characters, including spaces, will result in undefined behavior.
     * </p>
     *
     * <p>Insights :</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>outgoing</td><td>Outgoing call is made. Call state is in {@link Call.State#CONNECTING} state</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * <p>If connect fails, an error event is published to Insights.</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * @param context     An Android context.
     * @param accessToken The accessToken that provides the identity and grants of the caller.
     * @param listener    A listener that receives call status.
     * @return A new {@link Call}
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * Call call = Voice.connect(context, accessToken, new Call.Listener() {
     *     {@literal @}Override
     *      public void onRinging(@NonNull Call call) {
     *          Log.d(TAG, "Ringing");
     *      }
     *
     *     {@literal @}Override
     *      public void onConnected(@NonNull final Call call) {
     *          Log.d(TAG, "Received onConnected " + call.getSid());
     *      }
     *
     *     {@literal @}Override
     *      public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
     *          Log.d(TAG, "Received onConnectFailure with CallException: " + callException.getErrorCode()+ ":" + callException.getMessage());
     *      }
     *
     *     {@literal @}Override
     *      public void onDisconnected(@NonNull Call call, CallException callException) {
     *          if (callException != null) {
     *              Log.d(TAG, "Received onDisconnected with CallException: " + callException.getMessage() + ": " + call.getSid());
     *          } else {
     *              Log.d(TAG, "Received onDisconnected");
     *          }
     *      }
     *  });
     *  }
     * </code></pre>
     */
    @NonNull public static Call connect(@NonNull final Context context,
                                        @NonNull final String accessToken,
                                        @NonNull final Call.Listener listener) {
        return connect(context, new ConnectOptions.Builder(accessToken).build(), listener);
    }

    /**
     * Handle messages from FCM.
     *
     * <p>
     *     Twilio sends two types of notifications messages via GCM/FCM, {@code call} and {@code cancel} messages.
     *     The message type is encoded in the dictionary with the key {@code twi_message_type} and the values {@code twilio.voice.call} and {@code twilio.voice.cancel}.
     * </p>
     * <p>
     *     A {@code call} message is sent when someone wants to reach the registered {@code identity}.
     *     Passing a {@code call} message into {@link Voice#handleMessage(Map, MessageListener)} will result in a {@link CallInvite} raised in the {@link MessageListener} and return {@code true}.
     * </p>
     * <p>
     *     A {@code cancel} message is sent to an {@code identity} for the following reasons :
     * </p>
     * <ul>
     * <li> call made to this {@code identity} is prematurely {@code disconnected} by the caller </li>
     * <li> call is {@code rejected} </li>
     * <li> call is {@code ignored}</li>
     * <li> after a call is {@code accepted}. When a call is accepted, a {@code cancel} notification is sent to all the devices that have been registered with the {@code identity}
      accepting the call, even to the device that accepted the call. It is recommended that in this case the developer checks
      whether the call sid of {@code CancelledCallInvite} matches the accepted call, and if so disregard it.</li>
     * </ul>
     * <p>
     *     Passing a {@code cancel} message into {@link Voice#handleMessage(Map, MessageListener)} will result in a {@link CancelledCallInvite} raised in the {@link MessageListener} and return {@code true}.
     * </p>
     * <p>
     *     Passing malformed data to {@link Voice#handleMessage(Map, MessageListener)} will return {@code false} and no {@link MessageListener} callback will be raised.
     * </p>
     *
     * <p>Insights :</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>incoming</td><td>Incoming call notification received</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * @param data      Push notification payload.
     * @param listener  A {@link MessageListener} to receive incoming push notification callbacks.
     * @return          A boolean value that indicates whether the payload is a valid notification sent by Twilio.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * boolean valid = Voice.handleMessage(dataMap, new MessageListener() {
     *     {@literal @}Override
     *      public void onCallInvite(@NonNull CallInvite callInvite) {
     *          Log.d(TAG, "Received CallInvite");
     *      }
     *
     *     {@literal @}Override
     *      public void onCancelledCallInvite(@NonNull CancelledCallInvite cancelledCallInvite) {
     *          Log.d(TAG, "Received CancelledCallInvite");
     *      }
     * });
     * </code></pre>
     */
    public static synchronized boolean handleMessage(@NonNull final Map<String, String> data,
                                                     @NonNull final MessageListener listener) {
        Preconditions.checkNotNull(data, "data must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");

        if (CallInvite.isValid(data)) {
            final CallInvite callInvite = CallInvite.create(data);
            listener.onCallInvite(callInvite);
            return true;
        } else if (CancelledCallInvite.isValid(data)) {
            CancelledCallInvite cancelledCallInvite = CancelledCallInvite.create(data);
            listener.onCancelledCallInvite(cancelledCallInvite);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Handle messages from GCM
     *
     * <p>
     *     Twilio sends two types of notifications messages via GCM/FCM, {@code call} and {@code cancel} messages.
     *     The message type is encoded in the dictionary with the key {@code twi_message_type} and the values {@code twilio.voice.call} and {@code twilio.voice.cancel}.
     * </p>
     * <p>
     *     A {@code call} message is sent when someone wants to reach the registered {@code identity}.
     *     Passing a {@code call} message into {@link Voice#handleMessage(Bundle, MessageListener)} will result in a {@link CallInvite} raised in the {@link MessageListener} and return {@code true}.
     * </p>
     <p>
     *     A {@code cancel} message is sent to an {@code identity} for the following reasons :
     * </p>
     * <ul>
     * <li> call made to this {@code identity} is prematurely {@code disconnected} by the caller </li>
     * <li> call is {@code rejected} </li>
     * <li> call is {@code ignored}</li>
     * <li> after a call is {@code accepted}. When a call is accepted, a {@code cancel} notification is sent to all the devices that have been registered with the {@code identity}
     accepting the call, even to the device that accepted the call. It is recommended that in this case the developer checks
     whether the call sid of {@code CancelledCallInvite} matches the accepted call, and if so disregard it.</li>
     * </ul>
     * <p>
     *     Passing a {@code cancel} message into {@link Voice#handleMessage(Bundle, MessageListener)} will result in a {@link CancelledCallInvite} raised in the {@link MessageListener} and return {@code true}.
     * </p>
     * <p>
     *     Passing malformed data to {@link Voice#handleMessage(Bundle, MessageListener)} will return {@code false} and no {@link MessageListener} callback will be raised.
     * </p>
     *
     * <p>Insights :</p>
     *
     * <table border="1" summary="Insights events.">
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>incoming</td><td>Incoming call notification received</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * @param data      Push notification payload.
     * @param listener  A {@link MessageListener} to receive incoming push notification callbacks.
     * @return          A boolean value that indicates whether the payload is a valid notification sent by Twilio.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * boolean valid = Voice.handleMessage(dataBundle, new MessageListener() {
     *     {@literal @}Override
     *      public void onCallInvite(@NonNull CallInvite callInvite) {
     *          Log.d(TAG, "Received CallInvite");
     *      }
     *
     *     {@literal @}Override
     *      public void onCancelledCallInvite(@NonNull CancelledCallInvite cancelledCallInvite) {
     *          Log.d(TAG, "Received CancelledCallInvite");
     *      }
     * });
     * </code></pre>
     */
    public static boolean handleMessage(@NonNull final Bundle data,
                                        @NonNull final MessageListener listener) {
        Preconditions.checkNotNull(data, "data must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");

        Map<String, String> dataMap = new HashMap<>();
        for(String key : data.keySet()) {
            dataMap.put(key, String.valueOf(data.get(key)));
        }
        return handleMessage(dataMap, listener);
    }

    /**
     * Returns the version of the Voice SDK.
     *
     * @return The version of the SDK.
     */
    @NonNull public static String getVersion() {
        return BuildConfig.VERSION_NAME;
    }

    /**
     * Returns the logging level for messages logged by the Voice SDK. The default level is
     * {@link LogLevel#ERROR}.
     *
     * @return the logging level
     */
    @NonNull public static LogLevel getLogLevel() {
        return Voice.level;
    }

    /**
     * Returns the logging level for messages logged by the specified LogModule. The default log
     * level for each module is {@link LogLevel#ERROR}.
     *
     * @return the logging level
     */
    @NonNull public static LogLevel getModuleLogLevel(LogModule module) {
        if (Voice.moduleLogLevel.containsKey(module)) {
            return Voice.moduleLogLevel.get(module);
        }
        return defaultLogLevel;
    }

    /**
     * Sets the logging level for messages logged by the Voice SDK.
     *
     * @param level The logging level
     */
    public static void setLogLevel(@NonNull LogLevel level) {
        setSDKLogLevel(level);
        if (isLibraryLoaded) {
            nativeSetModuleLevel(LogModule.CORE.ordinal(), level.ordinal());
        }
        // Save the log level
        Voice.level = level;
    }

    /**
     * Sets the logging level for messages logged by a specific module.
     *
     * @param module The module for this log level.
     * @param level  The logging level.
     */
    public static void setModuleLogLevel(@NonNull LogModule module, @NonNull LogLevel level) {
        if (module == LogModule.PLATFORM) {
            setSDKLogLevel(level);
        }
        if (isLibraryLoaded) {
            nativeSetModuleLevel(module.ordinal(), level.ordinal());
        }
        // Save the module log level
        Voice.moduleLogLevel.put(module, level);
    }

    private static void setSDKLogLevel(LogLevel level) {
         /*
         * The Log Levels are defined differently in the Logger
         * which is based off android.util.Log.
         */
        switch (level) {
            case OFF:
                Logger.setLogLevel(Log.ASSERT);
                break;
            case ERROR:
                Logger.setLogLevel(Log.ERROR);
                break;
            case WARNING:
                Logger.setLogLevel(Log.WARN);
                break;
            case INFO:
                Logger.setLogLevel(Log.INFO);
                break;
            case DEBUG:
                Logger.setLogLevel(Log.DEBUG);
                break;
            case TRACE:
                Logger.setLogLevel(Log.VERBOSE);
                break;
            case ALL:
                Logger.setLogLevel(Log.VERBOSE);
                break;
            default:
                // Set the log level to assert/disabled if the value passed in is unknown
                Logger.setLogLevel(Log.ASSERT);
                break;
        }
    }

    static void onNetworkChanged(NetworkChangeEvent networkChangeEvent) {
        networkChangedCount.incrementAndGet();
        for (Call call : calls) {
            call.networkChange(networkChangeEvent);
        }
    }

    static void loadLibrary(Context context) {
        if (!Voice.isLibraryLoaded) {
            ReLinker.loadLibrary(context, "jingle_peerconnection_so");
            Voice.isLibraryLoaded = true;
            /*
             * The user may have set the log level prior to the native library being loaded.
             * Attempt to set the core log level now that the native library has loaded.
             */
            Voice.setLogLevel(Voice.level);
            /*
             * It is possible that the user has tried to set the log level for a specific module
             * before the library has loaded. Here we apply the log level for the module because we
             * know the native library is available
             */
            for (LogModule module : Voice.moduleLogLevel.keySet()) {
                Voice.setModuleLogLevel(module, Voice.moduleLogLevel.get(module));
            }
        }
    }

    enum NetworkChangeEvent {
        CONNECTION_LOST,
        CONNECTION_CHANGED
    }

    private native static void nativeSetModuleLevel(int module, int level);
}
