package com.twilio.voice;

import android.support.annotation.Nullable;

import java.io.IOException;
import java.net.URL;

import javax.net.ssl.HttpsURLConnection;

class VoiceURLConnection {
    private static final Logger logger = Logger.getLogger(VoiceURLConnection.class);

    public static final String METHOD_TYPE_POST = "POST";
    public static final String METHOD_TYPE_DELETE = "DELETE";

    private static final String HEADER_CONTENT_TYPE_KEY = "Content-Type";
    private static final String HEADER_CONTENT_TYPE_VALUE = "application/json";
    private static final String HEADER_ACCEPT_KEY = "Accept";
    private static final String HEADER_TWILIO_TOKEN_KEY = "X-Twilio-Token";
    /**
     * HTTP Status-Code 426: Upgrade Required.
     */
    static final int HTTP_UPGRADE_REQUIRED = 426;

    /**
     * HTTP Status-Code 429: Too Many Requests.
     */
    static final int HTTP_TOO_MANY_REQUEST = 429;

    static final int CONNECTION_TIMEOUT = 10000;
    static final int SOCKET_READ_TIMEOUT = 10000;

    private VoiceURLConnection() {
    }

    static HttpsURLConnection create(final String accessToken,
                                            final String hostURL,
                                            final String requestMethodType)
            throws IOException {
        URL url = new URL(hostURL);
        HttpsURLConnection urlConnection = (HttpsURLConnection) url.openConnection();
        if (requestMethodType.equals(METHOD_TYPE_DELETE)) {
            /*
             * Disable setDoOutput for DELETE requests due to an issue that results in the following
             * error: "DELETE does not support writing". This issue occurs on some devices on
             * Android API level 18 and below. For reference see https://stackoverflow.com/q/10338615
             */
            urlConnection.setDoOutput(false);
        } else {
            urlConnection.setDoOutput(true);
        }
        urlConnection.setDoInput(true);
        urlConnection.setRequestProperty(HEADER_CONTENT_TYPE_KEY, HEADER_CONTENT_TYPE_VALUE);
        urlConnection.setRequestProperty(HEADER_ACCEPT_KEY, HEADER_CONTENT_TYPE_VALUE);
        urlConnection.setRequestProperty(HEADER_TWILIO_TOKEN_KEY, accessToken);
        urlConnection.setConnectTimeout(CONNECTION_TIMEOUT);
        urlConnection.setReadTimeout(SOCKET_READ_TIMEOUT);
        urlConnection.setRequestMethod(requestMethodType);
        return urlConnection;
    }

    /*
     * Close all streams and disconnect URLConnection.
     *
     * All exceptions are swallowed to avoid excessive logging. This is intentional because
     * the purpose of this method is to ensure all resources associated with URLConnection are
     * released.
     *
     * TODO: Improve URLConnection management and usage CLIENT-5634
     */
    static void release(@Nullable HttpsURLConnection urlConnection) {
        if (urlConnection != null) {
            try {
                urlConnection.getInputStream().close();
            } catch (Exception e) {
                // No op
            }
            try {
                urlConnection.getOutputStream().close();
            } catch (Exception e) {
                // No op
            }
            try {
                urlConnection.getErrorStream().close();
            } catch (Exception e) {
                // No op
            }
            urlConnection.disconnect();
        }
    }
}
