package com.twilio.voice;

import android.util.Base64;

import java.util.HashMap;

/**
 * This class parses Access Tokens.  These are JWTs (JSON Web Tokens) that are
 * signed and broken out as so:
 * <p/>
 * HEADER.PAYLOAD.HASHED_SIGNATURE
 * <p/>
 * which has been base64-encoded.
 * <p/>
 * For example:
 * {"typ":"JWT","alg":"HS256"} . {"iss":"ACblahblahblahblah","exp":1317494391,"scope":"scope:client:incoming?clientName=rob scope:client:outgoing?clientName=rob&appSid=APblahblahblah"} . SIG
 */
class AccessTokenParser {
    private static final Logger logger = Logger.getLogger(AccessTokenParser.class);
    final String twilioVoiceHomeRegionSpecifier = "twr";
    final int numberOfSegmentsInAccessToken = 3;
    final int numberOfSegmentsInridgeToken = 5;
    String rawToken = null;
    String homeRegion = null;

    AccessTokenParser(String inToken) throws AccessTokenParseException {
        parse(inToken);
    }

    String getHomeRegion() {
        return homeRegion;
    }

    void parse(String inToken) throws AccessTokenParseException {
        rawToken = inToken;

        if (rawToken != null) {
            String[] parts = rawToken.trim().split("\\.");
            if (!(parts.length == numberOfSegmentsInAccessToken || parts.length == numberOfSegmentsInridgeToken)) {  // check for 5 in case of bridge token
                throw new AccessTokenParseException("Access token must have 3 or 5 segments");
            }

            // first base-64 decode each of the parts
            for (int i = 0; i < parts.length; i++) {
                byte[] decodedBytes = Base64.decode(parts[i].getBytes(), Base64.URL_SAFE); // uses UTF-8 (android's default charset)
                parts[i] = new String(decodedBytes);
            }

            // process each segment of the token
            extractHeader(parts[0]);
        }
    }

    void extractHeader(String inHeader) {
        logger.i("JWT token HEADER: " + inHeader);
        String header = inHeader.replace("{", "").replace("}", "");
        String[] tokens = header.split(",");
        HashMap<String, String> headersMap = new HashMap<>();
        for (int i = 0; i < tokens.length; i++) {
            String[] headers = tokens[i].split(":");
            headersMap.put(headers[0].replace("\"", ""), headers[1].replace("\"", ""));
        }
        homeRegion = headersMap.get(twilioVoiceHomeRegionSpecifier);
        if (this.homeRegion != null && this.homeRegion.equalsIgnoreCase("null")) {
            homeRegion = null;
        }
    }
}