package com.twilio.voice;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

/**
 * ConnectivityReceiver extends BroadcastReceiver to receive "android.net.conn.CONNECTIVITY_CHANGE"
 * notifications.
 */
class ConnectivityReceiver extends BroadcastReceiver {
    private static final Logger logger = Logger.getLogger(ConnectivityReceiver.class);
    private NetworkInfo currentNetworkInfo;

    @VisibleForTesting
    static void logNetworkChangeEvent(Voice.NetworkChangeEvent networkChangeEvent,
                                      @Nullable NetworkInfo currentNetworkInfo,
                                      @Nullable NetworkInfo newNetworkInfo,
                                      boolean reportedToSdk) {
        logger.d("networkChangeEvent: " + networkChangeEvent.name() +
                " reportedToSdk: " + Boolean.toString(reportedToSdk));
        logger.d("currentNetworkInfo: " + (currentNetworkInfo == null ?
                "null" : currentNetworkInfo.toString()));
        logger.d("newNetworkInfo: " + (newNetworkInfo == null ?
                "null" : newNetworkInfo.toString()));
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        logger.d("Network event received");
        if (intent.getAction()
                .equalsIgnoreCase(ConnectivityManager.CONNECTIVITY_ACTION)) {
            if (isInitialStickyBroadcast()) {
                logger.d("Ignoring network event, sticky broadcast");
                return;
            }
            ConnectivityManager conn =
                    (ConnectivityManager)
                            context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo newNetworkInfo = conn.getActiveNetworkInfo();
            Voice.NetworkChangeEvent networkChangeEvent =
                    Voice.NetworkChangeEvent.CONNECTION_CHANGED;

            if ((newNetworkInfo != null) &&
                    (currentNetworkInfo == null ||
                            currentNetworkInfo.getDetailedState() != newNetworkInfo.getDetailedState() ||
                            currentNetworkInfo.getType() != newNetworkInfo.getType() ||
                            currentNetworkInfo.getSubtype() != newNetworkInfo.getSubtype())) {
                if (!newNetworkInfo.isConnectedOrConnecting()) {
                    networkChangeEvent = Voice.NetworkChangeEvent.CONNECTION_LOST;
                }
                logNetworkChangeEvent(networkChangeEvent, currentNetworkInfo, newNetworkInfo, true);
                Voice.onNetworkChanged(networkChangeEvent);
            } else if (newNetworkInfo == null) {
                networkChangeEvent = Voice.NetworkChangeEvent.CONNECTION_LOST;
                logNetworkChangeEvent(networkChangeEvent, currentNetworkInfo, newNetworkInfo, true);
                Voice.onNetworkChanged(networkChangeEvent);
            } else {
                logNetworkChangeEvent(networkChangeEvent, currentNetworkInfo, newNetworkInfo, false);
            }
            currentNetworkInfo = newNetworkInfo;
        }
    }
}
