package com.twilio.voice;

import android.content.Context;
import android.os.AsyncTask;
import android.os.Handler;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import android.util.Log;

import com.twilio.voice.Constants.SeverityLevel;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;

class EventPublisher {

    private static final Logger logger = Logger.getLogger(EventPublisher.class);
    private static final String TAG = EventPublisher.class.getSimpleName();
    private Map<EventPublisherListener, Handler> listenerMap = new HashMap<>();
    /*
     * This param is nullable as {@link Voice#register(String, Voice.RegistrationChannel, String, RegistrationListener)}
     * and {@link Voice#unregister(String, Voice.RegistrationChannel, String, UnregistrationListener)}
     * does not pass Context as a param.
     * TODO This will be addressed as part of CLIENT-7068.
     */
    private Context context;
    private String accessToken;
    private String publisherName;
    private String homeRegion;
    String twilioProdSdkMetricsGatewayURL;
    String twilioProdSdkEventGatewayURL;
    int result = 0;

    private EventPublisherStatus eventPublisherStatus = new EventPublisherStatus();
    List<Integer> errorCodeList = Arrays.asList(HttpsURLConnection.HTTP_FORBIDDEN);
    @VisibleForTesting
    private EventPublisherEventListener publisherPublishEventListener;


    class EventPublisherStatus {
        private volatile boolean invalidatePublishing = false;
        private int errorCode = 0;
        private String responseMessage = "";
        private String explanation;

        public boolean isPublishingInvalidated() {
            return invalidatePublishing;
        }

        public void invalidatePublishing(boolean invalidatePublishing) {
            this.invalidatePublishing = invalidatePublishing;
        }

        public int getErrorCode() {
            return errorCode;
        }

        public String getResponseMessage() {
            return responseMessage;
        }

        public void setErrorDetails(int errorCode, String responseMessage, String explanation) {
            this.errorCode = errorCode;
            this.responseMessage = responseMessage;
            this.explanation = explanation;
        }

        public String getExplanation() {
            return explanation;
        }
    }

    /**
     * EventPublisher Constructor.
     *
     * @param publisherName - Name of the product publishing the event.
     * @param accessToken   - AccessToken used for server side authentication.
     */
    EventPublisher(final String publisherName, final String accessToken) {
        this(null, publisherName, accessToken);
    }

    /**
     * EventPublisher Constructor.
     *
     * @param context       - An Android context. This param is nullable as {@link Voice#register(String, Voice.RegistrationChannel, String, RegistrationListener)}
     *                      and {@link Voice#unregister(String, Voice.RegistrationChannel, String, UnregistrationListener)} does not pass Context as a param.
     * @param publisherName - Name of the product publishing the event.
     * @param accessToken   - AccessToken used for server side authentication.
     */
    EventPublisher(@Nullable final Context context, final String publisherName, final String accessToken) {
        if (accessToken == null) {
            throw new NullPointerException("accessToken must not be null.");
        }
        if (publisherName == null) {
            throw new NullPointerException("publisherName must not be null.");
        }
        this.context = context;
        this.accessToken = accessToken;
        this.publisherName = publisherName;
        this.twilioProdSdkMetricsGatewayURL = Constants.getKeyKibanaMetricsHostUrl();
        this.twilioProdSdkEventGatewayURL = Constants.getKeyKibanaEventGatewayHostUrl();
        try {
            this.homeRegion = new AccessTokenParser(accessToken).getHomeRegion();
        } catch (AccessTokenParseException e) {
            e.printStackTrace();
        }
        if (homeRegion != null) {
            updateServiceHostUrlsWithHomeRegion(homeRegion);
        }
    }

    private void updateServiceHostUrlsWithHomeRegion(String homeRegion) {
        this.twilioProdSdkMetricsGatewayURL = String.format("https://eventgw.%s.twilio.com/v4/EndpointMetrics", homeRegion);
        this.twilioProdSdkEventGatewayURL = String.format("https://eventgw.%s.twilio.com/v4/EndpointEvents", homeRegion);
    }

    /**
     * Method to add a listener to receive callback notification when publish fails.
     */
    void addListener(EventPublisherListener listener) {
        Handler handler = Utils.createHandler();
        listenerMap.put(listener, handler);
    }

    @VisibleForTesting
    void addEventPublisherEventListener(EventPublisherEventListener listener) {
        publisherPublishEventListener = listener;
    }

    /**
     * Method to remove as a EventPublisherListener.
     */
    void removeListener(EventPublisherListener listener) {
        listenerMap.remove(listener);
    }

    /**
     * Post a metric event to metric event gateway.
     *
     * @param event - A metric event to publish.
     * @throws Exception - Throws JSON related exception
     */
    void publishMetrics(final MetricEvent event) throws Exception {
        if (event != null) {
            JSONObject eventData = event.toJSONObject(context);
            publish(eventData.toString(), twilioProdSdkMetricsGatewayURL);
            if (publisherPublishEventListener != null) {
                publisherPublishEventListener.onMetricEventPublished(event);
            }
        }
    }

    /**
     * Create a metric event.
     *
     * @param group   - The name of group the metrics event belongs to.
     * @param name    - The designated metrics event name.
     * @param payload - An array of metrics payload to publish to the server.
     */
    MetricEvent createMetricEvent(final String group, final String name, final JSONArray payload) {
        return new MetricEvent.Builder()
                .productName(this.publisherName)
                .eventName(name)
                .groupName(group)
                .level(com.twilio.voice.Constants.SeverityLevel.INFO)
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOAD_TYPE)
                .payLoad(payload)
                .build();
    }

    /**
     * Publish an event to the event server.
     *
     * @param level - LEVEL of the event. This can be INFO, DEBUG, WARNING, ERROR.
     * @param group - The name of group the metrics event belongs to.
     * @param name  - The designated metrics event name.
     * @param event - The event to publish
     * @throws Exception - Throws JSON related exception
     */
    void publish(final SeverityLevel level, final String group, final String name, final Event event) throws Exception {
        if (publisherPublishEventListener != null && !eventPublisherStatus.isPublishingInvalidated()) {
            publisherPublishEventListener.onEventPublished(level, group, name);
        }
        JSONObject eventData = event.toJSONObject(context);
        publish(eventData.toString(), twilioProdSdkEventGatewayURL);
    }

    /**
     * Create an event.
     *
     * @param level   - LEVEL of the event. This can be INFO, DEBUG, WARNING, ERROR.
     * @param group   - The name of group the metrics event belongs to.
     * @param name    - The designated metrics event name.
     * @param payload - A payload to publish to the server.
     */
    Event createEvent(final SeverityLevel level, final String group, final String name, final JSONObject payload) {
        return new Event.Builder()
                .productName(this.publisherName)
                .eventName(name)
                .groupName(group)
                .level(level)
                .payLoadType(com.twilio.voice.Constants.APP_JSON_PAYLOAD_TYPE)
                .payLoad(payload)
                .build();
    }

    /**
     * Publish creates an AsyncTask for each network request. This
     * should work as AsyncTask has it's own thread pool.
     *
     * @param eventData - data
     * @param hostURL   - url where data is sent
     */
    private void publish(final String eventData, final String hostURL) {
        AsyncTask<Void, Void, Void> pushMetrics = new AsyncTask<Void, Void, Void>() {
            @Override
            protected Void doInBackground(Void... voids) {
                if (!eventPublisherStatus.isPublishingInvalidated()) {
                    logger.d("Start publishing events to : " + hostURL + "\n" + eventData);

                    HttpsURLConnection urlConnection = null;
                    try {
                        urlConnection =
                                VoiceURLConnection.create(
                                        accessToken,
                                        hostURL,
                                        VoiceURLConnection.METHOD_TYPE_POST);

                        OutputStreamWriter wr = new OutputStreamWriter(urlConnection.getOutputStream());
                        wr.write(eventData);
                        wr.close();
                        result = urlConnection.getResponseCode();
                        String responseMessage = urlConnection.getResponseMessage();

                        if (result == HttpsURLConnection.HTTP_OK) {
                            logger.d("Response: " + result + " - " + responseMessage);
                        } else {
                            if (errorCodeList.contains(result)) {
                                logger.e("Invalidating further publishing : " + result + " - " + responseMessage);
                                eventPublisherStatus.invalidatePublishing(true);
                            }
                            BufferedReader bufferedReader = new BufferedReader(
                                    new InputStreamReader((urlConnection.getErrorStream())));
                            StringBuilder stringBuilder = new StringBuilder();
                            String line;
                            while ((line = bufferedReader.readLine()) != null) {
                                stringBuilder.append(line);
                                stringBuilder.append('\n');
                            }
                            bufferedReader.close();
                            String jsonString = stringBuilder.toString();
                            String explanation = result + " - " + responseMessage + "-" + jsonString;
                            logger.d("Response: " + explanation);
                            eventPublisherStatus.setErrorDetails(result, responseMessage, explanation);
                            notifyListeners(result, responseMessage, explanation);
                        }
                    } catch (Exception e) {
                        Log.e(TAG, " " + e.toString());
                        logger.e(e.toString());
                    } finally {
                        VoiceURLConnection.release(urlConnection);
                    }
                } else {
                    notifyListeners(result, eventPublisherStatus.getResponseMessage(), eventPublisherStatus.getExplanation());
                }
                return null;
            }
        };
        pushMetrics.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
    }

    private void notifyListeners(final int errorCode, final String errorMessage, final String explanation) {
        for (Map.Entry<EventPublisherListener, Handler> entry : this.listenerMap.entrySet()) {
            final EventPublisherListener listener = entry.getKey();
            Handler handler = entry.getValue();
            if (handler != null) {
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (listener != null) {
                            listener.onError(new VoiceException(errorCode, errorMessage, explanation) {
                            });
                        }
                    }
                });
            }
        }
    }

    /**
     * Interface for a generic listener object.
     */
    interface EventPublisherListener {
        /**
         * Callback to report errorInfo status of an asynchronous call to the
         * back end.
         *
         * @param voiceException - Object containing errorInfo info
         */
        void onError(VoiceException voiceException);
    }

    /**
     * Interface for a generic listener object.
     */
    @VisibleForTesting
    interface EventPublisherEventListener {
        /**
         * Callback to report an event was published
         */
        void onEventPublished(final SeverityLevel level, final String group, final String name);

        /**
         * Callback to report a metric event was published
         */
        void onMetricEventPublished(MetricEvent event);
    }
}
