package com.twilio.voice;

import androidx.annotation.NonNull;

import java.util.HashSet;
import java.util.Set;

/**
 *  IceOptions specifies custom media connectivity configurations.
 *
 *  <p>Media connections are established using the ICE (Interactive Connectivity Establishment)
 *  protocol. These options allow you to customize how data flows to and from participants, and
 *  which protocols to use. You may also provide your own ICE servers, overriding the defaults.
 *  https://www.twilio.com/stun-turn.</p>
 */
public class IceOptions {
    private final Set<IceServer> iceServers;
    private final IceTransportPolicy iceTransportPolicy;

    private IceOptions(Builder builder) {
        this.iceServers = builder.iceServers != null ?
                builder.iceServers : new HashSet<IceServer>();
        this.iceTransportPolicy = builder.iceTransportPolicy;
    }

    @NonNull public Set<IceServer> getIceServers() {
        return iceServers;
    }

    @NonNull public IceTransportPolicy getIceTransportPolicy() {
        return iceTransportPolicy;
    }

    @SuppressWarnings("unused")
    IceServer[] getIceServersArray() {
        IceServer[] iceServersArray = new IceServer[0];
        if (iceServers.size() > 0) {
            iceServersArray = iceServers.toArray(new IceServer[iceServers.size()]);
        }
        return iceServersArray;
    }

    public static class Builder {
        private Set<IceServer> iceServers;
        private IceTransportPolicy iceTransportPolicy = IceTransportPolicy.ALL;

        public Builder() {}

        /**
         * Set of {@link IceServer} objects to be used during connection establishment.
         */
        @NonNull public Builder iceServers(@NonNull Set<IceServer> iceServers) {
            Preconditions.checkNotNull(iceServers, "iceServers must not be null");
            this.iceServers = iceServers;
            return this;
        }

        /**
         * The transport policy to use. Defaults to {@link IceTransportPolicy#ALL}.
         */
        @NonNull public Builder iceTransportPolicy(@NonNull IceTransportPolicy iceTransportPolicy) {
            Preconditions.checkNotNull(iceTransportPolicy, "iceTransportPolicy must not be null");
            this.iceTransportPolicy = iceTransportPolicy;
            return this;
        }

        @NonNull public IceOptions build() {
            return new IceOptions(this);
        }
    }
}
