package com.twilio.voice;

import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import android.util.Log;
import android.util.Pair;

import com.getkeepsafe.relinker.ReLinker;

import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import static com.twilio.voice.Utils.isAudioPermissionGranted;

public abstract class Voice {
    static AtomicInteger networkChangedCount = new AtomicInteger(0);
    static Map<LogModule, LogLevel> moduleLogLevel = new EnumMap(LogModule.class);
    private static final LogLevel defaultLogLevel = LogLevel.ERROR;
    static LogLevel level = defaultLogLevel;
    static boolean enableInsights = true;
    static String region = Constants.GLOBAL_LOW_LATENCY_REGION;
    static String edge = Constants.EDGE_ROAMING;
    static boolean isLibraryLoaded = false;
    static final Set<Call> calls = new HashSet<>();
    static Pair<String, String> callSidBridgeTokenPair;
    static final Map<String, CallInviteProxy> callInviteProxyMap = new HashMap<>();
    static final Set<Call> rejects = new HashSet<>();
    static AudioDevice audioDevice;

    public enum RegistrationChannel {
        FCM, GCM;

        @Override
        public String toString() {
            return name().toLowerCase();
        }
    }

    @SuppressWarnings("unused")
    Voice() {
        // Package scoped constructor ensures the Javadoc does not show a default constructor
    }

    /**
     * Register for incoming call messages. A successful registration will ensure that push
     * notifications will arrive via the GCM or FCM service for the lifetime of the registration
     * device token provided by the GCM or FCM service instance. When registration is successful,
     * {@code RegistrationListener.onRegistered(...)} callback is raised. Once successfully registered,
     * the registered binding has a time-to-live(TTL) of 1 year. If the registered binding is inactive
     * for 1 year, it is deleted and push notifications to the registered identity will not succeed.
     * However, whenever the registered binding is used to reach the registered identity, the TTL is
     * reset to 1 year.
     *
     * <p>
     * If registration fails, {@code RegistrationListener.onError(...)} callback is raised with
     * {@code RegistrationException}. {@code RegistrationException.getMessage(...)}
     * provides the root cause of the failure.
     * </p>
     *
     * <table border="1">
     * <caption>Registration Exceptions</caption>
     * <tr>
     * <td>Registration Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_ACCESS_TOKEN_REJECTED}</td><td>51007</td><td>Token authentication is rejected by authentication service</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REGISTRATION_ERROR}</td><td>31301</td><td>Registration failed. Look at RegistrationException.getMessage(...) for details</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request. The request could not be understood due to malformed syntax</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden. The server understood the request, but is refusing to fulfill it</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found. The server has not found anything matching the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout. A request timeout occurred</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_CONFLICT}</td><td>31409</td><td>Conflict. The request could not be processed because of a conflict in the current state of the resource. Another request may be in progress</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_UPGRADE_REQUIRED}</td><td>31426</td><td>Upgrade Required. The client should switch to a different protocol</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_TOO_MANY_REQUEST}</td><td>31429</td><td>Too Many Requests. Too many requests were sent in a given amount of time</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error. The server could not fulfill the request due to some unexpected condition</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway. The server is acting as a gateway or proxy, and received an invalid response from a downstream server while attempting to fulfill the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable. The server is currently unable to handle the request due to a temporary overloading or maintenance of the server</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout. The server, while acting as a gateway or proxy, did not receive a timely response from an upstream server</td>
     * </tr>
     *
     * </table>
     *
     * <p>
     * The identity provided in the {@code accessToken} may only contain alpha-numeric and
     * underscore characters. Other characters, including spaces, will result in undefined behavior.
     * </p>
     * Insights :
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>registration</td><td>Registration is successful</td><td>3.0.0-preview4</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>registration-error</td><td>Registration failed</td><td>3.0.0-preview4</td>
     * </tr>
     * </table>
     *
     * <p>
     * The maximum number of characters for the identity provided in the token is 121. The identity
     * may only contain alpha-numeric and underscore characters. Other characters, including spaces,
     * or exceeding the maximum number of characters, will result in not being able to place or receive
     * calls.
     * </p>
     *
     * @param accessToken          An access token.
     * @param registrationChannel  An enumeration of registration channels.
     * @param registrationToken    A GCM or FCM registration token.
     * @param listener             A listener that receives registration request status.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre>
     * final String registrationToken = FirebaseInstanceId.getInstance().getToken();
     * if (registrationToken != null) {
     *     Voice.register(accessToken, Voice.RegistrationChannel.FCM, registrationToken, registrationListener);
     * }
     * </pre>
     *
     * To specify a home region where your data is stored, use the `twr` specifier in the access token header as the example below:
     * <pre><code>
     * {
     *   "alg": "HS256",
     *   "typ": "JWT",
     *   "cty": "twilio-fpa;v=1",
     *   "twr": "au1"
     * }
     * </code></pre>
     */
    public static void register(@NonNull final String accessToken,
                                @NonNull final RegistrationChannel registrationChannel,
                                @NonNull final String registrationToken,
                                @NonNull final RegistrationListener listener) {
        Preconditions.checkNotNull(accessToken, "accessToken must not be null");
        Preconditions.checkNotNull(registrationChannel, "registrationChannel must not be null");
        Preconditions.checkNotNull(registrationToken, "registrationToken must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");
        Registrar registrar = new Registrar(accessToken, registrationChannel.toString(),
                registrationToken);
        registrar.register(listener);
    }

    /**
     * Unregister from receiving further incoming call messages. When unregistration is successful,
     * {@code UnregistrationListener.onUnregistered(...)} callback is raised.
     *
     * <p>
     * If unregistration fails, {@code UnregistrationListener.onError(...)} callback is raised with
     * {@code RegistrationException}. {@code RegistrationException.getMessage(...)}
     * provides the root cause of the failure.
     * </p>
     *
     * <table border="1">
     * <caption>Unregistration Exceptions</caption>
     * <tr>
     * <td>Registration Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_ACCESS_TOKEN_REJECTED}</td><td>51007</td><td>Token authentication is rejected by authentication service</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REGISTRATION_ERROR}</td><td>31301</td><td>Registration failed. Look at RegistrationException.getMessage(...) for details</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request. The request could not be understood due to malformed syntax</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden. The server understood the request, but is refusing to fulfill it</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found. The server has not found anything matching the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout. A request timeout occurred</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_CONFLICT}</td><td>31409</td><td>Conflict. The request could not be processed because of a conflict in the current state of the resource. Another request may be in progress</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_UPGRADE_REQUIRED}</td><td>31426</td><td>Upgrade Required. The client should switch to a different protocol</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_TOO_MANY_REQUEST}</td><td>31429</td><td>Too Many Requests. Too many requests were sent in a given amount of time</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error. The server could not fulfill the request due to some unexpected condition</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway. The server is acting as a gateway or proxy, and received an invalid response from a downstream server while attempting to fulfill the request</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable. The server is currently unable to handle the request due to a temporary overloading or maintenance of the server</td>
     * </tr>
     * <tr>
     * <td>{@link RegistrationException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout. The server, while acting as a gateway or proxy, did not receive a timely response from an upstream server</td>
     * </tr>
     * </table>
     *
     * <p>
     * The identity provided in the {@code accessToken} may only contain alpha-numeric and
     * underscore characters. Other characters, including spaces, will result in undefined behavior.
     * </p>
     * Insights :
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>unregistration</td><td>Unregistration is successful</td><td>3.0.0-preview4</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>unregistration-error</td><td>Unregistration failed</td><td>3.0.0-preview4</td>
     * </tr>
     * </table>
     *
     * <p>
     * The maximum number of characters for the identity provided in the token is 121. The identity
     * may only contain alpha-numeric and underscore characters. Other characters, including spaces,
     * or exceeding the maximum number of characters, will result in not being able to place or receive
     * calls.
     * </p>
     *
     * @param accessToken          An access token.
     * @param registrationChannel  An enumeration of registration channels.
     * @param registrationToken    An FCM or GCM device token.
     * @param listener             A listener that receives unregistration request status.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre>
     * Voice.unregister(accessToken, Voice.RegistrationChannel.FCM, registrationToken, unregistrationListener);
     * </pre>
     *
     * To specify a home region where your data is stored, use the `twr` specifier in the access token header as the example below:
     * <pre><code>
     * {
     *   "alg": "HS256",
     *   "typ": "JWT",
     *   "cty": "twilio-fpa;v=1",
     *   "twr": "au1"
     * }
     * </code></pre>
     */
    public static void unregister(@NonNull final String accessToken,
                                  @NonNull final RegistrationChannel registrationChannel,
                                  @NonNull final String registrationToken,
                                  @NonNull final UnregistrationListener listener) {
        Preconditions.checkNotNull(accessToken, "accessToken must not be null");
        Preconditions.checkNotNull(registrationChannel, "registrationChannel must not be null");
        Preconditions.checkNotNull(registrationToken, "registrationToken must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");
        Registrar registrar = new Registrar(accessToken, registrationChannel.toString(), registrationToken);
        registrar.unregister(listener);
    }

    /**
     * Creates and returns a new {@link Call}. A {@link SecurityException} will be thrown if RECORD_AUDIO
     * is not granted.
     *
     * <p>{@link Call.Listener} receives the state of the {@link Call}.</p>
     *
     * <table border="1">
     * <caption>Call.Listener events</caption>
     * <tr>
     * <td>Callback Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnectFailure(Call, CallException)}</td><td>The call failed to connect. {@link CallException} provides details of the root cause.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onRinging(Call)}</td><td>Emitted once before the {@link Call.Listener#onConnected(Call)} callback when the callee is being alerted of a {@link Call}.</td><td>3.0.0-preview2</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnected(Call)}</td><td>The call has connected.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onDisconnected(Call, CallException)}</td><td>The call was disconnected. If the call ends due to an error the `CallException` is non-null. If the call ends normally `CallException` is null.</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * <p>If {@code connect} fails, {@link Call.Listener#onConnectFailure(Call, CallException)} callback is raised with {@link CallException}. {@link CallException#getMessage()} and {@link CallException#getExplanation()}
     * provide details of the failure.
     *
     * If {@link Call#disconnect()} is called while attempting to connect, the {@link Call.Listener#onDisconnected(Call, CallException)} callback will be raised with no error.
     * </p>
     *
     * <p>If {@link Voice#connect(Context, ConnectOptions, Call.Listener)} fails due to an authentication error, the SDK receives one of the following errors.</p>
     *
     * <table border="1">
     * <caption>Authentication Exceptions</caption>
     * <tr>
     * <td>Authentication Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTH_FAILURE}</td><td>20151</td><td>Twilio failed to authenticate the client</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_APPLICATION_SID}</td><td>21218</td><td>Invalid ApplicationSid</td>
     * </tr>
     * </table>
     *
     * <p>If {@link Voice#connect(Context, ConnectOptions, Call.Listener)} fails due to any other reasons, the SDK receives one of the following errors.</p>
     *
     * <table border="1">
     * <caption>Call Exceptions</caption>
     * <tr>
     * <td>Call Exception</td><td>Error Code</td><td>Description</td>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CONNECTION_ERROR}</td><td>31005</td><td>Connection error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_CANCELLED}</td><td>31008</td><td>Call Cancelled</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TRANSPORT_ERROR}</td><td>31009</td><td>Transport error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MALFORMED_REQUEST}</td><td>31100</td><td>Malformed request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTHORIZATION_ERROR}</td><td>31201</td><td>Authorization error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TEMPORARILY_UNAVAILABLE_ERROR}</td><td>31480</td><td>Temporarily Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_OR_TRANSACTION_DOES_NOT_EXIST_ERROR}</td><td>31481</td><td>Call/Transaction Does Not Exist</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_ADDRESS_INCOMPLETE_ERROR}</td><td>31484</td><td>Address Incomplete</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_HERE_ERROR}</td><td>31486</td><td>Busy Here</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TERMINATED_ERROR}</td><td>31487</td><td>Request Terminated</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DNS_RESOLUTION}</td><td>31530</td><td>DNS Resolution Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_EVERYWHERE_ERROR}</td><td>31600</td><td>Busy Everywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DECLINE_ERROR}</td><td>31603</td><td>Decline</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DOES_NOT_EXIST_ANYWHERE_ERROR}</td><td>31604</td><td>Does Not Exist Anywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED}</td><td>53001</td><td>Signaling connection disconnected</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_LOCAL_MEDIA_DESCRIPTION}</td><td>53400</td><td>Client is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_LOCAL_MEDIA_DESCRIPTION}</td><td>53401</td><td>Server is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_REMOTE_MEDIA_DESCRIPTION}</td><td>53402</td><td>Client is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_REMOTE_MEDIA_DESCRIPTION}</td><td>53403</td><td>Server is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NO_SUPPORTED_CODEC}</td><td>53404</td><td>No supported code</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MEDIA_CONNECTION_FAILED}</td><td>53405</td><td>Media connection failed</td>
     * </tr>
     * </table>
     *
     * <p>Insights :</p>
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>outgoing</td><td>Outgoing call is made. Call state is in {@link Call.State#CONNECTING} state</td><td>3.0.0-beta2</td>
     * </tr>
     * <tr>
     * <td>settings</td><td>codec</td><td>Negotiated selected codec is received and remote SDP is set</td><td>5.0.1</td>
     * </tr>
     * </table>
     *
     * <p>If connect fails, an error event is published to Insights.</p>
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * @param context     An Android context.
     * @param connectOptions A set of options that allow you to configure your {@link Call}.
     * @param listener    A listener that receives call status.
     * @return A new {@link Call}
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * ConnectOptions connectOptions = new ConnectOptions.Builder(accessToken).build();
     * Call call = Voice.connect(context, connectOptions, new Call.Listener() {
     *     {@literal @}Override
     *      public void onRinging(@NonNull Call call) {
     *          Log.d(TAG, "Ringing");
     *      }
     *
     *     {@literal @}Override
     *      public void onConnected(@NonNull final Call call) {
     *          Log.d(TAG, "Received onConnected " + call.getSid());
     *      }
     *
     *     {@literal @}Override
     *      public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
     *          Log.d(TAG, "Received onConnectFailure with CallException: " + callException.getErrorCode()+ ":" + callException.getMessage());
     *      }
     *
     *     {@literal @}Override
     *      public void onDisconnected(@NonNull Call call, CallException callException) {
     *          if (callException != null) {
     *              Log.d(TAG, "Received onDisconnected with CallException: " + callException.getMessage() + ": " + call.getSid());
     *          } else {
     *              Log.d(TAG, "Received onDisconnected");
     *          }
     *      }
     *  });
     *  }
     * </code></pre>
     *
     * To specify a home region where your data is stored, use the `twr` specifier in the access token header as the example below:
     * <pre><code>
     * {
     *   "alg": "HS256",
     *   "typ": "JWT",
     *   "cty": "twilio-fpa;v=1",
     *   "twr": "au1"
     * }
     * </code></pre>
     *
     */
    @NonNull public static Call connect(@NonNull final Context context,
                                        @NonNull final ConnectOptions connectOptions,
                                        @NonNull final Call.Listener listener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(connectOptions, "connectOptions must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");

        if (!isAudioPermissionGranted(context)) {
            throw new SecurityException("Requires the RECORD_AUDIO permission");
        }

        /*
         * Build a new ConnectOptions by transferring the public interface portions of the provided
         * ConnectOptions until the remainder of the options become public.
         */
        ConnectOptions.Builder builder = new ConnectOptions.Builder(connectOptions.getAccessToken());
        builder.params(connectOptions.getParams());
        if (connectOptions.getIceOptions() != null) {
            builder.iceOptions(connectOptions.getIceOptions());
        }
        if (connectOptions.getPreferredAudioCodecs() != null) {
            builder.preferAudioCodecs(connectOptions.getPreferredAudioCodecs());
        }
        builder.enableDscp(connectOptions.enableDscp);
        builder.enableIceGatheringOnAnyAddressPorts(connectOptions.enableIceGatheringOnAnyAddressPorts);

        LocalAudioTrack localAudioTrack = LocalAudioTrack.create(context, true);
        builder.audioTracks(Collections.singletonList(localAudioTrack));
        builder.eventListener(connectOptions.getEventListener());
        ConnectOptions internalConnectOptions = builder.build();

        final Call call = new Call(context.getApplicationContext(), internalConnectOptions.getAccessToken(), listener);
        call.connect(internalConnectOptions);

        return call;
    }

    /**
     * Creates and returns a new {@link Call}. A {@link SecurityException} will be thrown if RECORD_AUDIO
     * is not granted.
     *
     * <p>{@link Call.Listener} receives the state of the {@link Call}.</p>
     *
     * <table border="1">
     * <caption>Call.Listener events</caption>
     * <tr>
     * <td>Callback Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnectFailure(Call, CallException)}</td><td>The call failed to connect. {@link CallException} provides details of the root cause.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onRinging(Call)}</td><td>Emitted once before the {@link Call.Listener#onConnected(Call)} callback when the callee is being alerted of a {@link Call}.</td><td>3.0.0-preview2</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onConnected(Call)}</td><td>The call has connected.</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>{@link Call.Listener#onDisconnected(Call, CallException)}</td><td>The call was disconnected. If the call ends due to an error the `CallException` is non-null. If the call ends normally `CallException` is null.</td><td>3.0.0-preview1</td>
     * </tr>
     * </table>
     *
     * <p>If {@code connect} fails, {@link Call.Listener#onConnectFailure(Call, CallException)} callback is raised with {@link CallException}. {@link CallException#getMessage()} and {@link CallException#getExplanation()}
     * provide details of the failure.
     *
     * If {@link Call#disconnect()} is called while attempting to connect, the {@link Call.Listener#onDisconnected(Call, CallException)} callback will be raised with no error.
     * </p>
     *
     * <p>If {@link Voice#connect(Context, String, Call.Listener)} fails due to an authentication error, the SDK receives one of the following errors.</p>
     *
     * <table border="1">
     * <caption>Authentication Exceptions</caption>
     * <tr>
     * <td>Authentication Exception</td><td>Error Code</td><td>Description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN}</td><td>20101</td><td>Twilio was unable to validate your Access Token</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_HEADER}</td><td>20102</td><td>Invalid Access Token header</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ISSUER_SUBJECT}</td><td>20103</td><td>Invalid Access Token issuer or subject</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_EXPIRY}</td><td>20104</td><td>Access Token has expired or expiration date is invalid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_NOT_VALID_YET}</td><td>20105</td><td>Access Token not yet valid</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_ACCESS_TOKEN_GRANT}</td><td>20106</td><td>Invalid Access Token grants</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTH_FAILURE}</td><td>20151</td><td>Twilio failed to authenticate the client</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TTL}</td><td>20157</td><td>Expiration Time in the Access Token Exceeds Maximum Time Allowed</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_TOKEN}</td><td>20403</td><td>Forbidden. The account lacks permission to access the Twilio API</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INVALID_APPLICATION_SID}</td><td>21218</td><td>Invalid ApplicationSid</td>
     * </tr>
     * </table>
     *
     * <p>If {@link Voice#connect(Context, String, Call.Listener)} fails due to any other reason, the SDK receives one of the following errors.</p>
     *
     * <table border="1">
     * <caption>Call Exceptions</caption>
     * <tr>
     * <td>Call Exception</td><td>Error Code</td><td>Description</td>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CONNECTION_ERROR}</td><td>31005</td><td>Connection error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_CANCELLED}</td><td>31008</td><td>Call Cancelled</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TRANSPORT_ERROR}</td><td>31009</td><td>Transport error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MALFORMED_REQUEST}</td><td>31100</td><td>Malformed request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_AUTHORIZATION_ERROR}</td><td>31201</td><td>Authorization error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_REQUEST}</td><td>31400</td><td>Bad Request</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_FORBIDDEN}</td><td>31403</td><td>Forbidden</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NOT_FOUND}</td><td>31404</td><td>Not Found</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TIMEOUT}</td><td>31408</td><td>Request Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_TEMPORARILY_UNAVAILABLE_ERROR}</td><td>31480</td><td>Temporarily Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CALL_OR_TRANSACTION_DOES_NOT_EXIST_ERROR}</td><td>31481</td><td>Call/Transaction Does Not Exist</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_ADDRESS_INCOMPLETE_ERROR}</td><td>31484</td><td>Address Incomplete</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_HERE_ERROR}</td><td>31486</td><td>Busy Here</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_REQUEST_TERMINATED_ERROR}</td><td>31487</td><td>Request Terminated</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_INTERNAL_SERVER_ERROR}</td><td>31500</td><td>Internal Server Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BAD_GATEWAY}</td><td>31502</td><td>Bad Gateway</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVICE_UNAVAILABLE}</td><td>31503</td><td>Service Unavailable</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_GATEWAY_TIMEOUT}</td><td>31504</td><td>Gateway Timeout</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DNS_RESOLUTION}</td><td>31530</td><td>DNS Resolution Error</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_BUSY_EVERYWHERE_ERROR}</td><td>31600</td><td>Busy Everywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DECLINE_ERROR}</td><td>31603</td><td>Decline</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_DOES_NOT_EXIST_ANYWHERE_ERROR}</td><td>31604</td><td>Does Not Exist Anywhere</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SIGNALING_CONNECTION_DISCONNECTED}</td><td>53001</td><td>Signaling connection disconnected</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_LOCAL_MEDIA_DESCRIPTION}</td><td>53400</td><td>Client is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_LOCAL_MEDIA_DESCRIPTION}</td><td>53401</td><td>Server is unable to create or apply a local media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_CLIENT_REMOTE_MEDIA_DESCRIPTION}</td><td>53402</td><td>Client is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_SERVER_REMOTE_MEDIA_DESCRIPTION}</td><td>53403</td><td>Server is unable to apply a remote media description</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_NO_SUPPORTED_CODEC}</td><td>53404</td><td>No supported code</td>
     * </tr>
     * <tr>
     * <td>{@link CallException#EXCEPTION_MEDIA_CONNECTION_FAILED}</td><td>53405</td><td>Media connection failed</td>
     * </tr>
     * </table>
     *
     * <p>
     * The identity provided in the {@code accessToken} may only contain alpha-numeric and
     * underscore characters. Other characters, including spaces, will result in undefined behavior.
     * </p>
     *
     * <p>Insights :</p>
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>outgoing</td><td>Outgoing call is made. Call state is in {@link Call.State#CONNECTING} state</td><td>3.0.0-beta2</td>
     * </tr>
     * <tr>
     * <td>settings</td><td>codec</td><td>Negotiated selected codec is received and remote SDP is set</td><td>5.0.1</td>
     * </tr>
     * </table>
     *
     * <p>If connect fails, an error event is published to Insights.</p>
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>error</td><td>Error description. Call state transitions to {@link Call.State#DISCONNECTED}</td><td>3.0.0-beta2</td>
     * </tr>
     * </table>
     *
     * <p>
     * The maximum number of characters for the identity provided in the token is 121. The identity
     * may only contain alpha-numeric and underscore characters. Other characters, including spaces,
     * or exceeding the maximum number of characters, will result in not being able to place or receive
     * calls.
     * </p>
     *
     * @param context     An Android context.
     * @param accessToken The accessToken that provides the identity and grants of the caller.
     * @param listener    A listener that receives call status.
     * @return A new {@link Call}
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * Call call = Voice.connect(context, accessToken, new Call.Listener() {
     *     {@literal @}Override
     *      public void onRinging(@NonNull Call call) {
     *          Log.d(TAG, "Ringing");
     *      }
     *
     *     {@literal @}Override
     *      public void onConnected(@NonNull final Call call) {
     *          Log.d(TAG, "Received onConnected " + call.getSid());
     *      }
     *
     *     {@literal @}Override
     *      public void onConnectFailure(@NonNull Call call, @NonNull CallException callException) {
     *          Log.d(TAG, "Received onConnectFailure with CallException: " + callException.getErrorCode()+ ":" + callException.getMessage());
     *      }
     *
     *     {@literal @}Override
     *      public void onDisconnected(@NonNull Call call, CallException callException) {
     *          if (callException != null) {
     *              Log.d(TAG, "Received onDisconnected with CallException: " + callException.getMessage() + ": " + call.getSid());
     *          } else {
     *              Log.d(TAG, "Received onDisconnected");
     *          }
     *      }
     *  });
     *  }
     * </code></pre>
     *
     * To specify a home region where your data is stored, use the `twr` specifier in the access token header as the example below:
     * <pre><code>
     * {
     *   "alg": "HS256",
     *   "typ": "JWT",
     *   "cty": "twilio-fpa;v=1",
     *   "twr": "au1"
     * }
     * </code></pre>
     */
    @NonNull public static Call connect(@NonNull final Context context,
                                        @NonNull final String accessToken,
                                        @NonNull final Call.Listener listener) {
        return connect(context, new ConnectOptions.Builder(accessToken).build(), listener);
    }

    /**
     * Handle messages from FCM.
     *
     * <p>
     *     Twilio sends {@code call} notification messages via GCM/FCM.
     *     The message type is encoded in the dictionary with the key {@code twi_message_type} with the value {@code twilio.voice.call}.
     * </p>
     * <p>
     *     A {@code call} message is sent when someone wants to reach the registered {@code identity}.
     *     Passing a {@code call} message into {@link Voice#handleMessage(Context, Bundle, MessageListener)} will result in a {@link CallInvite} raised in the {@link MessageListener} and return {@code true}.
     * </p>
     * <p>
     *     A {@link CancelledCallInvite} will be raised to the provided {@link MessageListener} for
     *     the following reasons:
     * </p>
     * <ul>
     * <li> The call is prematurely disconnected by the caller</li>
     * <li> The callee does not accept or reject the call within 30 seconds</li>
     * <li> The SDK is unable to establish a connection to Twilio</li>
     * </ul>
     * <p>
     *     Passing malformed data to {@link Voice#handleMessage(Context, Map, MessageListener)} will return {@code false} and no {@link MessageListener} callback will be raised.
     * </p>
     *
     * <p>Insights :</p>
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>incoming</td><td>Incoming call notification received</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>listen</td><td>Raised when an attempt to listen for cancellations is made</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>listening</td><td>Raised when an attempt to listen for cancellations has succeeded</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>cancel</td><td>Raised when a cancellation has been reported</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>listening-error</td><td>Raised when an attempt to listen for cancellation has failed</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>unsupported-cancel-message-error</td><td>Raised when a "cancel" push notification is processed by the SDK. This version of the SDK does not support "cancel" push notifications</td><td>5.0.0</td>
     * </tr>
     * </table>
     *
     * @param context   An Android context.
     * @param data      Push notification payload.
     * @param listener  A {@link MessageListener} to receive incoming push notification callbacks.
     * @return          A boolean value that indicates whether the payload is a valid notification sent by Twilio.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * boolean valid = Voice.handleMessage(dataMap, new MessageListener() {
     *     {@literal @}Override
     *      public void onCallInvite(@NonNull CallInvite callInvite) {
     *          Log.d(TAG, "Received CallInvite");
     *      }
     *
     *     {@literal @}Override
     *      public void onCancelledCallInvite(@NonNull CancelledCallInvite cancelledCallInvite) {
     *          Log.d(TAG, "Received CancelledCallInvite");
     *      }
     * });
     * </code></pre>
     */
    public static synchronized boolean handleMessage(@NonNull final Context context,
                                                     @NonNull final Map<String, String> data,
                                                     @NonNull final MessageListener listener) {
        return handleMessage(context, data, listener, null);
    }

    /**
     * Handle messages from GCM.
     *
     * <p>
     *     Twilio sends {@code call} notification messages via GCM/FCM.
     *     The message type is encoded in the dictionary with the key {@code twi_message_type} with the value {@code twilio.voice.call}.
     * </p>
     * <p>
     *     A {@code call} message is sent when someone wants to reach the registered {@code identity}.
     *     Passing a {@code call} message into {@link Voice#handleMessage(Context, Bundle, MessageListener)} will result in a {@link CallInvite} raised in the {@link MessageListener} and return {@code true}.
     * </p>
     * <p>
     *     A {@link CancelledCallInvite} will be raised to the provided {@link MessageListener} for
     *     the following reasons:
     * </p>
     * <ul>
     * <li> The call is prematurely disconnected by the caller</li>
     * <li> The callee does not accept or reject the call within 30 seconds</li>
     * <li> The SDK is unable to establish a connection to Twilio</li>
     * </ul>
     * <p>
     *     Passing malformed data to {@link Voice#handleMessage(Context, Map, MessageListener)} will return {@code false} and no {@link MessageListener} callback will be raised.
     * </p>
     *
     * <p>Insights :</p>
     *
     * <table border="1">
     * <caption>Insights events</caption>
     * <tr>
     * <td>Group Name</td><td>Event Name</td><td>Description</td><td>Since version</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>incoming</td><td>Incoming call notification received</td><td>3.0.0-preview1</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>listen</td><td>Raised when an attempt to listen for cancellations is made</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>listening</td><td>Raised when an attempt to listen for cancellations has succeeded</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>cancel</td><td>Raised when a cancellation has been reported</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>connection</td><td>listening-error</td><td>Raised when an attempt to listen for cancellation has failed</td><td>5.0.0</td>
     * </tr>
     * <tr>
     * <td>registration</td><td>unsupported-cancel-message-error</td><td>Raised when a "cancel" push notification is processed by the SDK. This version of the SDK does not support "cancel" push notifications</td><td>5.0.0</td>
     * </tr>
     * </table>
     *
     * @param context   An Android context.
     * @param data      Push notification payload.
     * @param listener  A {@link MessageListener} to receive incoming push notification callbacks.
     * @return          A boolean value that indicates whether the payload is a valid notification sent by Twilio.
     *
     * <dl>
     * <dt><span class="strong">Usage example:</span></dt>
     * </dl>
     * <pre><code>
     * boolean valid = Voice.handleMessage(dataMap, new MessageListener() {
     *     {@literal @}Override
     *      public void onCallInvite(@NonNull CallInvite callInvite) {
     *          Log.d(TAG, "Received CallInvite");
     *      }
     *
     *     {@literal @}Override
     *      public void onCancelledCallInvite(@NonNull CancelledCallInvite cancelledCallInvite) {
     *          Log.d(TAG, "Received CancelledCallInvite");
     *      }
     * });
     * </code></pre>
     */
    public static boolean handleMessage(@NonNull final Context context,
                                        @NonNull final Bundle data,
                                        @NonNull final MessageListener listener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(data, "data must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");

        return handleMessage(context, Utils.bundleToMap(data), listener);
    }

    /**
     * Returns the version of the Voice SDK.
     *
     * @return The version of the SDK.
     */
    @NonNull public static String getVersion() {
        return BuildConfig.VERSION_NAME;
    }

    /**
     * Returns the logging level for messages logged by the Voice SDK. The default level is
     * {@link LogLevel#ERROR}.
     *
     * @return the logging level
     */
    @NonNull public static LogLevel getLogLevel() {
        return Voice.level;
    }

    /**
     * Returns the logging level for messages logged by the specified LogModule. The default log
     * level for each module is {@link LogLevel#ERROR}.
     *
     * @return the logging level
     */
    @NonNull public static LogLevel getModuleLogLevel(LogModule module) {
        if (Voice.moduleLogLevel.containsKey(module)) {
            return Voice.moduleLogLevel.get(module);
        }
        return defaultLogLevel;
    }

    /**
     * Returns if reporting statistics to Insights is enabled. Sending stats data to Insights is
     * enabled by default. The setting is specified in {@link Voice#enableInsights} and can be
     * updated via {@link Voice#enableInsights(boolean)}
     */
    public static boolean isInsightsEnabled() {
        return Voice.enableInsights;
    }

    /**
     * Returns the region specified {@link Voice#region}. The default region uses Global Low Latency
     * routing, which establishes a connection with the closest region to the user. This value can
     * be updated via {@link Voice#setRegion(String)}
     *
     * @deprecated  As of release 5.3.0, use {@link Voice#getEdge()} instead.
     */
    @Deprecated
    @NonNull public static String getRegion() {
        return Voice.region;
    }

    /**
     * Returns the edge specified by {@link Voice#edge}. The default edge uses Global Low Latency
     * (roaming edge location)routing, which establishes a connection with the closest data center to
     * the user. This value can be updated via {@link Voice#setRegion(String)}
     */
    @NonNull public static String getEdge() {
        return Voice.edge;
    }

    /**
     * Returns the AudioDevice.
     */
    @NonNull
    public static AudioDevice getAudioDevice() {
        if (Voice.audioDevice == null) {
            Voice.audioDevice = new DefaultAudioDevice();
        }
        return Voice.audioDevice;
    }


    /**
     * Sets the logging level for messages logged by the Voice SDK.
     *
     * @param level The logging level
     */
    public static void setLogLevel(@NonNull LogLevel level) {
        setSDKLogLevel(level);
        if (isLibraryLoaded) {
            nativeSetModuleLevel(LogModule.CORE.ordinal(), level.ordinal());
        }
        // Save the log level
        Voice.level = level;
    }

    /**
     * Sets the logging level for messages logged by a specific module.
     *
     * @param module The module for this log level.
     * @param level  The logging level.
     */
    public static void setModuleLogLevel(@NonNull LogModule module, @NonNull LogLevel level) {
        if (module == LogModule.PLATFORM) {
            setSDKLogLevel(level);
        }
        if (isLibraryLoaded) {
            nativeSetModuleLevel(module.ordinal(), level.ordinal());
        }
        // Save the module log level
        Voice.moduleLogLevel.put(module, level);
    }

    /**
     * Specify reporting statistics to Insights. Sending stats data to Insights is enabled
     * by default.
     * <p>
     * NOTE: Setting the flag during a call will not be applied to ongoing calls. The flag will be
     * applied to subsequent {@link Voice#connect(Context, String, Call.Listener)} or
     * {@link Voice#handleMessage(Context, Bundle, MessageListener)} API calls.
     * </p>
     */
    public static void enableInsights(boolean enable) {
        if (isLibraryLoaded) {
            nativeEnableInsights(enable);
        }
        Voice.enableInsights = enable;
    }

    /**
     * Sets the region (Twilio data center) for the SDK.
     *
     * The default region uses Global Low Latency routing, which establishes a connection with
     * the closest region to the user.
     * <p>
     * NOTE: Setting the region during a call will not be applied to ongoing calls. The region will
     * be applied to subsequent {@link Voice#connect(Context, String, Call.Listener)} or
     * {@link Voice#handleMessage(Context, Bundle, MessageListener)} API calls.
     * </p>
     *
     * @deprecated  As of release 5.3.0, use {@link Voice#setEdge(String)} instead.
     *
     * @param region The region.
     */
    @Deprecated
    public static void setRegion(@NonNull String region) {
        Preconditions.checkNotNull(region, "region must not be null");
        // TODO: remove this in phase2
        Preconditions.checkArgument(
                Voice.edge.equals(Constants.EDGE_ROAMING),
                String.format("Non default edge value %s has already been specified. Please use " +
                        "Voice.edge or Voice.region to specify the Twilio Region that the " +
                        "SDK connects to.", Voice.getEdge()));
        if (isLibraryLoaded) {
            nativeSetRegion(region);
        }
        Voice.region = region;
    }

    /**
     * Sets the edge (Twilio data center) for the SDK.
     *
     * The edge value is a Twilio Edge name that corresponds to a geographic location of Twilio infrastructure
     * that the client will connect to. The default value `roaming` automatically selects an edge based on the
     * latency between client and available edges. `roaming` requires the upstream DNS to support [RFC7871](https://tools.ietf.org/html/rfc7871).
     * See [Global Low Latency requirements](https://www.twilio.com/docs/voice/client/javascript/voice-client-js-and-mobile-sdks-network-connectivity-requirements#global-low-latency-requirements) for more information.
     *
     * <p>
     * NOTE: Setting the edge during a call will not be applied to ongoing calls. The edge will
     * be applied to subsequent {@link Voice#connect(Context, String, Call.Listener)} or
     * {@link Voice#handleMessage(Context, Bundle, MessageListener)} API calls.
     *
     * The SDK will throw the {@link IllegalArgumentException} if both {@link Voice#edge} and `{@link Voice#region}`
     * values are specified.
     * </p>
     *
     * @param edge The edge.
     */
    public static void setEdge(@NonNull String edge) {
        Preconditions.checkNotNull(edge, "edge must not be null");
        // TODO: remove this in phase2
        Preconditions.checkArgument(
                Voice.region.equals(Constants.GLOBAL_LOW_LATENCY_REGION),
                String.format("Non default region value %s has already been specified. Please use " +
                        "Voice.edge or Voice.region to specify the Twilio Region that the " +
                        "SDK connects to.", Voice.getRegion()));
        if (isLibraryLoaded) {
            nativeSetEdge(edge);
        }
        Voice.edge = edge;
    }


    /**
     * Sets the custom audio device. The {@link Voice#audioDevice} can be updated when there is no
     * call in progress and will be applied to subsequent {@link Voice#connect(Context, String, Call.Listener)}
     * or {@link Voice#handleMessage(Context, Bundle,MessageListener)} API calls. Setting the
     * {@link Voice#audioDevice} during a call will result in {@link UnsupportedOperationException}.
     *
     * @param audioDevice The audio device.
     */
    public static void setAudioDevice(@NonNull AudioDevice audioDevice) throws UnsupportedOperationException {

        Preconditions.checkNotNull(audioDevice, "audioDevice must not be null");
        if (calls.isEmpty()) {
            Voice.audioDevice = audioDevice;
        } else {
            throw new UnsupportedOperationException("Changing the audio device during a call is not allowed");
        }
    }

    private static void setSDKLogLevel(LogLevel level) {
        /*
         * The Log Levels are defined differently in the Logger
         * which is based off android.util.Log.
         */
        switch (level) {
            case OFF:
                Logger.setLogLevel(Log.ASSERT);
                break;
            case ERROR:
                Logger.setLogLevel(Log.ERROR);
                break;
            case WARNING:
                Logger.setLogLevel(Log.WARN);
                break;
            case INFO:
                Logger.setLogLevel(Log.INFO);
                break;
            case DEBUG:
                Logger.setLogLevel(Log.DEBUG);
                break;
            case TRACE:
                Logger.setLogLevel(Log.VERBOSE);
                break;
            case ALL:
                Logger.setLogLevel(Log.VERBOSE);
                break;
            default:
                // Set the log level to assert/disabled if the value passed in is unknown
                Logger.setLogLevel(Log.ASSERT);
                break;
        }
    }

    static void onNetworkChanged(NetworkChangeEvent networkChangeEvent) {
        networkChangedCount.incrementAndGet();
        for (Call call : calls) {
            call.networkChange(networkChangeEvent);
        }
        for (CallInviteProxy callInviteProxy : callInviteProxyMap.values()) {
            callInviteProxy.networkChange(networkChangeEvent);
        }
    }

    static void loadLibrary(Context context) {
        if (!Voice.isLibraryLoaded) {
            ReLinker.loadLibrary(context, BuildConfig.TWILIO_VOICE_ANDROID_LIBRARY);
            Voice.isLibraryLoaded = true;
            /*
             * The user may have set the log level prior to the native library being loaded.
             * Attempt to set the core log level now that the native library has loaded.
             */
            Voice.setLogLevel(Voice.level);

            if (!edge.equals(Constants.EDGE_ROAMING)) {
                /*
                 * The user may have set the edge prior to the native library being loaded.
                 * Attempt to set the edge now that the native library has loaded.
                 */
                nativeSetEdge(edge);
            }

            if (!region.equals(Constants.GLOBAL_LOW_LATENCY_REGION)) {
                /*
                 * The user may have set the region prior to the native library being loaded.
                 * Attempt to set the edge now that the native library has loaded.
                 */
                // TODO remove this in phase2
                nativeSetRegion(region);
            }

            /*
             * The user may have set the enableInsights flag prior to the native library being loaded.
             * Attempt to set the enableInsights flag now that the native library has loaded.
             */
            Voice.enableInsights(Voice.enableInsights);

            /*
             * It is possible that the user has tried to set the log level for a specific module
             * before the library has loaded. Here we apply the log level for the module because we
             * know the native library is available
             */
            for (LogModule module : Voice.moduleLogLevel.keySet()) {
                Voice.setModuleLogLevel(module, Voice.moduleLogLevel.get(module));
            }
        }
    }

    enum NetworkChangeEvent {
        CONNECTION_LOST,
        CONNECTION_CHANGED
    }

    @VisibleForTesting
    static synchronized boolean handleMessage(@NonNull final Context context,
                                              @NonNull final Map<String, String> data,
                                              @NonNull final MessageListener listener,
                                              @Nullable Call.EventListener eventListener) {
        Preconditions.checkNotNull(context, "context must not be null");
        Preconditions.checkNotNull(data, "data must not be null");
        Preconditions.checkNotNull(listener, "listener must not be null");

        boolean callInviteValid = CallInvite.isValid(context, data);
        /*
         * This version of the SDK does not support "cancel" push notifications. SDK reports
         * "unsupported-cancel-message-error" event to Insights when a "cancel" push
         * notification is passed to handleMessage(). Calling nativeHandleMessage()
         * for both "call" and "cancel" push notification so that Voice C++ SDK can report
         * expected events.
         */
        if (callInviteValid || (CancelledCallInvite.isValid(data) && Voice.isInsightsEnabled()) ) {
            callNativeHandleMessage(context, data, listener, eventListener);
        }
        return callInviteValid;
    }

    private static void callNativeHandleMessage(@NonNull final Context context,
                                                @NonNull final Map<String, String> data,
                                                @NonNull final MessageListener listener,
                                                Call.EventListener eventListener) {
        loadLibrary(context);

        Handler handler = Utils.createHandler();

        handler.post(new Runnable() {
            @Override
            public void run() {
                /*
                 * The CallInvite and CancelledCallInvite are created as optimizations so
                 * they do not have to be created at the JNI level. These objects will be
                 * surfaced in the provided MessageListener callbacks only if the C++
                 * SDK informs the Android SDK that a CallInvite or CancelledCallInvite
                 * occurred.
                 */
                CallInvite callInvite = CallInvite.create(data);
                CancelledCallInvite cancelledCallInvite = CancelledCallInvite.create(data);
                if (callInvite.getBridgeToken() != null) {
                    callSidBridgeTokenPair = Pair.create(callInvite.getCallSid(), callInvite.getBridgeToken());
                }
                Pair<String[], String[]> dataKeyValues =
                        Utils.mapToArrays(data);
                // Use the runnable as a temporary MediaFactory owner
                MediaFactory mediaFactory = MediaFactory.instance(this,
                        context.getApplicationContext());
                nativeHandleMessage(context.getApplicationContext(),
                        dataKeyValues.first,
                        dataKeyValues.second,
                        callInvite,
                        cancelledCallInvite,
                        handler,
                        listener,
                        eventListener,
                        mediaFactory.getNativeMediaFactoryHandle());
                // Release temporary MediaFactory ownership
                mediaFactory.release(this);
            }
        });
    }

    private native static void nativeEnableInsights(boolean enable);
    private native static void nativeSetRegion(String region);
    private native static void nativeSetEdge(String edge);
    private native static void nativeSetModuleLevel(int module, int level);
    private native static boolean nativeHandleMessage(Context context,
                                                      String[] dataKeys,
                                                      String[] dataValues,
                                                      CallInvite callInvite,
                                                      CancelledCallInvite cancelledCallInvite,
                                                      Handler handler,
                                                      MessageListener messageListener,
                                                      Call.EventListener eventListener,
                                                      long nativeMediaFactoryHandle);
}