// =================================================================================================
// Copyright 2011 Twitter, Inc.
// -------------------------------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this work except in compliance with the License.
// You may obtain a copy of the License in the LICENSE file, or at:
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// =================================================================================================

package com.twitter.common.net;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.twitter.common.collections.Pair;

import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Utility methods to help with url tokenization.
 *
 * @author Adam Samet
 */
public class UrlTokenizerUtil {

  /**
   * Given an url host, tokenizes the host parts by dot, reverses them, and pads (with "") or
   * truncates up to numParts pieces.  Truncation happens post token reversal.  This method properly
   * accounts for two-part top-level domains as listed at
   * http://www.surbl.org/two-level-tlds as of 05/12/2010.  For example, www.twitter.co.uk will
   * be returned as  ["co.uk", "twitter", "www"].
   *
   * @param host A url host
   * @param numParts The number of host parts to return.
   * @return The reversed and tokenized list of host parts from host.
   */
  public static List<String> getReversedDomainParts(String host, int numParts) {
    List<String> domainParts = Lists.newArrayList();
    for (String item : host.split("\\.")) {
      domainParts.add(item);
    }
    Collections.reverse(domainParts);

    /*
     * If the result is at least length two, see if we can squash together the last two
     * parts as a top-level domain.
     */
    if (domainParts.size() >= 2 && TWO_PART_TLDS.contains(
        Pair.of(domainParts.get(1), domainParts.get(0)))) {
      domainParts.set(0, domainParts.get(1) + "." + domainParts.get(0));
      domainParts.remove(1);
    }

    if (domainParts.size() > numParts) {
      return domainParts.subList(0, numParts);
    } else if (domainParts.size() < numParts) {
      int diff = numParts - domainParts.size();
      for (int i = 0; i < diff; ++i) {
        domainParts.add("");
      }
    }

    return domainParts;
  }

  public static boolean isTLD(final String url, boolean reversed) {
    if (url == null) return false;
    String[] domainParts = url.split("\\.");
    if (domainParts.length == 1) {
      return true;
    } else if (domainParts.length == 2) {
      return TWO_PART_TLDS.contains(
        Pair.of(domainParts[reversed ? 1 : 0], domainParts[reversed ? 0 : 1]));
    }
    return false;
  }

  /*
   * These come from http://www.surbl.org/two-level-tlds as of 05/12/2010.  Be very careful about
   * changing this list, because it can screw up our ability to query historical data depending on
   * this list being static.
   */
  // Initializing this in a static block is not pretty, but doing 1.5k+ chained .add() calls on an
  // immutable set builder causes StackOverFlowError during compilation.
  private static final ImmutableSet.Builder<Pair<String, String>> TWO_PART_TLD_BUILDER =
      ImmutableSet.builder();
  private static Set<Pair<String, String>> TWO_PART_TLDS = null;
  static {
    TWO_PART_TLD_BUILDER.add(Pair.of("2000", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ab", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ab", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("abo", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "at"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "be"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "gn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "im"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ma"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "pg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "rs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "tz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "ug"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "yu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "zm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ac", "zw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("act", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ad", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("adm", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("adult", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("adv", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("adygeya", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aero", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aero", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aeroport", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("agr", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("agrar", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("agro", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ah", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aichi", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aid", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ak", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("akita", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("al", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aland", "fi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("alderney", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("alt", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("alt", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("altai", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("am", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("amur", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("amursk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("aomori", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ar", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("arkhangelsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("army", "mil"));
    TWO_PART_TLD_BUILDER.add(Pair.of("arq", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("art", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("art", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("art", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("art", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("art", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("arts", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("arts", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("arts", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asn", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asn", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ass", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("assedic", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("assn", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asso", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asso", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asso", "gp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asso", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asso", "mc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("asso", "re"));
    TWO_PART_TLD_BUILDER.add(Pair.of("astrakhan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("at", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("at", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("atm", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ato", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("au", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("au", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("auto", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("av", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("avocat", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("avoues", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("az", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("baikal", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("barreau", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bashkiria", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bbs", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bc", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bd", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("be", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bel", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("belgie", "be"));
    TWO_PART_TLD_BUILDER.add(Pair.of("belgorod", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bg", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bialystok", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bib", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bio", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bir", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("biz", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bj", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bl", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bmd", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bolt", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bourse", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("br", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("brand", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("british-library", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("bryansk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("buryatia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("busan", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("c", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ca", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ca", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ca", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("casino", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cbg", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cc", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cci", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ch", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ch", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chambagri", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chel", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chelyabinsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cherkassy", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chernigov", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chernovtsy", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chiba", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chirurgiens-dentistes", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chita", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chukotka", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chungbuk", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chungnam", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("chuvashia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cim", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("city", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("city", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ck", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("club", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cmw", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cn", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cn", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cng", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cnt", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ag"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ao"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "at"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "bw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ck"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "dk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "im"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ls"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ma"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "mu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "mz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "rs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "tv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "tz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ug"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "uz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "vi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "yu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "zm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("co", "zw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ac"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "af"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ag"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ai"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "an"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "aw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "bz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "cd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ch"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "dm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "es"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "gu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "hk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "hn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "hr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "io"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "jm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "jo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "kg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ki"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "kw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ky"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "kz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "la"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "lb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "lc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "li"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "lr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "mx"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "nc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "nf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ni"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "np"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "py"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "qa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "re"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "sy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "uy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "uz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "vc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "vi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ws"));
    TWO_PART_TLD_BUILDER.add(Pair.of("com", "ye"));
    TWO_PART_TLD_BUILDER.add(Pair.of("conf", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("conf", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("consulado", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("coop", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("coop", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("coop", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("coop", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("coop", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cpa", "pro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cq", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cri", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("crimea", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("csiro", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ct", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cul", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cv", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("cz", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("d", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("daegu", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("daejeon", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dagestan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dc", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("de", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("de", "net"));
    TWO_PART_TLD_BUILDER.add(Pair.of("de", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("de", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("de", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("de", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dk", "org"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dk", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dn", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dnepropetrovsk", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dni", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dns", "be"));
    TWO_PART_TLD_BUILDER.add(Pair.of("donetsk", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dp", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dpn", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dr", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("dudinka", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("e-burg", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("e", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("e164", "arpa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ebiz", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ecn", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ed", "ao"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ed", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ed", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ac"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "af"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ai"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "an"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "bt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ck"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "dm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "es"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "gh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "gi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "gp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "gr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "gu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "hk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "hn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "it"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "jm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "jo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "kg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "kw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ky"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "kz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "lb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "lc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "lr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "mx"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ni"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "np"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "py"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "qa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "rs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "sv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "uy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "vi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ws"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "ye"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "yu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edu", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("edunet", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ehime", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ekloges", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("embaixada", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eng", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ens", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ernet", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("erotica", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("erotika", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("es", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("es", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("esp", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("etc", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eti", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eu", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eu", "org"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eu", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eu", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("eun", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("experts-comptables", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("f", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fam", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("far", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fareast", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fax", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fed", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fgov", "be"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fh", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fhs", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fhsk", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fhv", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fi", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fie", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("film", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fin", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fin", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("firm", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("firm", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("firm", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("firm", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("firm", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fj", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fl", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fm", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fnd", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("folkebibl", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("forum", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fot", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fr", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fr", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("from", "hr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fst", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fukui", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fukuoka", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fukushima", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("fylkesbibl", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("g", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("g12", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ga", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("game", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("games", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gangwon", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gb", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gb", "net"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gc", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gd", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gda", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gdansk", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("geek", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gen", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gen", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gen", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("geometre-expert", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ggf", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gifu", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gmina", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "tz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("go", "ug"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "es"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "hn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "mx"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "ni"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gob", "sv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gok", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gon", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gop", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gos", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gouv", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gouv", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gouv", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ac"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "af"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ai"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "bt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "by"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ch"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ck"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "cx"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "dm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "gh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "gi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "gn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "gr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "gu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "hk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ie"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "im"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "io"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "it"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "jm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "jo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "kg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "kw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ky"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "kz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "li"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ma"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "np"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "py"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "qa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "rs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "sy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "to"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "vi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ws"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "ye"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "zm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gov", "zw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("govt", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gr", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("greta", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("grozny", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("grp", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gs", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gsm", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gub", "uy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("guernsey", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gunma", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gv", "ao"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gv", "at"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gwangju", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gx", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gyeongbuk", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gyeonggi", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gyeongnam", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("gz", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("h", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ha", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hb", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("he", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("health", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("herad", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hi", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hi", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hiroshima", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hk", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hl", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hn", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hokkaido", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hotel", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hotel", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hs", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hu", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("huissier-justice", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("hyogo", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("i", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ia", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ibaraki", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("icnet", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("id", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("id", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("id", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("id", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("id", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("id", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("idf", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("idn", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("idrett", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("idv", "hk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("idv", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("if", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("il", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("imb", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("in-addr", "arpa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("in", "rs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("in", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("in", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("in", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("incheon", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ind", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("inf", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("inf", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("info", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ing", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ingatlan", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("inima", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("int", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("intl", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ip6", "arpa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("iris", "arpa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("irkutsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("isa", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ishikawa", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("isla", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("it", "ao"));
    TWO_PART_TLD_BUILDER.add(Pair.of("it", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ivano-frankivsk", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ivanovo", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("iwate", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("iwi", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("iz", "hr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("izhevsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jamal", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jar", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jeju", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jeonbuk", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jeonnam", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jersey", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jet", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jl", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jobs", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jogasz", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jor", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("joshkar-ola", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("js", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("jx", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("k-uralsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("k", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("k12", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("k12", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("k12", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kagawa", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kagoshima", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kalmykia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kaluga", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kamchatka", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kanagawa", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kanazawa", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("karelia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("katowice", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kawasaki", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kazan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kchr", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kemerovo", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kg", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kh", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("khabarovsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("khakassia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kharkov", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kherson", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("khmelnitskiy", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("khv", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kids", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kiev", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kirov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kirovograd", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kitakyushu", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("km", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kms", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kobe", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kochi", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("koenig", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("komforb", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("komi", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kommunalforbund", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kommune", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("komvux", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("konyvelo", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kostroma", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kr", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("krakow", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("krasnoyarsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ks", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ks", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kuban", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kumamoto", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kurgan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kursk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kustanai", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kuzbass", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kv", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ky", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kyonggi", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("kyoto", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("la", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lakas", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lanarb", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lanbib", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("law", "pro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("law", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lel", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lg", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lg", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lipetsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lkd", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ln", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lodz", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "gi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ltd", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lublin", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lugansk", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lutsk", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("lviv", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("m", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ma", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("magadan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("magnitka", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mail", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("maori", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mari-el", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mari", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("marine", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mat", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("matsuyama", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mb", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("md", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("me", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("me", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "pro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("med", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("medecin", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("media", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("media", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mi", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mi", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("miasta", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mie", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ac"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "by"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "gh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "gu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "hn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "io"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "jo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "kg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "kw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "kz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "lb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "lt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "lu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "np"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "sh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "sk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "uy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "ye"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mil", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("miyagi", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("miyazaki", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mk", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mn", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mo", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mo", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mob", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mobi", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mobil", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mobile", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mod", "gi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mod", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mod", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mordovia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mosreg", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ms", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ms", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("msk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mt", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("muni", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("murmansk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mus", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("museum", "mn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("museum", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("museum", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("museum", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("museum", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("museum", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("music", "mobi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("mytis", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("n", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nagano", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nagasaki", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nagoya", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nakhodka", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nalchik", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "hr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("name", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nara", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nat", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("national-library-scotland", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("naturbruksgymn", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("navy", "mil"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nb", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nc", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nd", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ne", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ne", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ne", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ne", "tz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ne", "ug"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ne", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nel", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ac"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "af"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ag"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ai"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "an"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "bz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "cd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ch"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ck"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "dm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "gn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "gp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "gr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "gu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "hk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "hn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "im"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "io"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "jm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "jo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "kg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ki"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "kw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ky"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "kz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "la"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "lb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "lc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "li"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "lr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "lu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ma"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "mx"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "nc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "nf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ni"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "np"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "py"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "qa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "rw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "sy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "uy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "uz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "vc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "vi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ws"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "ye"));
    TWO_PART_TLD_BUILDER.add(Pair.of("net", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("new", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("news", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nf", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ngo", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ngo", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ngo", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ngo", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nh", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nhs", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nic", "im"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nic", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nic", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nic", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nieruchomosci", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("niigata", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nikolaev", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nj", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nkz", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nl", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nls", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nm", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nm", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nnov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("no", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "ad"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "ag"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "es"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "ni"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "re"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nom", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nome", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("norilsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("not", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("notaires", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("novosibirsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ns", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nsn", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nsw", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nt", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nt", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nt", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ntr", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nu", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nui", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nv", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("nx", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ny", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("o", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("od", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("odessa", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("odo", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("off", "ai"));
    TWO_PART_TLD_BUILDER.add(Pair.of("og", "ao"));
    TWO_PART_TLD_BUILDER.add(Pair.of("oh", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("oita", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ok", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("okayama", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("okinawa", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("olsztyn", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("omsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("on", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("opole", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "at"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "th"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "tz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "ug"));
    TWO_PART_TLD_BUILDER.add(Pair.of("or", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("orenburg", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ac"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ag"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ai"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "an"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ar"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "bz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "cd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ch"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ck"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "cu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "dm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "er"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "es"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "et"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "fk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "gu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "hk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "hn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "il"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "im"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "io"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "jm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "jo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "kg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ki"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "kw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ky"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "kz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "la"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "lb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "lc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "li"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "lr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ls"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "lu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "lv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ma"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "mx"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "my"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "nc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ni"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "np"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pe"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ph"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "py"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "qa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "rs"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sb"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "sy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "tw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "uy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "uz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "vc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "vi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "vu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ws"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "ye"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "yu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "zm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("org", "zw"));
    TWO_PART_TLD_BUILDER.add(Pair.of("oryol", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("osaka", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("oskol", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("otc", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("oz", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pa", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("palana", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("parliament", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("parliament", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("parti", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pb", "ao"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pc", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pe", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pe", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("penza", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("per", "kh"));
    TWO_PART_TLD_BUILDER.add(Pair.of("per", "sg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("perm", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("perso", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pharmacien", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pl", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pl", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("plc", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("plc", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("plc", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("plo", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pol", "dz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pol", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pol", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("police", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("poltava", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("port", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("powiat", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("poznan", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pp", "az"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pp", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pp", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ppg", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("prd", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("prd", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("press", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("press", "ma"));
    TWO_PART_TLD_BUILDER.add(Pair.of("press", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("presse", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pri", "ee"));
    TWO_PART_TLD_BUILDER.add(Pair.of("principe", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("priv", "at"));
    TWO_PART_TLD_BUILDER.add(Pair.of("priv", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("priv", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("priv", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "ec"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "mv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "pr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pro", "vn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("psc", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("psi", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pskov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ptz", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pub", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("publ", "pt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pvt", "ge"));
    TWO_PART_TLD_BUILDER.add(Pair.of("pyatigorsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("qc", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("qc", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("qh", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("qld", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("qsl", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("re", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("realestate", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rec", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rec", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rec", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rec", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("red", "sv"));
    TWO_PART_TLD_BUILDER.add(Pair.of("reklam", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rel", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rel", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("res", "in"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ri", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rnd", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rnrt", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rns", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rnu", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rovno", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rs", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ru", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ru", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rubtsovsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("rv", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ryazan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("s", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sa", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sa", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sa", "cr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("saga", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("saitama", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sakhalin", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("samara", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("saotome", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sapporo", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("saratov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sark", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sc", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sc", "ke"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sc", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sc", "ug"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sc", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "ae"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "gg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "ir"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "je"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "ly"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "ng"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "om"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "sa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "uk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sch", "zm"));
    TWO_PART_TLD_BUILDER.add(Pair.of("school", "fj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("school", "nz"));
    TWO_PART_TLD_BUILDER.add(Pair.of("school", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sci", "eg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sd", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sd", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("se", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("se", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sebastopol", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sec", "ps"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sendai", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("seoul", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sex", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sex", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sg", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sh", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("shiga", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("shimane", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("shizuoka", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("shop", "ht"));
    TWO_PART_TLD_BUILDER.add(Pair.of("shop", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("shop", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("simbirsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sk", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sklep", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sld", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sld", "pa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("slg", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("slupsk", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("smolensk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sn", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("snz", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("soc", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("soros", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sos", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("spb", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sport", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("srv", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sshn", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("stat", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("stavropol", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("store", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("store", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("store", "st"));
    TWO_PART_TLD_BUILDER.add(Pair.of("store", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("stv", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("suli", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sumy", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("surgut", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("sx", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("syzran", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("szczecin", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("szex", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("szkola", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("t", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("takamatsu", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tambov", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("targi", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tas", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tatarstan", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("te", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tec", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tel", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tel", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tel", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("telecom", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("telememo", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ternopil", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("test", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tirana", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tj", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tld", "am"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tlf", "nr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "cy"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "mc"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "mg"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tm", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tmp", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tn", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tochigi", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tokushima", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tokyo", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tom", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tomsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("torun", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tottori", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tourism", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tourism", "tn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("toyama", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tozsde", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("travel", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("travel", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("trd", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tsaritsyn", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tula", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tur", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("turystyka", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tuva", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tv", "bo"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tv", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tv", "sd"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tver", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tw", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tx", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("tyumen", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("u", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("udm", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("udmurtia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uk", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uk", "net"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uk", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ulan-ude", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ulsan", "kr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("unam", "na"));
    TWO_PART_TLD_BUILDER.add(Pair.of("unbi", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uniti", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("unsa", "ba"));
    TWO_PART_TLD_BUILDER.add(Pair.of("upt", "al"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uri", "arpa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("urn", "arpa"));
    TWO_PART_TLD_BUILDER.add(Pair.of("us", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("us", "tf"));
    TWO_PART_TLD_BUILDER.add(Pair.of("us", "tt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("ut", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("utazas", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("utsunomiya", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uu", "mt"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uy", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("uzhgorod", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("va", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vatican", "va"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vdonsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vet", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("veterinaire", "fr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vgs", "no"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vic", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("video", "hu"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vinnica", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vladikavkaz", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vladimir", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vladivostok", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vn", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("volgograd", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vologda", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("voronezh", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vrn", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vt", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("vyatka", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("w", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wa", "au"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wa", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wakayama", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("warszawa", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("waw", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("weather", "mobi"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "co"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "do"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "id"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "lk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "pk"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "tj"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "tr"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "ve"));
    TWO_PART_TLD_BUILDER.add(Pair.of("web", "za"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wi", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wroc", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wroclaw", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wv", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("www", "ro"));
    TWO_PART_TLD_BUILDER.add(Pair.of("wy", "us"));
    TWO_PART_TLD_BUILDER.add(Pair.of("x", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("xj", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("xz", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("y", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yakutia", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yamagata", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yamaguchi", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yamal", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yamanashi", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yaroslavl", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yekaterinburg", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yk", "ca"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yn", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yokohama", "jp"));
    TWO_PART_TLD_BUILDER.add(Pair.of("yuzhno-sakhalinsk", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("z", "se"));
    TWO_PART_TLD_BUILDER.add(Pair.of("za", "com"));
    TWO_PART_TLD_BUILDER.add(Pair.of("za", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zaporizhzhe", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zgora", "pl"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zgrad", "ru"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zhitomir", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zj", "cn"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zlg", "br"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zp", "ua"));
    TWO_PART_TLD_BUILDER.add(Pair.of("zt", "ua"));
    TWO_PART_TLDS = TWO_PART_TLD_BUILDER.build();
  }
}
