// =================================================================================================
// Copyright 2011 Twitter, Inc.
// -------------------------------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this work except in compliance with the License.
// You may obtain a copy of the License in the LICENSE file, or at:
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// =================================================================================================

package com.twitter.common.stats;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;

import com.twitter.common.base.MorePreconditions;
import com.twitter.common.util.Sampler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.LinkedBlockingDeque;

import javax.annotation.Nullable;

/**
 * A stats tracker to export percentiles of inputs based on a sampling rate.
 *
 * A percentile tracker will randomly sample recorded events with the given sampling rate, and
 * will automatically register variables to track the percentiles requested.
 * Percentiles are calculated based on the K most recent sampling windows, where each sampling
 * window has the recorded events for a sampling period.
 *
 * Uses a trick to avoid allocation every period. Samples are stored in an ArrayList to start with
 * but on hitting maximum size becomes a ring buffer thus always retaining last MAX_BUFFER_SIZE 
 * samples. The ArrayList is cleared but reused thus avoiding reallocation next time.
 */
public class Percentile<T extends Number & Comparable<T>> {

  @VisibleForTesting
  static final int MAX_BUFFER_SIZE = 10001;

  private final Sampler sampler;

  private final Map<Double, SampledStat<Double>> statsByPercentile;
  private int ringBufferStart = 0;
  @VisibleForTesting
  final List<T> samples = Lists.newArrayList();

  private final LinkedBlockingDeque<ArrayList<T>> sampleQueue;
  private final ArrayList<T> allSamples = new ArrayList<T>();

  /**
   * Creates a new percentile tracker.
   *
   * @param name The name of the value whose percentile is being tracked.
   * @param samplePercent The percent of events to sample [0, 100].
   * @param percentiles The percentiles to track.
   */
  public Percentile(String name, float samplePercent, double... percentiles) {
    this(name, new Sampler(samplePercent), percentiles);
  }

  /**
   * Creates a new percentile tracker.
   *
   * @param name The name of the value whose percentile is being tracked.
   * @param sampler The sampler to use for selecting recorded events.
   * @param percentiles The percentiles to track.
   */
  public Percentile(String name, Sampler sampler, double... percentiles) {
    this(name, 1, sampler, percentiles);
  }

  /**
   * Creates a new percentile tracker.
   *
   * A percentile tracker will randomly sample recorded events with the given sampling rate, and
   * will automatically register variables to track the percentiles requested.
   * When allowFlushAfterSample is set to true, once the last percentile is sampled,
   * all recorded values are flushed in preparation for the next window; otherwise, the percentile
   * is calculated using the moving window of the most recent values.
   *
   * @param name The name of the value whose percentile is being tracked.
   * @param numSampleWindows How many sampling windows are used for calculation.
   * @param sampler The sampler to use for selecting recorded events. You may set sampler to null
   *        to sample all input.
   * @param percentiles The percentiles to track.
   */
  public Percentile(String name, int numSampleWindows,
      @Nullable Sampler sampler, double... percentiles) {
    MorePreconditions.checkNotBlank(name);
    Preconditions.checkArgument(numSampleWindows >= 1, "Must have one or more sample windows.");
    Preconditions.checkNotNull(percentiles);
    Preconditions.checkArgument(percentiles.length > 0, "Must specify at least one percentile.");

    this.sampler = sampler;

    sampleQueue = new LinkedBlockingDeque<ArrayList<T>>(numSampleWindows);

    ImmutableMap.Builder<Double, SampledStat<Double>> builder =
        new ImmutableMap.Builder<Double, SampledStat<Double>>();

    for (int i = 0; i < percentiles.length; i++) {
      boolean sortFirst = i == 0;
      String statName = String.format("%s_%s_percentile", name, percentiles[i])
          .replace('.', '_');

      SampledStat<Double> stat = new PercentileVar(statName, percentiles[i], sortFirst);
      Stats.export(stat);
      builder.put(percentiles[i], stat);
    }

    statsByPercentile = builder.build();
  }

  /**
   * Get the variables associated with this percentile tracker.
   *
   * @return A map from tracked percentile to the Stat corresponding to it
   */
  public Map<Double, ? extends Stat<?>> getPercentiles() {
    return ImmutableMap.copyOf(statsByPercentile);
  }

  @VisibleForTesting
  SampledStat<Double> getPercentile(double percentile) {
    return statsByPercentile.get(percentile);
  }

  /**
   * Records an event.
   *
   * @param value The value to record if it is randomly selected based on the sampling rate.
   */
  public void record(T value) {
    if (sampler == null || sampler.select()) {
      synchronized (samples) {
        if (samples.size() < MAX_BUFFER_SIZE) {
          samples.add(value);
        } else {
          // Acts as Ring Buffer
          // ringBufferStart points to the oldest item which is overwritten
          samples.set(ringBufferStart, value);
          ringBufferStart = (ringBufferStart + 1) % MAX_BUFFER_SIZE;
        }
      }
    }
  }

  private class PercentileVar extends SampledStat<Double> {
    private final double percentile;
    private final boolean sortFirst;

    PercentileVar(String name, double percentile, boolean sortFirst) {
      super(name, 0d);
      this.percentile = percentile;
      this.sortFirst = sortFirst;
    }

    @Override
    public Double doSample() {
      synchronized (sampleQueue) {
        if (sortFirst) {
          if (sampleQueue.remainingCapacity() == 0) {
            sampleQueue.removeFirst();
          }

          ArrayList<T> copyOfSamples;
          synchronized (samples) {
            copyOfSamples = new ArrayList<T>(samples);
            // Note that clear retains capacity which is intentional here
            samples.clear();
            ringBufferStart = 0;
          }

          // If samples became a ring buffer the start may not be at zero
          // but that's ok, we sort the samples later any way.
          sampleQueue.addLast(copyOfSamples);

          allSamples.clear();
          for (ArrayList<T> sample : sampleQueue) {
            allSamples.addAll(sample);
          }

          Collections.sort(allSamples, Ordering.<T>natural());
        }

        if (allSamples.isEmpty()) {
          return 0d;
        }

        int maxIndex = allSamples.size() - 1;
        double selectIndex = maxIndex * percentile / 100;
        selectIndex = selectIndex < 0d ? 0d : selectIndex;
        selectIndex = selectIndex > maxIndex ? maxIndex : selectIndex;

        int indexLeft = (int) selectIndex;
        if (indexLeft == maxIndex) {
          return allSamples.get(indexLeft).doubleValue();
        }

        double residue = selectIndex - indexLeft;
        return allSamples.get(indexLeft).doubleValue() * (1 - residue) +
            allSamples.get(indexLeft + 1).doubleValue() * residue;
      }
    }
  }
}
