/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2018-2019 Ping Identity Corporation
 */

package com.unboundid.directory.sdk.broker.api;

import com.fasterxml.jackson.databind.JsonNode;
import com.unboundid.directory.sdk.broker.config.AdviceConfig;
import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.broker.types.AdviceStatement;
import com.unboundid.directory.sdk.broker.types.BrokerContext;
import com.unboundid.directory.sdk.broker.types.NotFulfilledException;
import com.unboundid.directory.sdk.broker.types.PolicyRequestDetails;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;

import java.util.Collections;
import java.util.List;
import java.util.Map;


/**
 * This class defines an API that must be implemented by extensions that
 * implement custom Advice for policies.  This advice is invoked when policy
 * evaluation results in this type of advice being returned to the Policy
 * Enforcement Point.
 *
 * <H2>Configuring Advice</H2>
 * In order to configure policy advice created using this API,
 * use a command like:
 *
 * <PRE>
 *      dsconfig create-advice \
 *           --advice-name "<I>{name}</I>" \
 *           --type third-party \
 *           --set "advice-id:<I>{advice-code}</I>
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>" \
 *           --set "evaluation-order-index:<I>{index}</I>
 * </PRE>
 * where "<I>{name}</I>" is a user-friendly name to use for the advice
 * type, "<I>{advice-code}</I>" is a unique advice code that  must match the
 * "code" string returned from a policy invocation, and "<I>{class-name}</I>"
 * is the fully-qualified name of the Java class that extends
 * {@code com.unboundid.directory.sdk.broker.api.Advice}.
 *
 * "<I>{index}</I>" is an integer from 1 to 9999 that is used to determine the
 * order in which this type of advice should be invoked relative to other advice
 * that may be returned from the same policy evaluation.
 * "<I>{name=value}</I>" pairs specified using extension-arguments are values
 * that are provided to the advice implementation at initialization time.
 * If multiple initialization arguments should be provided to the extension,
 * then the "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option
 * should be provided multiple times.
 */
@Extensible()
@BrokerExtension
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class Advice
    implements UnboundIDExtension, ExampleUsageProvider {

  /**
   * No-args constructor.
   */
  public Advice() {
    // no implementation is required.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String getExtensionName();


  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String[] getExtensionDescription();


  /**
   * {@inheritDoc}
   */
  @Override
  public Map<List<String>,String> getExamplesArgumentSets() {
    return Collections.emptyMap();
  }


  /**
   * {@inheritDoc}
   */
  @Override
  public void defineConfigArguments(final ArgumentParser parser)
      throws ArgumentException {
    // No arguments will be allowed by default.
  }

  /**
   * Initializes this Advice implementation.
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this Advice.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this Advice.
   * @throws Exception      If a problem occurs while initializing this Advice.
   */
  public void initializeAdvice(
      final BrokerContext serverContext,
      final AdviceConfig config,
      final ArgumentParser parser) throws Exception {
    // No initialization performed by default.
  }


  /**
   * Performs any cleanup which may be necessary when this advice
   * is to be taken out of service.
   */
  public void finalizeAdvice() {
    // No implementation is required.
  }


  /**
   * This method is invoked when the specified Advice type is returned by
   * policy evaluation.
   * @param requestDetails Details of the authorization request that
   *                       triggered the advice.
   * @param statements     List of advice instances with payload and
   *                       attributes containing advice details.
   * @param resourceNode   JSON resource that may be examined or manipulated
   *                       by the advice implementation.  The resource
   *                       details are dependent upon the API method that
   *                       requested authorization and the type of request
   *                       being authorized. As an example, for a create
   *                       operation the resource object may represent the
   *                       resource the client is attempting to create, while
   *                       for a retrieve operation the resourceNode may
   *                       represent the resource to be returned to the client.
   * @return the (possibly modified) resource object.
   * @throws NotFulfilledException if the advice cannot be successfully
   * applied.  If the advice is defined to be obligatory, throwing this
   * exception will cause the requested operation to fail.  If the advice is
   * not obligatory, throwing this exception will cause an error to be logged,
   * however the requested operation will not otherwise be impacted.
   */
  public abstract JsonNode apply(
      final PolicyRequestDetails requestDetails,
      List<? extends AdviceStatement> statements,
      JsonNode resourceNode) throws NotFulfilledException;
}
