/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file. If applicable, add the following below this CDDL HEADER,
 * with the fields enclosed by brackets "[]" replaced with your own
 * identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2013-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.broker.api;

import com.unboundid.directory.sdk.broker.config.StoreAdapterConfig;
import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.broker.types.BrokerContext;
import com.unboundid.directory.sdk.broker.types.StoreAttributeDefinition;
import com.unboundid.directory.sdk.broker.types.StoreCreateRequest;
import com.unboundid.directory.sdk.broker.types.StoreDeleteRequest;
import com.unboundid.directory.sdk.broker.types.StoreSearchRequest;
import com.unboundid.directory.sdk.broker.types.StoreRetrieveRequest;
import com.unboundid.directory.sdk.broker.types.StoreUpdateRequest;
import com.unboundid.directory.sdk.common.internal.Configurable;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.scim2.common.exceptions.ScimException;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;


/**
 * This class defines an API that must be implemented by extensions that need
 * to store user data in some non-LDAP data store. This adapter API is generic
 * and can support a wide range of repositories. When using multiple
 * ${BROKER_BASE_NAME} instances in a deployment, the data store should be
 * accessible from all instances.
 * <p>
 * After the Store Adapter is initialized (via the initializeStoreAdapter()
 * method), all methods will be guarded by a call to {@link #isAvailable()} to
 * make sure that the adapter is currently connected and ready to service
 * requests. If this returns {@code false}, the SCIMResourceType will return an
 * appropriate 503 response code to clients until the adapter becomes available
 * again.
 * <H2>Configuring Store Adapters</H2>
 * In order to configure a store adapter created using this API, use
 * a command like:
 * <PRE>
 *      dsconfig create-store-adapter \
 *           ---adapter-name "<I>{name}</I>" \
 *           --type third-party \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{name}</I>" is the name to use for the store adapter
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends
 * {@code com.unboundid.directory.sdk.broker.api.StoreAdapter},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the store adapter. If multiple arguments should be
 * provided to the extension, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@BrokerExtension
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class StoreAdapter implements UnboundIDExtension,
        Configurable, ExampleUsageProvider
{
  /**
   * Creates a new instance of this store adapter.  All
   * implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@link #initializeStoreAdapter} method.
   */
  public StoreAdapter()
  {
    // No implementation is required.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String getExtensionName();

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String[] getExtensionDescription();

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }

  /**
   * Initializes this store adapter. Any initialization should be performed
   * here. This method should generally store the {@link BrokerContext} in
   * a class member so that it can be used elsewhere in the implementation.
   * <p>
   * The default implementation is empty.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running. Extensions should
   *                        typically store this in a class member.
   * @param  config         The general configuration for this object.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this store adapter.
   * @throws Exception      If a problem occurs while initializing this store
   *                        adapter.
   */
  public void initializeStoreAdapter(final BrokerContext serverContext,
                                     final StoreAdapterConfig config,
                                     final ArgumentParser parser)
      throws Exception
  {
    // No initialization will be performed by default.
  }

  /**
   * This hook is called when the SCIMResourceType is disabled or the
   * ${BROKER_BASE_NAME} shuts down. Any clean-up of this store adapter should
   * be performed here.
   * <p>
   * The default implementation is empty.
   */
  public void finalizeStoreAdapter()
  {
    // No implementation is performed by default.
  }

  /**
   * Retrieves a collection of attribute definitions describing the schema
   * for objects supported by this Store Adapter. The default implementation
   * returns an empty collection indicating that the schema is not defined.
   * Use StoreAttributeDefinition.Builder to create instances of
   * StoreAttributeDefinition.
   *
   * @return  The attribute definitions describing the schema, or an empty
   *          collection if the schema is not defined.
   */
  public Collection<StoreAttributeDefinition> getNativeSchema()
  {
    return Collections.emptyList();
  }

  /**
   * Determines whether this Store Adapter is currently available and
   * in-service. This may return {@code false} in the case that all
   * backend servers are down, for example; during this time the
   * SCIMResourceType will return an appropriate 503 response code to clients.
   *
   * @return {@code true} if the Store Adapter initialized and connected;
   *         {@code false} otherwise.
   */
  public abstract boolean isAvailable();

  /**
   * Fetches the specified entry.
   *
   * @param request  The retrieve request.
   * @return  The retrieved entry as a JSON object. This string can be created
   *          by any JSON library. For example, the Jackson library's
   *          ObjectNode.toString() or the UnboundID LDAP SDK's
   *          JSONObject.toString().
   * @throws ScimException  If there is a problem fulfilling the request.
   */
  public abstract String retrieve(
          final StoreRetrieveRequest request) throws ScimException;

  /**
   * Search for entries in the native store which could match the
   * specified criteria. The contract is for the store adapter to return a
   * superset of matching entries, i.e. The store adapter must return all
   * entries which match the specified filter but may also return entries
   * which do not match. The results from the store adapter are subsequently
   * filtered by the ${BROKER_BASE_NAME}.
   *
   * @param request  The search request.
   * @throws ScimException  If there is a problem fulfilling the search request.
   */
  public abstract void search(
          final StoreSearchRequest request) throws ScimException;

  /**
   * Create the specified entry in the native data store.
   *
   * @param request  The create request.
   * @return The entry that was just created as a JSON object. This string can
   *         be created by any JSON library. For example, the Jackson library's
   *         ObjectNode.toString() or the UnboundID LDAP SDK's
   *         JSONObject.toString().
   * @throws ScimException  If there is a problem creating the entry.
   */
  public abstract String create(final StoreCreateRequest request)
          throws ScimException;

  /**
   * Update the specified entry in the native data store.
   *
   * @param request  The update request.
   * @return  The updated resource as a JSON object. This string can be created
   *          by any JSON library. For example, the Jackson library's
   *          ObjectNode.toString() or the UnboundID LDAP SDK's
   *          JSONObject.toString().
   * @throws ScimException  If there is a problem modifying the entry.
   */
  public abstract String update(final StoreUpdateRequest request)
          throws ScimException;

  /**
   * Delete the specified entry from the native data store.
   *
   * @param request  The delete request.
   * @throws ScimException  If there is a problem deleting the entry.
   */
  public abstract void delete(final StoreDeleteRequest request)
          throws ScimException;
}
