/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2016-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.broker.api;

import com.unboundid.directory.sdk.broker.config.StoreAdapterPluginConfig;
import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.broker.types.BrokerContext;
import com.unboundid.directory.sdk.broker.types.StorePostCreateRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePostDeleteRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePostRetrieveRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePostSearchEntryContext;
import com.unboundid.directory.sdk.broker.types.StorePostUpdateRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePreCreateRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePreDeleteRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePreRetrieveRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePreSearchRequestContext;
import com.unboundid.directory.sdk.broker.types.StorePreUpdateRequestContext;
import com.unboundid.directory.sdk.common.internal.Configurable;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.scim2.common.exceptions.ScimException;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;

import java.util.Collections;
import java.util.List;
import java.util.Map;



/**
 * This class defines an API that must be implemented by extensions that need
 * to intercept operations processed by a Store Adapter. The API has
 * pre-request methods to intercept and make changes to store adapter requests
 * before they are processed by the Store Adapter, and corresponding
 * post-request methods to intercept and make changes to the results returned
 * by the Store Adapter. All SCIM attributes in this API refer to native Store
 * Adapter attributes, not attributes in the external client SCIM schema.
 * <p>
 * The Store Adapter Plugin is defined in the configuration and then referenced
 * by a Store Adapter configuration object.
 */
@Extensible()
@BrokerExtension
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class StoreAdapterPlugin implements UnboundIDExtension,
    Configurable, ExampleUsageProvider
{
  /**
   * Creates a new instance of this store adapter plugin.  All
   * implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@link #initializeStoreAdapterPlugin} method.
   */
  public StoreAdapterPlugin()
  {
    // No implementation is required.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String getExtensionName();

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract String[] getExtensionDescription();

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }

  /**
   * Initializes this store adapter plugin. Any initialization should be
   * performed here. This method should generally store the
   * {@link BrokerContext} in a class member so that it can be used elsewhere in
   * the implementation.
   * <p>
   * The default implementation is empty.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running. Extensions should
   *                        typically store this in a class member.
   * @param  config         The general configuration for this object.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this store adapter plugin.
   * @throws Exception      If a problem occurs while initializing this store
   *                        adapter plugin.
   */
  public void initializeStoreAdapterPlugin(
      final BrokerContext serverContext,
      final StoreAdapterPluginConfig config,
      final ArgumentParser parser)
      throws Exception
  {
    // No initialization will be performed by default.
  }

  /**
   * This hook is called when the store adapter plugin is disabled or the
   * ${BROKER_BASE_NAME} shuts down. Any clean-up of this store adapter plugin
   * should be performed here.
   * <p>
   * The default implementation is empty.
   */
  public void finalizeStoreAdapterPlugin()
  {
    // No finalization will be performed by default.
  }

  /**
   * This method is called before an entry is created in the native data store.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, the operation will be aborted and a
   *                       corresponding SCIM ErrorResponse will be returned to
   *                       the client.
   */
  public void preCreate(final StorePreCreateRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called after an entry is created in the native data store.
   * This method will not be called if the create fails.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, a corresponding SCIM ErrorResponse will be
   *                       returned to the client. No attempt is made to roll
   *                       back the committed create operation.
   */
  public void postCreate(final StorePostCreateRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called before an entry is retrieved from the native data
   * store in the following cases:
   * <UL>
   *   <LI>
   *     To fetch an entry for a SCIM retrieve operation.
   *   </LI>
   *   <LI>
   *     To fetch an entry from a secondary store adapter for each primary store
   *     adapter entry obtained during a SCIM search.
   *   </LI>
   *   <LI>
   *     To fetch an entry from the primary store adapter for each secondary
   *     store adapter entry obtained when a SCIM search filter is processed
   *     by a secondary store adapter.
   *   </LI>
   * </UL>
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, the operation will be aborted and a
   *                       corresponding SCIM ErrorResponse will be returned to
   *                       the client.
   */
  public void preRetrieve(final StorePreRetrieveRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called after an entry is retrieved from the native data
   * store.
   * This method will not be called if the retrieve fails.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, a corresponding SCIM ErrorResponse will be
   *                       returned to the client.
   */
  public void postRetrieve(final StorePostRetrieveRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called before an entry is updated in the native data
   * store.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, the operation will be aborted and a
   *                       corresponding SCIM ErrorResponse will be returned to
   *                       the client.
   */
  public void preUpdate(final StorePreUpdateRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called after an entry is updated in the native data
   * store.
   * This method will not be called if the update fails.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, a corresponding SCIM ErrorResponse will be
   *                       returned to the client. No attempt is made to roll
   *                       back the committed update operation.
   */
  public void postUpdate(final StorePostUpdateRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called before an entry is deleted in the native data
   * store.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, the operation will be aborted and a
   *                       corresponding SCIM ErrorResponse will be returned to
   *                       the client.
   */
  public void preDelete(final StorePreDeleteRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called after an entry is deleted in the native data
   * store.
   * This method will not be called if the delete fails.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, a corresponding SCIM ErrorResponse will be
   *                       returned to the client. No attempt is made to roll
   *                       back the committed delete operation.
   */
  public void postDelete(final StorePostDeleteRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called before searching for entries in the native data
   * store.
   * <p>
   * Note that the contract for store adapter searches allows the store adapter
   * to return a superset of entries matching the original search filter. The
   * results from the store adapter are subsequently filtered by the
   * ${BROKER_BASE_NAME}.
   *
   * @param context The store adapter request context.
   * @throws ScimException If thrown, the operation will be aborted and a
   *                       corresponding SCIM ErrorResponse will be returned to
   *                       the client.
   */
  public void preSearch(final StorePreSearchRequestContext context)
      throws ScimException
  {
    // No implementation is required by default.
  }

  /**
   * This method is called for each entry found during a search in the native
   * data store.
   * This method will not be called if the search fails.
   * <p>
   * Note that the contract for store adapter searches allows the store adapter
   * to return a superset of entries matching the original search filter. The
   * results from the store adapter are subsequently filtered by the
   * ${BROKER_BASE_NAME}.
   *
   * @param context The store adapter request context.
   */
  public void postSearchEntry(final StorePostSearchEntryContext context)
  {
    // No implementation is required by default.
  }
}
