/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file. If applicable, add the following below this CDDL HEADER,
 * with the fields enclosed by brackets "[]" replaced with your own
 * identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2013-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.broker.types;



import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.scim2.client.ScimInterface;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import java.util.Map;



/**
 * This interface may be used to obtain information about the
 * ${BROKER_BASE_NAME} in which an extension is running.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface BrokerContext extends ServerContext
{
  /**
   * Indicates whether trace log messages from extensions are loggable
   * to any trace log publishers. This allows callers to avoid expensive
   * message construction for messages that are not going to be logged.
   *
   * @return  {@code true} if trace log messages from extensions are
   *          loggable to any trace log publishers
   */
  boolean isTraceMessageLoggable();



  /**
   * Writes a message to the server trace log publishers. Sensitive information
   * must not be included in the contents of the message.
   * <p>
   * Trace log messages should be used in methods that are called when
   * processing an incoming request. The trace log will automatically
   * include request identifiers in the log messages so that all log
   * messages about a request can be correlated.
   * <p>
   * You can use the <i>search-logs</i> command-line tool on the trace log
   * file to extract only the messages logged by your extension.
   *
   * @param  message    The message to be logged. It must not be
   *                    {@code null}.
   */
  void logTraceMessage(final String message);



  /**
   * Writes a message to the server trace log publishers, including key value
   * pairs to be logged with the message. Sensitive information
   * must not be included in the contents of the message or the key value
   * pairs.
   * <p>
   * Trace log messages should be used in methods that are called when
   * processing an incoming request. The trace log will automatically
   * include request identifiers in the log messages so that all log
   * messages about a request can be correlated.
   * <p>
   * You can use the <i>search-logs</i> command-line tool on the trace log
   * file to extract only the messages logged by your extension.
   *
   * @param  message    The message to be logged. It must not be
   *                    {@code null}.
   * @param  keyValues  Keys and values that are to be logged with
   *                    key=value. It may be {@code null}.
   */
  void logTraceMessage(final String message,
                       final Map<String, String> keyValues);

  /**
   * Gets an internal implementation of the ScimInterface.
   *
   * @return an object that implements the ScimInterface using
   * in-process access to the internal SCIM objects.  This implementation
   * will not use HTTP/REST.
   */
  ScimInterface getInternalScimInterface();
}
