/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2016-2019 Ping Identity Corporation
 */

package com.unboundid.directory.sdk.broker.types;

import com.unboundid.util.NotExtensible;

/**
 * The result of the status request of an Identity Authenticator. Use
 * StatusResult.Builder to create instances of this class.
 */
@NotExtensible()
public final class StatusResult
{
  /**
   * Whether the authenticator is ready to process authentication requests.
   */
  private final boolean ready;

  /**
   * The error code if an error occurred while processing the status
   * request.
   */
  private final String error;

  /**
   * The message describing the error that occurred while processing the
   * status request.
   */
  private final String errorDetail;

  /**
   * Parameters to return that may be used to initiate the authentication
   * process. May be {@code null}.
   */
  private final String responseParams;

  /**
   * The message that will be only be included in the trace log for diagnostic
   * purposes.
   */
  private final String diagnosticMessage;



  /**
   * Builder class to build an instance of StatusResult.
   */
  public static class Builder
  {
    private boolean ready;
    private String error;
    private String errorDetail;
    private String responseParams;
    private String diagnosticMessage;



    /**
     * Specifies the JSON object specifying the parameters to return that may be
     * used to initiate the authentication process.
     * <p>
     * Since this response is visible to the client in a potentially
     * unauthenticated context, the authenticator should not expose any
     * sensitive user or server information in the response.
     * @param responseParams  The parameters to return.
     * @return  this builder.
     */
    public Builder setResponseParams(final String responseParams)
    {
      this.responseParams = responseParams;
      return this;
    }

    /**
     * Specifies the message that will be only be included in the trace log
     * for diagnostic purposes.
     * @param diagnosticMessage  The diagnostic message
     * @return  this builder.
     */
    public Builder setDiagnosticMessage(final String diagnosticMessage)
    {
      this.diagnosticMessage = diagnosticMessage;
      return this;
    }



    /**
     * Builds a new AuthenticationResult.
     *
     * @return  A new AuthenticationResult.
     */
    public StatusResult build()
    {
      return new StatusResult(
          ready,
          error,
          errorDetail,
          responseParams,
          diagnosticMessage);
    }
  }



  /**
   * Create a new builder for a ready response.
   * @return  A new builder for a ready response.
   */
  public static Builder ready()
  {
    final Builder b = new Builder();
    b.ready = true;
    return b;
  }



  /**
   * Create a new builder for a not ready response. An authenticator
   * should return a not ready response if it may not be used to authenticate
   * the user/client/etc... at this time.
   * @return  A new builder for a not ready response.
   */
  public static Builder notReady()
  {
    final Builder b = new Builder();
    b.ready = false;
    return b;
  }



  /**
   * Create a new builder for an error response. An authenticator
   * should return an error response if an error occurred while determining
   * whether the authenticator is ready.
   * <p>
   * Since this response is visible to the client in a potentially
   * unauthenticated context, the authenticator should not expose any
   * sensitive user or server information in the error response.
   * @param error The error code.
   * @param errorDetail The details about the error.
   *
   * @return  A new builder for an error response.
   */
  public static Builder notReady(final String error, final String errorDetail)
  {
    final Builder b = new Builder();
    b.ready = false;
    b.error = error;
    b.errorDetail = errorDetail;
    return b;
  }



  /**
   * Create a new StatusResult from the provided information.
   * @param ready           Whether the authenticator is ready.
   * @param error           The error code.
   * @param errorDetail     The error detail.
   * @param responseParams  The response parameters.
   * @param diagnosticMessage  The diagnostic message.
   */
  private StatusResult(
      final boolean ready,
      final String error,
      final String errorDetail,
      final String responseParams,
      final String diagnosticMessage)
  {
    this.ready = ready;
    this.error = error;
    this.errorDetail = errorDetail;
    this.responseParams = responseParams;
    this.diagnosticMessage = diagnosticMessage;
  }



  /**
   * Indicates whether this authenticator is ready to process authentication
   * requests.
   *
   * @return {@code true} if this authenticator is ready, or
   *         {@code false} otherwise.
   */
  public boolean isReady()
  {
    return ready;
  }



  /**
   * Retrieve the error code to return if an error occurred while processing the
   * authentication request or {@code null} if an error did not occur.
   *
   * @return The error code.
   */
  public String getError()
  {
    return error;
  }



  /**
   * Retrieve the message to return if an error occurred while processing
   * the authentication request describing the details of the error
   * or {@code null} if an error did not occur.
   *
   * @return The error details.
   */
  public String getErrorDetail()
  {
    return errorDetail;
  }



  /**
   * Retrieve the JSON object specifying the parameters to return that may be
   * used to initiate the authentication process. This string can be parsed
   * using any JSON library. For example, Jackson's ObjectMapper.readTree() or
   * the UnboundID LDAP SDK's JSONObject constructor.
   * @return  The parameters to return, may be {@code null}.
   */
  public String getResponseParams()
  {
    return responseParams;
  }

  /**
   * Gets the message that will be only be included in the trace log
   * for diagnostic purposes.
   *
   * @return The message that will be only be included in the trace log
   */
  public String getDiagnosticMessage()
  {
    return diagnosticMessage;
  }



  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object o)
  {
    if (this == o)
    {
      return true;
    }
    if (o == null || getClass() != o.getClass())
    {
      return false;
    }

    StatusResult that = (StatusResult) o;

    if (ready != that.ready)
    {
      return false;
    }
    if (error != null ? !error.equals(that.error) : that.error != null)
    {
      return false;
    }
    if (errorDetail != null ?
        !errorDetail.equals(that.errorDetail) : that.errorDetail != null)
    {
      return false;
    }
    return responseParams != null ?
        !responseParams.equals(that.responseParams) :
        that.responseParams != null;

  }



  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode()
  {
    int result = (ready ? 1 : 0);
    result = 31 * result + (error != null ? error.hashCode() : 0);
    result = 31 * result + (errorDetail != null ? errorDetail.hashCode() : 0);
    result = 31 * result +
        (responseParams != null ? responseParams.hashCode() : 0);
    return result;
  }



  /**
   * {@inheritDoc}
   */
  @Override
  public String toString()
  {
    final StringBuilder sb = new StringBuilder("StatusResult{");
    sb.append("ready=").append(ready);
    sb.append(", error='").append(error).append('\'');
    sb.append(", errorDetail='").append(errorDetail).append('\'');
    sb.append(", responseParams='").append(responseParams).append('\'');
    sb.append('}');
    return sb.toString();
  }
}
